# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    Canonical,
    DateTime,
    Markdown,
    Date,
    Id,
)

from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    ContactDetail,
    UsageContext,
    Period,
    RelatedArtifact,
    BackboneElement,
    Quantity,
    Range,
    Duration,
    TriggerDefinition,
    Expression,
    DataRequirement,
    Age,
    Timing,
    DomainResource,
)


class PlanDefinitionGoalTarget(BackboneElement):
    """
    Indicates what should be done and within what timeframe.
    """

    measure: Optional[CodeableConcept] = Field(
        description="The parameter whose value is to be tracked",
        default=None,
    )
    detailQuantity: Optional[Quantity] = Field(
        description="The target value to be achieved",
        default=None,
    )
    detailRange: Optional[Range] = Field(
        description="The target value to be achieved",
        default=None,
    )
    detailCodeableConcept: Optional[CodeableConcept] = Field(
        description="The target value to be achieved",
        default=None,
    )
    due: Optional[Duration] = Field(
        description="Reach goal within",
        default=None,
    )

    @property
    def detail(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="detail",
        )

    @field_validator(
        *(
            "due",
            "measure",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def detail_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Quantity, Range, CodeableConcept],
            field_name_base="detail",
            required=False,
        )


class PlanDefinitionGoal(BackboneElement):
    """
    A goal describes an expected outcome that activities within the plan are intended to achieve. For example, weight loss, restoring an activity of daily living, obtaining herd immunity via immunization, meeting a process improvement objective, meeting the acceptance criteria for a test as specified by a quality specification, etc.
    """

    category: Optional[CodeableConcept] = Field(
        description="E.g. Treatment, dietary, behavioral",
        default=None,
    )
    description: Optional[CodeableConcept] = Field(
        description="Code or text describing the goal",
        default=None,
    )
    priority: Optional[CodeableConcept] = Field(
        description="high-priority | medium-priority | low-priority",
        default=None,
    )
    start: Optional[CodeableConcept] = Field(
        description="When goal pursuit begins",
        default=None,
    )
    addresses: Optional[List[CodeableConcept]] = Field(
        description="What does the goal address",
        default=None,
    )
    documentation: Optional[List[RelatedArtifact]] = Field(
        description="Supporting documentation for the goal",
        default=None,
    )
    target: Optional[List[PlanDefinitionGoalTarget]] = Field(
        description="Target outcome for the goal",
        default=None,
    )

    @field_validator(
        *(
            "target",
            "documentation",
            "addresses",
            "start",
            "priority",
            "description",
            "category",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )


class PlanDefinitionActionCondition(BackboneElement):
    """
    An expression that describes applicability criteria or start/stop conditions for the action.
    """

    kind: Optional[Code] = Field(
        description="applicability | start | stop",
        default=None,
    )
    kind_ext: Optional[Element] = Field(
        description="Placeholder element for kind extensions",
        default=None,
        alias="_kind",
    )
    expression: Optional[Expression] = Field(
        description="Boolean-valued expression",
        default=None,
    )

    @field_validator(
        *(
            "expression",
            "kind",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class PlanDefinitionActionRelatedAction(BackboneElement):
    """
    A relationship to another action such as "before" or "30-60 minutes after start of".
    """

    actionId: Optional[Id] = Field(
        description="What action is this related to",
        default=None,
    )
    actionId_ext: Optional[Element] = Field(
        description="Placeholder element for actionId extensions",
        default=None,
        alias="_actionId",
    )
    relationship: Optional[Code] = Field(
        description="before-start | before | before-end | concurrent-with-start | concurrent | concurrent-with-end | after-start | after | after-end",
        default=None,
    )
    relationship_ext: Optional[Element] = Field(
        description="Placeholder element for relationship extensions",
        default=None,
        alias="_relationship",
    )
    offsetDuration: Optional[Duration] = Field(
        description="Time offset for the relationship",
        default=None,
    )
    offsetRange: Optional[Range] = Field(
        description="Time offset for the relationship",
        default=None,
    )

    @property
    def offset(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="offset",
        )

    @field_validator(
        *(
            "relationship",
            "actionId",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def offset_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Duration, Range],
            field_name_base="offset",
            required=False,
        )


class PlanDefinitionActionParticipant(BackboneElement):
    """
    Indicates who should participate in performing the action described.
    """

    type: Optional[Code] = Field(
        description="patient | practitioner | related-person | device",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    role: Optional[CodeableConcept] = Field(
        description="E.g. Nurse, Surgeon, Parent",
        default=None,
    )

    @field_validator(
        *(
            "role",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class PlanDefinitionActionDynamicValue(BackboneElement):
    """
    Customizations that should be applied to the statically defined resource. For example, if the dosage of a medication must be computed based on the patient's weight, a customization would be used to specify an expression that calculated the weight, and the path on the resource that would contain the result.
    """

    path: Optional[String] = Field(
        description="The path to the element to be set dynamically",
        default=None,
    )
    path_ext: Optional[Element] = Field(
        description="Placeholder element for path extensions",
        default=None,
        alias="_path",
    )
    expression: Optional[Expression] = Field(
        description="An expression that provides the dynamic value for the customization",
        default=None,
    )

    @field_validator(
        *(
            "expression",
            "path",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class PlanDefinitionAction(BackboneElement):
    """
    An action or group of actions to be taken as part of the plan. For example, in clinical care, an action would be to prescribe a particular indicated medication, or perform a particular test as appropriate. In pharmaceutical quality, an action would be the test that needs to be performed on a drug product as defined in the quality specification.
    """

    prefix: Optional[String] = Field(
        description="User-visible prefix for the action (e.g. 1. or A.)",
        default=None,
    )
    prefix_ext: Optional[Element] = Field(
        description="Placeholder element for prefix extensions",
        default=None,
        alias="_prefix",
    )
    title: Optional[String] = Field(
        description="User-visible title",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    description: Optional[String] = Field(
        description="Brief description of the action",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    textEquivalent: Optional[String] = Field(
        description="Static text equivalent of the action, used if the dynamic aspects cannot be interpreted by the receiving system",
        default=None,
    )
    textEquivalent_ext: Optional[Element] = Field(
        description="Placeholder element for textEquivalent extensions",
        default=None,
        alias="_textEquivalent",
    )
    priority: Optional[Code] = Field(
        description="routine | urgent | asap | stat",
        default=None,
    )
    priority_ext: Optional[Element] = Field(
        description="Placeholder element for priority extensions",
        default=None,
        alias="_priority",
    )
    code: Optional[List[CodeableConcept]] = Field(
        description="Code representing the meaning of the action or sub-actions",
        default=None,
    )
    reason: Optional[List[CodeableConcept]] = Field(
        description="Why the action should be performed",
        default=None,
    )
    documentation: Optional[List[RelatedArtifact]] = Field(
        description="Supporting documentation for the intended performer of the action",
        default=None,
    )
    goalId: Optional[List[Id]] = Field(
        description="What goals this action supports",
        default=None,
    )
    goalId_ext: Optional[Element] = Field(
        description="Placeholder element for goalId extensions",
        default=None,
        alias="_goalId",
    )
    subjectCodeableConcept: Optional[CodeableConcept] = Field(
        description="Type of individual the action is focused on",
        default=None,
    )
    subjectReference: Optional[Reference] = Field(
        description="Type of individual the action is focused on",
        default=None,
    )
    subjectCanonical: Optional[Canonical] = Field(
        description="Type of individual the action is focused on",
        default=None,
    )
    subjectCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for subjectCanonical extensions",
        default=None,
        alias="_subjectCanonical",
    )
    trigger: Optional[List[TriggerDefinition]] = Field(
        description="When the action should be triggered",
        default=None,
    )
    condition: Optional[List[PlanDefinitionActionCondition]] = Field(
        description="Whether or not the action is applicable",
        default=None,
    )
    input: Optional[List[DataRequirement]] = Field(
        description="Input data requirements",
        default=None,
    )
    output: Optional[List[DataRequirement]] = Field(
        description="Output data definition",
        default=None,
    )
    relatedAction: Optional[List[PlanDefinitionActionRelatedAction]] = Field(
        description="Relationship to another action",
        default=None,
    )
    timingDateTime: Optional[DateTime] = Field(
        description="When the action should take place",
        default=None,
    )
    timingDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for timingDateTime extensions",
        default=None,
        alias="_timingDateTime",
    )
    timingAge: Optional[Age] = Field(
        description="When the action should take place",
        default=None,
    )
    timingPeriod: Optional[Period] = Field(
        description="When the action should take place",
        default=None,
    )
    timingDuration: Optional[Duration] = Field(
        description="When the action should take place",
        default=None,
    )
    timingRange: Optional[Range] = Field(
        description="When the action should take place",
        default=None,
    )
    timingTiming: Optional[Timing] = Field(
        description="When the action should take place",
        default=None,
    )
    participant: Optional[List[PlanDefinitionActionParticipant]] = Field(
        description="Who should participate in the action",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="create | update | remove | fire-event",
        default=None,
    )
    groupingBehavior: Optional[Code] = Field(
        description="visual-group | logical-group | sentence-group",
        default=None,
    )
    groupingBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for groupingBehavior extensions",
        default=None,
        alias="_groupingBehavior",
    )
    selectionBehavior: Optional[Code] = Field(
        description="any | all | all-or-none | exactly-one | at-most-one | one-or-more",
        default=None,
    )
    selectionBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for selectionBehavior extensions",
        default=None,
        alias="_selectionBehavior",
    )
    requiredBehavior: Optional[Code] = Field(
        description="must | could | must-unless-documented",
        default=None,
    )
    requiredBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for requiredBehavior extensions",
        default=None,
        alias="_requiredBehavior",
    )
    precheckBehavior: Optional[Code] = Field(
        description="yes | no",
        default=None,
    )
    precheckBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for precheckBehavior extensions",
        default=None,
        alias="_precheckBehavior",
    )
    cardinalityBehavior: Optional[Code] = Field(
        description="single | multiple",
        default=None,
    )
    cardinalityBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for cardinalityBehavior extensions",
        default=None,
        alias="_cardinalityBehavior",
    )
    definitionCanonical: Optional[Canonical] = Field(
        description="Description of the activity to be performed",
        default=None,
    )
    definitionCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for definitionCanonical extensions",
        default=None,
        alias="_definitionCanonical",
    )
    definitionUri: Optional[Uri] = Field(
        description="Description of the activity to be performed",
        default=None,
    )
    definitionUri_ext: Optional[Element] = Field(
        description="Placeholder element for definitionUri extensions",
        default=None,
        alias="_definitionUri",
    )
    transform: Optional[Canonical] = Field(
        description="Transform to apply the template",
        default=None,
    )
    transform_ext: Optional[Element] = Field(
        description="Placeholder element for transform extensions",
        default=None,
        alias="_transform",
    )
    dynamicValue: Optional[List[PlanDefinitionActionDynamicValue]] = Field(
        description="Dynamic aspects of the definition",
        default=None,
    )
    action: Optional[List["PlanDefinitionAction"]] = Field(
        description="A sub-action",
        default=None,
    )

    @property
    def subject(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="subject",
        )

    @property
    def timing(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="timing",
        )

    @property
    def definition(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="definition",
        )

    @field_validator(
        *(
            "action",
            "dynamicValue",
            "transform",
            "cardinalityBehavior",
            "precheckBehavior",
            "requiredBehavior",
            "selectionBehavior",
            "groupingBehavior",
            "type",
            "participant",
            "relatedAction",
            "output",
            "input",
            "condition",
            "trigger",
            "goalId",
            "documentation",
            "reason",
            "code",
            "priority",
            "textEquivalent",
            "description",
            "title",
            "prefix",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def subject_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Reference, Canonical],
            field_name_base="subject",
            required=False,
        )

    @model_validator(mode="after")
    def timing_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, Age, Period, Duration, Range, Timing],
            field_name_base="timing",
            required=False,
        )

    @model_validator(mode="after")
    def definition_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Canonical, Uri],
            field_name_base="definition",
            required=False,
        )


class PlanDefinition(DomainResource):
    """
    This resource allows for the definition of various types of plans as a sharable, consumable, and executable artifact. The resource is general enough to support the description of a broad range of clinical and non-clinical artifacts such as clinical decision support rules, order sets, protocols, and drug quality specifications.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/PlanDefinition"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this plan definition, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifier for the plan definition",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the plan definition",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    name: Optional[String] = Field(
        description="Name for this plan definition (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this plan definition (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    subtitle: Optional[String] = Field(
        description="Subordinate title of the plan definition",
        default=None,
    )
    subtitle_ext: Optional[Element] = Field(
        description="Placeholder element for subtitle extensions",
        default=None,
        alias="_subtitle",
    )
    type: Optional[CodeableConcept] = Field(
        description="order-set | clinical-protocol | eca-rule | workflow-definition",
        default=None,
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    subjectCodeableConcept: Optional[CodeableConcept] = Field(
        description="Type of individual the plan definition is focused on",
        default=None,
    )
    subjectReference: Optional[Reference] = Field(
        description="Type of individual the plan definition is focused on",
        default=None,
    )
    subjectCanonical: Optional[Canonical] = Field(
        description="Type of individual the plan definition is focused on",
        default=None,
    )
    subjectCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for subjectCanonical extensions",
        default=None,
        alias="_subjectCanonical",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the plan definition",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for plan definition (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this plan definition is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    usage: Optional[String] = Field(
        description="Describes the clinical usage of the plan",
        default=None,
    )
    usage_ext: Optional[Element] = Field(
        description="Placeholder element for usage extensions",
        default=None,
        alias="_usage",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    approvalDate: Optional[Date] = Field(
        description="When the plan definition was approved by publisher",
        default=None,
    )
    approvalDate_ext: Optional[Element] = Field(
        description="Placeholder element for approvalDate extensions",
        default=None,
        alias="_approvalDate",
    )
    lastReviewDate: Optional[Date] = Field(
        description="When the plan definition was last reviewed",
        default=None,
    )
    lastReviewDate_ext: Optional[Element] = Field(
        description="Placeholder element for lastReviewDate extensions",
        default=None,
        alias="_lastReviewDate",
    )
    effectivePeriod: Optional[Period] = Field(
        description="When the plan definition is expected to be used",
        default=None,
    )
    topic: Optional[List[CodeableConcept]] = Field(
        description="E.g. Education, Treatment, Assessment",
        default=None,
    )
    author: Optional[List[ContactDetail]] = Field(
        description="Who authored the content",
        default=None,
    )
    editor: Optional[List[ContactDetail]] = Field(
        description="Who edited the content",
        default=None,
    )
    reviewer: Optional[List[ContactDetail]] = Field(
        description="Who reviewed the content",
        default=None,
    )
    endorser: Optional[List[ContactDetail]] = Field(
        description="Who endorsed the content",
        default=None,
    )
    relatedArtifact: Optional[List[RelatedArtifact]] = Field(
        description="Additional documentation, citations",
        default=None,
    )
    library: Optional[List[Canonical]] = Field(
        description="Logic used by the plan definition",
        default=None,
    )
    library_ext: Optional[Element] = Field(
        description="Placeholder element for library extensions",
        default=None,
        alias="_library",
    )
    goal: Optional[List[PlanDefinitionGoal]] = Field(
        description="What the plan is trying to accomplish",
        default=None,
    )
    action: Optional[List[PlanDefinitionAction]] = Field(
        description="Action defined by the plan",
        default=None,
    )
    resourceType: Literal["PlanDefinition"] = Field(
        description=None,
        default="PlanDefinition",
    )

    @property
    def subject(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="subject",
        )

    @field_validator(
        *(
            "action",
            "goal",
            "library",
            "relatedArtifact",
            "endorser",
            "reviewer",
            "editor",
            "author",
            "topic",
            "effectivePeriod",
            "lastReviewDate",
            "approvalDate",
            "copyright",
            "usage",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "experimental",
            "status",
            "type",
            "subtitle",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def subject_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Reference, Canonical],
            field_name_base="subject",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_cnl_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.exists() implies name.matches('[A-Z]([A-Za-z0-9_]){0,254}')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="cnl-0",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
