# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    Instant,
    Boolean,
    Integer,
    Time,
)

from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    CodeableConcept,
    Period,
    Timing,
    Quantity,
    Range,
    Ratio,
    SampledData,
    Annotation,
    BackboneElement,
    DomainResource,
)


class ObservationReferenceRange(BackboneElement):
    """
    Guidance on how to interpret the value by comparison to a normal or recommended range.  Multiple reference ranges are interpreted as an "OR".   In other words, to represent two distinct target populations, two `referenceRange` elements would be used.
    """

    low: Optional[Quantity] = Field(
        description="Low Range, if relevant",
        default=None,
    )
    high: Optional[Quantity] = Field(
        description="High Range, if relevant",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="Reference range qualifier",
        default=None,
    )
    appliesTo: Optional[List[CodeableConcept]] = Field(
        description="Reference range population",
        default=None,
    )
    age: Optional[Range] = Field(
        description="Applicable age range, if relevant",
        default=None,
    )
    text: Optional[String] = Field(
        description="Text based reference range in an observation",
        default=None,
    )
    text_ext: Optional[Element] = Field(
        description="Placeholder element for text extensions",
        default=None,
        alias="_text",
    )

    @field_validator(
        *(
            "text",
            "age",
            "appliesTo",
            "type",
            "high",
            "low",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ObservationComponentReferenceRange(BackboneElement):
    """
    Guidance on how to interpret the value by comparison to a normal or recommended range.
    """

    low: Optional[Quantity] = Field(
        description="Low Range, if relevant",
        default=None,
    )
    high: Optional[Quantity] = Field(
        description="High Range, if relevant",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="Reference range qualifier",
        default=None,
    )
    appliesTo: Optional[List[CodeableConcept]] = Field(
        description="Reference range population",
        default=None,
    )
    age: Optional[Range] = Field(
        description="Applicable age range, if relevant",
        default=None,
    )
    text: Optional[String] = Field(
        description="Text based reference range in an observation",
        default=None,
    )
    text_ext: Optional[Element] = Field(
        description="Placeholder element for text extensions",
        default=None,
        alias="_text",
    )

    @field_validator(
        *(
            "text",
            "age",
            "appliesTo",
            "type",
            "high",
            "low",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ObservationComponent(BackboneElement):
    """
    Some observations have multiple component observations.  These component observations are expressed as separate code value pairs that share the same attributes.  Examples include systolic and diastolic component observations for blood pressure measurement and multiple component observations for genetics observations.
    """

    code: Optional[CodeableConcept] = Field(
        description="Type of component observation (code / type)",
        default=None,
    )
    valueQuantity: Optional[Quantity] = Field(
        description="Actual component result",
        default=None,
    )
    valueCodeableConcept: Optional[CodeableConcept] = Field(
        description="Actual component result",
        default=None,
    )
    valueString: Optional[String] = Field(
        description="Actual component result",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueBoolean: Optional[Boolean] = Field(
        description="Actual component result",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueInteger: Optional[Integer] = Field(
        description="Actual component result",
        default=None,
    )
    valueInteger_ext: Optional[Element] = Field(
        description="Placeholder element for valueInteger extensions",
        default=None,
        alias="_valueInteger",
    )
    valueRange: Optional[Range] = Field(
        description="Actual component result",
        default=None,
    )
    valueRatio: Optional[Ratio] = Field(
        description="Actual component result",
        default=None,
    )
    valueSampledData: Optional[SampledData] = Field(
        description="Actual component result",
        default=None,
    )
    valueTime: Optional[Time] = Field(
        description="Actual component result",
        default=None,
    )
    valueTime_ext: Optional[Element] = Field(
        description="Placeholder element for valueTime extensions",
        default=None,
        alias="_valueTime",
    )
    valueDateTime: Optional[DateTime] = Field(
        description="Actual component result",
        default=None,
    )
    valueDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for valueDateTime extensions",
        default=None,
        alias="_valueDateTime",
    )
    valuePeriod: Optional[Period] = Field(
        description="Actual component result",
        default=None,
    )
    dataAbsentReason: Optional[CodeableConcept] = Field(
        description="Why the component result is missing",
        default=None,
    )
    interpretation: Optional[List[CodeableConcept]] = Field(
        description="High, low, normal, etc.",
        default=None,
    )
    referenceRange: Optional[List[ObservationComponentReferenceRange]] = Field(
        description="Provides guide for interpretation of component result",
        default=None,
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *(
            "referenceRange",
            "interpretation",
            "dataAbsentReason",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[
                Quantity,
                CodeableConcept,
                String,
                Boolean,
                Integer,
                Range,
                Ratio,
                SampledData,
                Time,
                DateTime,
                Period,
            ],
            field_name_base="value",
            required=False,
        )


class Observation(DomainResource):
    """
    Measurements and simple assertions made about a patient, device or other subject.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Observation"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business Identifier for observation",
        default=None,
    )
    basedOn: Optional[List[Reference]] = Field(
        description="Fulfills plan, proposal or order",
        default=None,
    )
    partOf: Optional[List[Reference]] = Field(
        description="Part of referenced event",
        default=None,
    )
    status: Optional[Code] = Field(
        description="registered | preliminary | final | amended +",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Classification of  type of observation",
        default=None,
    )
    code: Optional[CodeableConcept] = Field(
        description="Type of observation (code / type)",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="Who and/or what the observation is about",
        default=None,
    )
    focus: Optional[List[Reference]] = Field(
        description="What the observation is about, when it is not about the subject of record",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Healthcare event during which this observation is made",
        default=None,
    )
    effectiveDateTime: Optional[DateTime] = Field(
        description="Clinically relevant time/time-period for observation",
        default=None,
    )
    effectiveDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for effectiveDateTime extensions",
        default=None,
        alias="_effectiveDateTime",
    )
    effectivePeriod: Optional[Period] = Field(
        description="Clinically relevant time/time-period for observation",
        default=None,
    )
    effectiveTiming: Optional[Timing] = Field(
        description="Clinically relevant time/time-period for observation",
        default=None,
    )
    effectiveInstant: Optional[Instant] = Field(
        description="Clinically relevant time/time-period for observation",
        default=None,
    )
    effectiveInstant_ext: Optional[Element] = Field(
        description="Placeholder element for effectiveInstant extensions",
        default=None,
        alias="_effectiveInstant",
    )
    issued: Optional[Instant] = Field(
        description="Date/Time this version was made available",
        default=None,
    )
    issued_ext: Optional[Element] = Field(
        description="Placeholder element for issued extensions",
        default=None,
        alias="_issued",
    )
    performer: Optional[List[Reference]] = Field(
        description="Who is responsible for the observation",
        default=None,
    )
    valueQuantity: Optional[Quantity] = Field(
        description="Actual result",
        default=None,
    )
    valueCodeableConcept: Optional[CodeableConcept] = Field(
        description="Actual result",
        default=None,
    )
    valueString: Optional[String] = Field(
        description="Actual result",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueBoolean: Optional[Boolean] = Field(
        description="Actual result",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueInteger: Optional[Integer] = Field(
        description="Actual result",
        default=None,
    )
    valueInteger_ext: Optional[Element] = Field(
        description="Placeholder element for valueInteger extensions",
        default=None,
        alias="_valueInteger",
    )
    valueRange: Optional[Range] = Field(
        description="Actual result",
        default=None,
    )
    valueRatio: Optional[Ratio] = Field(
        description="Actual result",
        default=None,
    )
    valueSampledData: Optional[SampledData] = Field(
        description="Actual result",
        default=None,
    )
    valueTime: Optional[Time] = Field(
        description="Actual result",
        default=None,
    )
    valueTime_ext: Optional[Element] = Field(
        description="Placeholder element for valueTime extensions",
        default=None,
        alias="_valueTime",
    )
    valueDateTime: Optional[DateTime] = Field(
        description="Actual result",
        default=None,
    )
    valueDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for valueDateTime extensions",
        default=None,
        alias="_valueDateTime",
    )
    valuePeriod: Optional[Period] = Field(
        description="Actual result",
        default=None,
    )
    dataAbsentReason: Optional[CodeableConcept] = Field(
        description="Why the result is missing",
        default=None,
    )
    interpretation: Optional[List[CodeableConcept]] = Field(
        description="High, low, normal, etc.",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Comments about the observation",
        default=None,
    )
    bodySite: Optional[CodeableConcept] = Field(
        description="Observed body part",
        default=None,
    )
    method: Optional[CodeableConcept] = Field(
        description="How it was done",
        default=None,
    )
    specimen: Optional[Reference] = Field(
        description="Specimen used for this observation",
        default=None,
    )
    device: Optional[Reference] = Field(
        description="(Measurement) Device",
        default=None,
    )
    referenceRange: Optional[List[ObservationReferenceRange]] = Field(
        description="Provides guide for interpretation",
        default=None,
    )
    hasMember: Optional[List[Reference]] = Field(
        description="Related resource that belongs to the Observation group",
        default=None,
    )
    derivedFrom: Optional[List[Reference]] = Field(
        description="Related measurements the observation is made from",
        default=None,
    )
    component: Optional[List[ObservationComponent]] = Field(
        description="Component results",
        default=None,
    )
    resourceType: Literal["Observation"] = Field(
        description=None,
        default="Observation",
    )

    @property
    def effective(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="effective",
        )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *(
            "component",
            "derivedFrom",
            "hasMember",
            "referenceRange",
            "device",
            "specimen",
            "method",
            "bodySite",
            "note",
            "interpretation",
            "dataAbsentReason",
            "performer",
            "issued",
            "encounter",
            "focus",
            "subject",
            "code",
            "category",
            "status",
            "partOf",
            "basedOn",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("referenceRange",), mode="after", check_fields=None)
    @classmethod
    def FHIR_obs_3_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="low.exists() or high.exists() or text.exists()",
            human="Must have at least a low or a high or text",
            key="obs-3",
            severity="error",
        )

    @model_validator(mode="after")
    def effective_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, Period, Timing, Instant],
            field_name_base="effective",
            required=False,
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[
                Quantity,
                CodeableConcept,
                String,
                Boolean,
                Integer,
                Range,
                Ratio,
                SampledData,
                Time,
                DateTime,
                Period,
            ],
            field_name_base="value",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_obs_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="dataAbsentReason.empty() or value.empty()",
            human="dataAbsentReason SHALL only be present if Observation.value[x] is not present",
            key="obs-6",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_obs_7_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="value.empty() or component.code.where(coding.intersect(%resource.code.coding).exists()).empty()",
            human="If Observation.code is the same as an Observation.component.code then the value element associated with the code SHALL NOT be present",
            key="obs-7",
            severity="error",
        )
