# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    DateTime,
    Canonical,
    UnsignedInt,
)

from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    Annotation,
    Dosage,
    BackboneElement,
    Quantity,
    Duration,
    Period,
    DomainResource,
)


class MedicationRequestDispenseRequestInitialFill(BackboneElement):
    """
    Indicates the quantity or duration for the first dispense of the medication.
    """

    quantity: Optional[Quantity] = Field(
        description="First fill quantity",
        default=None,
    )
    duration: Optional[Duration] = Field(
        description="First fill duration",
        default=None,
    )

    @field_validator(
        *(
            "duration",
            "quantity",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationRequestDispenseRequest(BackboneElement):
    """
    Indicates the specific details for the dispense or medication supply part of a medication request (also known as a Medication Prescription or Medication Order).  Note that this information is not always sent with the order.  There may be in some settings (e.g. hospitals) institutional or system support for completing the dispense details in the pharmacy department.
    """

    initialFill: Optional[MedicationRequestDispenseRequestInitialFill] = Field(
        description="First fill details",
        default=None,
    )
    dispenseInterval: Optional[Duration] = Field(
        description="Minimum period of time between dispenses",
        default=None,
    )
    validityPeriod: Optional[Period] = Field(
        description="Time period supply is authorized for",
        default=None,
    )
    numberOfRepeatsAllowed: Optional[UnsignedInt] = Field(
        description="Number of refills authorized",
        default=None,
    )
    numberOfRepeatsAllowed_ext: Optional[Element] = Field(
        description="Placeholder element for numberOfRepeatsAllowed extensions",
        default=None,
        alias="_numberOfRepeatsAllowed",
    )
    quantity: Optional[Quantity] = Field(
        description="Amount of medication to supply per dispense",
        default=None,
    )
    expectedSupplyDuration: Optional[Duration] = Field(
        description="Number of days supply per dispense",
        default=None,
    )
    performer: Optional[Reference] = Field(
        description="Intended dispenser",
        default=None,
    )

    @field_validator(
        *(
            "performer",
            "expectedSupplyDuration",
            "quantity",
            "numberOfRepeatsAllowed",
            "validityPeriod",
            "dispenseInterval",
            "initialFill",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationRequestSubstitution(BackboneElement):
    """
    Indicates whether or not substitution can or should be part of the dispense. In some cases, substitution must happen, in other cases substitution must not happen. This block explains the prescriber's intent. If nothing is specified substitution may be done.
    """

    allowedBoolean: Optional[Boolean] = Field(
        description="Whether substitution is allowed or not",
        default=None,
    )
    allowedBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for allowedBoolean extensions",
        default=None,
        alias="_allowedBoolean",
    )
    allowedCodeableConcept: Optional[CodeableConcept] = Field(
        description="Whether substitution is allowed or not",
        default=None,
    )
    reason: Optional[CodeableConcept] = Field(
        description="Why should (not) substitution be made",
        default=None,
    )

    @property
    def allowed(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="allowed",
        )

    @field_validator(
        *("reason", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def allowed_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Boolean, CodeableConcept],
            field_name_base="allowed",
            required=True,
        )


class MedicationRequest(DomainResource):
    """
    An order or request for both supply of the medication and the instructions for administration of the medication to a patient. The resource is called "MedicationRequest" rather than "MedicationPrescription" or "MedicationOrder" to generalize the use across inpatient and outpatient settings, including care plans, etc., and to harmonize with workflow patterns.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/MedicationRequest"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="External ids for this request",
        default=None,
    )
    status: Optional[Code] = Field(
        description="active | on-hold | cancelled | completed | entered-in-error | stopped | draft | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    statusReason: Optional[CodeableConcept] = Field(
        description="Reason for current status",
        default=None,
    )
    intent: Optional[Code] = Field(
        description="proposal | plan | order | original-order | reflex-order | filler-order | instance-order | option",
        default=None,
    )
    intent_ext: Optional[Element] = Field(
        description="Placeholder element for intent extensions",
        default=None,
        alias="_intent",
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Type of medication usage",
        default=None,
    )
    priority: Optional[Code] = Field(
        description="routine | urgent | asap | stat",
        default=None,
    )
    priority_ext: Optional[Element] = Field(
        description="Placeholder element for priority extensions",
        default=None,
        alias="_priority",
    )
    doNotPerform: Optional[Boolean] = Field(
        description="True if request is prohibiting action",
        default=None,
    )
    doNotPerform_ext: Optional[Element] = Field(
        description="Placeholder element for doNotPerform extensions",
        default=None,
        alias="_doNotPerform",
    )
    reportedBoolean: Optional[Boolean] = Field(
        description="Reported rather than primary record",
        default=None,
    )
    reportedBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for reportedBoolean extensions",
        default=None,
        alias="_reportedBoolean",
    )
    reportedReference: Optional[Reference] = Field(
        description="Reported rather than primary record",
        default=None,
    )
    medicationCodeableConcept: Optional[CodeableConcept] = Field(
        description="Medication to be taken",
        default=None,
    )
    medicationReference: Optional[Reference] = Field(
        description="Medication to be taken",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="Who or group medication request is for",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Encounter created as part of encounter/admission/stay",
        default=None,
    )
    supportingInformation: Optional[List[Reference]] = Field(
        description="Information to support ordering of the medication",
        default=None,
    )
    authoredOn: Optional[DateTime] = Field(
        description="When request was initially authored",
        default=None,
    )
    authoredOn_ext: Optional[Element] = Field(
        description="Placeholder element for authoredOn extensions",
        default=None,
        alias="_authoredOn",
    )
    requester: Optional[Reference] = Field(
        description="Who/What requested the Request",
        default=None,
    )
    performer: Optional[Reference] = Field(
        description="Intended performer of administration",
        default=None,
    )
    performerType: Optional[CodeableConcept] = Field(
        description="Desired kind of performer of the medication administration",
        default=None,
    )
    recorder: Optional[Reference] = Field(
        description="Person who entered the request",
        default=None,
    )
    reasonCode: Optional[List[CodeableConcept]] = Field(
        description="Reason or indication for ordering or not ordering the medication",
        default=None,
    )
    reasonReference: Optional[List[Reference]] = Field(
        description="Condition or observation that supports why the prescription is being written",
        default=None,
    )
    instantiatesCanonical: Optional[List[Canonical]] = Field(
        description="Instantiates FHIR protocol or definition",
        default=None,
    )
    instantiatesCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesCanonical extensions",
        default=None,
        alias="_instantiatesCanonical",
    )
    instantiatesUri: Optional[List[Uri]] = Field(
        description="Instantiates external protocol or definition",
        default=None,
    )
    instantiatesUri_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesUri extensions",
        default=None,
        alias="_instantiatesUri",
    )
    basedOn: Optional[List[Reference]] = Field(
        description="What request fulfills",
        default=None,
    )
    groupIdentifier: Optional[Identifier] = Field(
        description="Composite request this is part of",
        default=None,
    )
    courseOfTherapyType: Optional[CodeableConcept] = Field(
        description="Overall pattern of medication administration",
        default=None,
    )
    insurance: Optional[List[Reference]] = Field(
        description="Associated insurance coverage",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Information about the prescription",
        default=None,
    )
    dosageInstruction: Optional[List[Dosage]] = Field(
        description="How the medication should be taken",
        default=None,
    )
    dispenseRequest: Optional[MedicationRequestDispenseRequest] = Field(
        description="Medication supply authorization",
        default=None,
    )
    substitution: Optional[MedicationRequestSubstitution] = Field(
        description="Any restrictions on medication substitution",
        default=None,
    )
    priorPrescription: Optional[Reference] = Field(
        description="An order/prescription that is being replaced",
        default=None,
    )
    detectedIssue: Optional[List[Reference]] = Field(
        description="Clinical Issue with action",
        default=None,
    )
    eventHistory: Optional[List[Reference]] = Field(
        description="A list of events of interest in the lifecycle",
        default=None,
    )
    resourceType: Literal["MedicationRequest"] = Field(
        description=None,
        default="MedicationRequest",
    )

    @property
    def reported(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="reported",
        )

    @property
    def medication(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="medication",
        )

    @field_validator(
        *(
            "eventHistory",
            "detectedIssue",
            "priorPrescription",
            "substitution",
            "dispenseRequest",
            "dosageInstruction",
            "note",
            "insurance",
            "courseOfTherapyType",
            "groupIdentifier",
            "basedOn",
            "instantiatesUri",
            "instantiatesCanonical",
            "reasonReference",
            "reasonCode",
            "recorder",
            "performerType",
            "performer",
            "requester",
            "authoredOn",
            "supportingInformation",
            "encounter",
            "subject",
            "doNotPerform",
            "priority",
            "category",
            "intent",
            "statusReason",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def reported_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Boolean, Reference],
            field_name_base="reported",
            required=False,
        )

    @model_validator(mode="after")
    def medication_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Reference],
            field_name_base="medication",
            required=True,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
