# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Canonical,
    DateTime,
    Integer,
)

from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    Period,
    CodeableConcept,
    BackboneElement,
    Quantity,
    DomainResource,
)


class MeasureReportGroupPopulation(BackboneElement):
    """
    The populations that make up the population group, one for each type of population appropriate for the measure.
    """

    code: Optional[CodeableConcept] = Field(
        description="initial-population | numerator | numerator-exclusion | denominator | denominator-exclusion | denominator-exception | measure-population | measure-population-exclusion | measure-observation",
        default=None,
    )
    count: Optional[Integer] = Field(
        description="Size of the population",
        default=None,
    )
    count_ext: Optional[Element] = Field(
        description="Placeholder element for count extensions",
        default=None,
        alias="_count",
    )
    subjectResults: Optional[Reference] = Field(
        description="For subject-list reports, the subject results in this population",
        default=None,
    )

    @field_validator(
        *(
            "subjectResults",
            "count",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MeasureReportGroupStratifierStratumComponent(BackboneElement):
    """
    A stratifier component value.
    """

    code: Optional[CodeableConcept] = Field(
        description="What stratifier component of the group",
        default=None,
    )
    value: Optional[CodeableConcept] = Field(
        description="The stratum component value, e.g. male",
        default=None,
    )

    @field_validator(
        *(
            "value",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MeasureReportGroupStratifierStratumPopulation(BackboneElement):
    """
    The populations that make up the stratum, one for each type of population appropriate to the measure.
    """

    code: Optional[CodeableConcept] = Field(
        description="initial-population | numerator | numerator-exclusion | denominator | denominator-exclusion | denominator-exception | measure-population | measure-population-exclusion | measure-observation",
        default=None,
    )
    count: Optional[Integer] = Field(
        description="Size of the population",
        default=None,
    )
    count_ext: Optional[Element] = Field(
        description="Placeholder element for count extensions",
        default=None,
        alias="_count",
    )
    subjectResults: Optional[Reference] = Field(
        description="For subject-list reports, the subject results in this population",
        default=None,
    )

    @field_validator(
        *(
            "subjectResults",
            "count",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MeasureReportGroupStratifierStratum(BackboneElement):
    """
    This element contains the results for a single stratum within the stratifier. For example, when stratifying on administrative gender, there will be four strata, one for each possible gender value.
    """

    value: Optional[CodeableConcept] = Field(
        description="The stratum value, e.g. male",
        default=None,
    )
    component: Optional[List[MeasureReportGroupStratifierStratumComponent]] = Field(
        description="Stratifier component values",
        default=None,
    )
    population: Optional[List[MeasureReportGroupStratifierStratumPopulation]] = Field(
        description="Population results in this stratum",
        default=None,
    )
    measureScore: Optional[Quantity] = Field(
        description="What score this stratum achieved",
        default=None,
    )

    @field_validator(
        *(
            "measureScore",
            "population",
            "component",
            "value",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MeasureReportGroupStratifier(BackboneElement):
    """
    When a measure includes multiple stratifiers, there will be a stratifier group for each stratifier defined by the measure.
    """

    code: Optional[List[CodeableConcept]] = Field(
        description="What stratifier of the group",
        default=None,
    )
    stratum: Optional[List[MeasureReportGroupStratifierStratum]] = Field(
        description="Stratum results, one for each unique value, or set of values, in the stratifier, or stratifier components",
        default=None,
    )

    @field_validator(
        *(
            "stratum",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )


class MeasureReportGroup(BackboneElement):
    """
    The results of the calculation, one for each population group in the measure.
    """

    code: Optional[CodeableConcept] = Field(
        description="Meaning of the group",
        default=None,
    )
    population: Optional[List[MeasureReportGroupPopulation]] = Field(
        description="The populations in the group",
        default=None,
    )
    measureScore: Optional[Quantity] = Field(
        description="What score this group achieved",
        default=None,
    )
    stratifier: Optional[List[MeasureReportGroupStratifier]] = Field(
        description="Stratification results",
        default=None,
    )

    @field_validator(
        *(
            "stratifier",
            "measureScore",
            "population",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )


class MeasureReport(DomainResource):
    """
    The MeasureReport resource contains the results of the calculation of a measure; and optionally a reference to the resources involved in that calculation.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/MeasureReport"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifier for the MeasureReport",
        default=None,
    )
    status: Optional[Code] = Field(
        description="complete | pending | error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    type: Optional[Code] = Field(
        description="individual | subject-list | summary | data-collection",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    measure: Optional[Canonical] = Field(
        description="What measure was calculated",
        default=None,
    )
    measure_ext: Optional[Element] = Field(
        description="Placeholder element for measure extensions",
        default=None,
        alias="_measure",
    )
    subject: Optional[Reference] = Field(
        description="What individual(s) the report is for",
        default=None,
    )
    date: Optional[DateTime] = Field(
        description="When the report was generated",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    reporter: Optional[Reference] = Field(
        description="Who is reporting the data",
        default=None,
    )
    period: Optional[Period] = Field(
        description="What period the report covers",
        default=None,
    )
    improvementNotation: Optional[CodeableConcept] = Field(
        description="increase | decrease",
        default=None,
    )
    group: Optional[List[MeasureReportGroup]] = Field(
        description="Measure results for each group",
        default=None,
    )
    evaluatedResource: Optional[List[Reference]] = Field(
        description="What data was used to calculate the measure score",
        default=None,
    )
    resourceType: Literal["MeasureReport"] = Field(
        description=None,
        default="MeasureReport",
    )

    @field_validator(
        *(
            "evaluatedResource",
            "group",
            "improvementNotation",
            "period",
            "reporter",
            "date",
            "subject",
            "measure",
            "type",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_mrp_1_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="(type != 'data-collection') or group.exists().not()",
            human="Measure Reports used for data collection SHALL NOT communicate group and score information",
            key="mrp-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_mrp_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="group.stratifier.stratum.all(value.exists() xor component.exists())",
            human="Stratifiers SHALL be either a single criteria or a set of criteria components",
            key="mrp-2",
            severity="error",
        )
