# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    DateTime,
    Markdown,
    Date,
    Canonical,
)

from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    ContactDetail,
    UsageContext,
    Period,
    RelatedArtifact,
    BackboneElement,
    Expression,
    DomainResource,
)


class MeasureGroupPopulation(BackboneElement):
    """
    A population criteria for the measure.
    """

    code: Optional[CodeableConcept] = Field(
        description="initial-population | numerator | numerator-exclusion | denominator | denominator-exclusion | denominator-exception | measure-population | measure-population-exclusion | measure-observation",
        default=None,
    )
    description: Optional[String] = Field(
        description="The human readable description of this population criteria",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    criteria: Optional[Expression] = Field(
        description="The criteria that defines this population",
        default=None,
    )

    @field_validator(
        *(
            "criteria",
            "description",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MeasureGroupStratifierComponent(BackboneElement):
    """
    A component of the stratifier criteria for the measure report, specified as either the name of a valid CQL expression defined within a referenced library or a valid FHIR Resource Path.
    """

    code: Optional[CodeableConcept] = Field(
        description="Meaning of the stratifier component",
        default=None,
    )
    description: Optional[String] = Field(
        description="The human readable description of this stratifier component",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    criteria: Optional[Expression] = Field(
        description="Component of how the measure should be stratified",
        default=None,
    )

    @field_validator(
        *(
            "criteria",
            "description",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MeasureGroupStratifier(BackboneElement):
    """
    The stratifier criteria for the measure report, specified as either the name of a valid CQL expression defined within a referenced library or a valid FHIR Resource Path.
    """

    code: Optional[CodeableConcept] = Field(
        description="Meaning of the stratifier",
        default=None,
    )
    description: Optional[String] = Field(
        description="The human readable description of this stratifier",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    criteria: Optional[Expression] = Field(
        description="How the measure should be stratified",
        default=None,
    )
    component: Optional[List[MeasureGroupStratifierComponent]] = Field(
        description="Stratifier criteria component for the measure",
        default=None,
    )

    @field_validator(
        *(
            "component",
            "criteria",
            "description",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )


class MeasureGroup(BackboneElement):
    """
    A group of population criteria for the measure.
    """

    code: Optional[CodeableConcept] = Field(
        description="Meaning of the group",
        default=None,
    )
    description: Optional[String] = Field(
        description="Summary description",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    population: Optional[List[MeasureGroupPopulation]] = Field(
        description="Population criteria",
        default=None,
    )
    stratifier: Optional[List[MeasureGroupStratifier]] = Field(
        description="Stratifier criteria for the measure",
        default=None,
    )

    @field_validator(
        *(
            "stratifier",
            "population",
            "description",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )


class MeasureSupplementalData(BackboneElement):
    """
    The supplemental data criteria for the measure report, specified as either the name of a valid CQL expression within a referenced library, or a valid FHIR Resource Path.
    """

    code: Optional[CodeableConcept] = Field(
        description="Meaning of the supplemental data",
        default=None,
    )
    usage: Optional[List[CodeableConcept]] = Field(
        description="supplemental-data | risk-adjustment-factor",
        default=None,
    )
    description: Optional[String] = Field(
        description="The human readable description of this supplemental data",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    criteria: Optional[Expression] = Field(
        description="Expression describing additional data to be reported",
        default=None,
    )

    @field_validator(
        *(
            "criteria",
            "description",
            "usage",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class Measure(DomainResource):
    """
    The Measure resource provides the definition of a quality measure.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Measure"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this measure, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifier for the measure",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the measure",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    name: Optional[String] = Field(
        description="Name for this measure (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this measure (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    subtitle: Optional[String] = Field(
        description="Subordinate title of the measure",
        default=None,
    )
    subtitle_ext: Optional[Element] = Field(
        description="Placeholder element for subtitle extensions",
        default=None,
        alias="_subtitle",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    subjectCodeableConcept: Optional[CodeableConcept] = Field(
        description="E.g. Patient, Practitioner, RelatedPerson, Organization, Location, Device",
        default=None,
    )
    subjectReference: Optional[Reference] = Field(
        description="E.g. Patient, Practitioner, RelatedPerson, Organization, Location, Device",
        default=None,
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the measure",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for measure (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this measure is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    usage: Optional[String] = Field(
        description="Describes the clinical usage of the measure",
        default=None,
    )
    usage_ext: Optional[Element] = Field(
        description="Placeholder element for usage extensions",
        default=None,
        alias="_usage",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    approvalDate: Optional[Date] = Field(
        description="When the measure was approved by publisher",
        default=None,
    )
    approvalDate_ext: Optional[Element] = Field(
        description="Placeholder element for approvalDate extensions",
        default=None,
        alias="_approvalDate",
    )
    lastReviewDate: Optional[Date] = Field(
        description="When the measure was last reviewed",
        default=None,
    )
    lastReviewDate_ext: Optional[Element] = Field(
        description="Placeholder element for lastReviewDate extensions",
        default=None,
        alias="_lastReviewDate",
    )
    effectivePeriod: Optional[Period] = Field(
        description="When the measure is expected to be used",
        default=None,
    )
    topic: Optional[List[CodeableConcept]] = Field(
        description="The category of the measure, such as Education, Treatment, Assessment, etc.",
        default=None,
    )
    author: Optional[List[ContactDetail]] = Field(
        description="Who authored the content",
        default=None,
    )
    editor: Optional[List[ContactDetail]] = Field(
        description="Who edited the content",
        default=None,
    )
    reviewer: Optional[List[ContactDetail]] = Field(
        description="Who reviewed the content",
        default=None,
    )
    endorser: Optional[List[ContactDetail]] = Field(
        description="Who endorsed the content",
        default=None,
    )
    relatedArtifact: Optional[List[RelatedArtifact]] = Field(
        description="Additional documentation, citations, etc.",
        default=None,
    )
    library: Optional[List[Canonical]] = Field(
        description="Logic used by the measure",
        default=None,
    )
    library_ext: Optional[Element] = Field(
        description="Placeholder element for library extensions",
        default=None,
        alias="_library",
    )
    disclaimer: Optional[Markdown] = Field(
        description="Disclaimer for use of the measure or its referenced content",
        default=None,
    )
    disclaimer_ext: Optional[Element] = Field(
        description="Placeholder element for disclaimer extensions",
        default=None,
        alias="_disclaimer",
    )
    scoring: Optional[CodeableConcept] = Field(
        description="proportion | ratio | continuous-variable | cohort",
        default=None,
    )
    compositeScoring: Optional[CodeableConcept] = Field(
        description="opportunity | all-or-nothing | linear | weighted",
        default=None,
    )
    type: Optional[List[CodeableConcept]] = Field(
        description="process | outcome | structure | patient-reported-outcome | composite",
        default=None,
    )
    riskAdjustment: Optional[String] = Field(
        description="How risk adjustment is applied for this measure",
        default=None,
    )
    riskAdjustment_ext: Optional[Element] = Field(
        description="Placeholder element for riskAdjustment extensions",
        default=None,
        alias="_riskAdjustment",
    )
    rateAggregation: Optional[String] = Field(
        description="How is rate aggregation performed for this measure",
        default=None,
    )
    rateAggregation_ext: Optional[Element] = Field(
        description="Placeholder element for rateAggregation extensions",
        default=None,
        alias="_rateAggregation",
    )
    rationale: Optional[Markdown] = Field(
        description="Detailed description of why the measure exists",
        default=None,
    )
    rationale_ext: Optional[Element] = Field(
        description="Placeholder element for rationale extensions",
        default=None,
        alias="_rationale",
    )
    clinicalRecommendationStatement: Optional[Markdown] = Field(
        description="Summary of clinical guidelines",
        default=None,
    )
    clinicalRecommendationStatement_ext: Optional[Element] = Field(
        description="Placeholder element for clinicalRecommendationStatement extensions",
        default=None,
        alias="_clinicalRecommendationStatement",
    )
    improvementNotation: Optional[CodeableConcept] = Field(
        description="increase | decrease",
        default=None,
    )
    definition: Optional[List[Markdown]] = Field(
        description="Defined terms used in the measure documentation",
        default=None,
    )
    definition_ext: Optional[Element] = Field(
        description="Placeholder element for definition extensions",
        default=None,
        alias="_definition",
    )
    guidance: Optional[Markdown] = Field(
        description="Additional guidance for implementers",
        default=None,
    )
    guidance_ext: Optional[Element] = Field(
        description="Placeholder element for guidance extensions",
        default=None,
        alias="_guidance",
    )
    group: Optional[List[MeasureGroup]] = Field(
        description="Population criteria group",
        default=None,
    )
    supplementalData: Optional[List[MeasureSupplementalData]] = Field(
        description="What other data should be reported with the measure",
        default=None,
    )
    resourceType: Literal["Measure"] = Field(
        description=None,
        default="Measure",
    )

    @property
    def subject(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="subject",
        )

    @field_validator(
        *(
            "supplementalData",
            "group",
            "guidance",
            "definition",
            "improvementNotation",
            "clinicalRecommendationStatement",
            "rationale",
            "rateAggregation",
            "riskAdjustment",
            "type",
            "compositeScoring",
            "scoring",
            "disclaimer",
            "library",
            "relatedArtifact",
            "endorser",
            "reviewer",
            "editor",
            "author",
            "topic",
            "effectivePeriod",
            "lastReviewDate",
            "approvalDate",
            "copyright",
            "usage",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "experimental",
            "status",
            "subtitle",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def subject_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Reference],
            field_name_base="subject",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_cnl_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.exists() implies name.matches('[A-Z]([A-Za-z0-9_]){0,254}')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="cnl-0",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_mea_1_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="group.stratifier.all((code | description | criteria).exists() xor component.exists())",
            human="Stratifier SHALL be either a single criteria or a set of criteria components",
            key="mea-1",
            severity="error",
        )
