# Fhircraft modules
from enum import Enum

# Standard modules
from typing import Literal, Optional, Union

# Pydantic modules
from pydantic import BaseModel, Field, field_validator, model_validator
from pydantic.fields import FieldInfo

import fhircraft
import fhircraft.fhir.resources.validators as fhir_validators
from fhircraft.fhir.resources.base import FHIRBaseModel
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.utils import model_rebuild_all

NoneType = type(None)

# Dynamic modules

from typing import List as ListType
from typing import Literal, Optional

from fhircraft.fhir.resources.base import FHIRBaseModel
from fhircraft.fhir.resources.datatypes.primitives import (
    Boolean,
    Code,
    DateTime,
    String,
    Uri,
)
from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Annotation,
    BackboneElement,
    CodeableConcept,
    Element,
    Extension,
    Identifier,
    Meta,
    Narrative,
    Reference,
    Resource,
    DomainResource,
)


class ListEntry(BackboneElement):
    """
    Entries in this list.
    """

    flag: Optional[CodeableConcept] = Field(
        description="Status/Workflow information about this item",
        default=None,
    )
    deleted: Optional[Boolean] = Field(
        description="If this item is actually marked as deleted",
        default=None,
    )
    deleted_ext: Optional[Element] = Field(
        description="Placeholder element for deleted extensions",
        default=None,
        alias="_deleted",
    )
    date: Optional[DateTime] = Field(
        description="When item added to list",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    item: Optional[Reference] = Field(
        description="Actual entry",
        default=None,
    )

    @field_validator(
        *(
            "item",
            "date",
            "deleted",
            "flag",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class List(DomainResource):
    """
    A list is a curated collection of resources.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/List"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[ListType[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[ListType[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[ListType[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[ListType[Identifier]] = Field(
        description="Business identifier",
        default=None,
    )
    status: Optional[Code] = Field(
        description="current | retired | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    mode: Optional[Code] = Field(
        description="working | snapshot | changes",
        default=None,
    )
    mode_ext: Optional[Element] = Field(
        description="Placeholder element for mode extensions",
        default=None,
        alias="_mode",
    )
    title: Optional[String] = Field(
        description="Descriptive name for the list",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    code: Optional[CodeableConcept] = Field(
        description="What the purpose of this list is",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="If all resources have the same subject",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Context in which list created",
        default=None,
    )
    date: Optional[DateTime] = Field(
        description="When the list was prepared",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    source: Optional[Reference] = Field(
        description="Who and/or what defined the list contents (aka Author)",
        default=None,
    )
    orderedBy: Optional[CodeableConcept] = Field(
        description="What order the list has",
        default=None,
    )
    note: Optional[ListType[Annotation]] = Field(
        description="Comments about the list",
        default=None,
    )
    entry: Optional[ListType[ListEntry]] = Field(
        description="Entries in the list",
        default=None,
    )
    emptyReason: Optional[CodeableConcept] = Field(
        description="Why list is empty",
        default=None,
    )
    resourceType: Literal["List"] = Field(
        description=None,
        default="List",
    )

    @field_validator(
        *(
            "emptyReason",
            "entry",
            "note",
            "orderedBy",
            "source",
            "date",
            "encounter",
            "subject",
            "code",
            "title",
            "mode",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_lst_1_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="emptyReason.empty() or entry.empty()",
            human="A list can only have an emptyReason if it is empty",
            key="lst-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_lst_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="mode = 'changes' or entry.deleted.empty()",
            human='The deleted flag can only be used if the mode of the list is "changes"',
            key="lst-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_lst_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="mode = 'working' or entry.date.empty()",
            human='An entry date can only be used if the mode of the list is "working"',
            key="lst-3",
            severity="error",
        )
