# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code, Boolean

from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    CodeableConcept,
    BackboneElement,
    CodeableReference,
    Ratio,
    RatioRange,
    DomainResource,
)


class IngredientManufacturer(BackboneElement):
    """
    The organization(s) that manufacture this ingredient. Can be used to indicate:         1) Organizations we are aware of that manufacture this ingredient         2) Specific Manufacturer(s) currently being used         3) Set of organisations allowed to manufacture this ingredient for this product         Users must be clear on the application of context relevant to their use case.
    """

    role: Optional[Code] = Field(
        description="allowed | possible | actual",
        default=None,
    )
    role_ext: Optional[Element] = Field(
        description="Placeholder element for role extensions",
        default=None,
        alias="_role",
    )
    manufacturer: Optional[Reference] = Field(
        description="An organization that manufactures this ingredient",
        default=None,
    )

    @field_validator(
        *(
            "manufacturer",
            "role",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class IngredientSubstanceStrengthReferenceStrength(BackboneElement):
    """
    Strength expressed in terms of a reference substance. For when the ingredient strength is additionally expressed as equivalent to the strength of some other closely related substance (e.g. salt vs. base). Reference strength represents the strength (quantitative composition) of the active moiety of the active substance. There are situations when the active substance and active moiety are different, therefore both a strength and a reference strength are needed.
    """

    substance: Optional[CodeableReference] = Field(
        description="Relevant reference substance",
        default=None,
    )
    strengthRatio: Optional[Ratio] = Field(
        description="Strength expressed in terms of a reference substance",
        default=None,
    )
    strengthRatioRange: Optional[RatioRange] = Field(
        description="Strength expressed in terms of a reference substance",
        default=None,
    )
    measurementPoint: Optional[String] = Field(
        description="When strength is measured at a particular point or distance",
        default=None,
    )
    measurementPoint_ext: Optional[Element] = Field(
        description="Placeholder element for measurementPoint extensions",
        default=None,
        alias="_measurementPoint",
    )
    country: Optional[List[CodeableConcept]] = Field(
        description="Where the strength range applies",
        default=None,
    )

    @property
    def strength(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="strength",
        )

    @field_validator(
        *(
            "country",
            "measurementPoint",
            "substance",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def strength_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Ratio, RatioRange],
            field_name_base="strength",
            required=True,
        )


class IngredientSubstanceStrength(BackboneElement):
    """
    The quantity of substance in the unit of presentation, or in the volume (or mass) of the single pharmaceutical product or manufactured item. The allowed repetitions do not represent different strengths, but are different representations - mathematically equivalent - of a single strength.
    """

    presentationRatio: Optional[Ratio] = Field(
        description="The quantity of substance in the unit of presentation",
        default=None,
    )
    presentationRatioRange: Optional[RatioRange] = Field(
        description="The quantity of substance in the unit of presentation",
        default=None,
    )
    textPresentation: Optional[String] = Field(
        description="Text of either the whole presentation strength or a part of it (rest being in Strength.presentation as a ratio)",
        default=None,
    )
    textPresentation_ext: Optional[Element] = Field(
        description="Placeholder element for textPresentation extensions",
        default=None,
        alias="_textPresentation",
    )
    concentrationRatio: Optional[Ratio] = Field(
        description="The strength per unitary volume (or mass)",
        default=None,
    )
    concentrationRatioRange: Optional[RatioRange] = Field(
        description="The strength per unitary volume (or mass)",
        default=None,
    )
    textConcentration: Optional[String] = Field(
        description="Text of either the whole concentration strength or a part of it (rest being in Strength.concentration as a ratio)",
        default=None,
    )
    textConcentration_ext: Optional[Element] = Field(
        description="Placeholder element for textConcentration extensions",
        default=None,
        alias="_textConcentration",
    )
    measurementPoint: Optional[String] = Field(
        description="When strength is measured at a particular point or distance",
        default=None,
    )
    measurementPoint_ext: Optional[Element] = Field(
        description="Placeholder element for measurementPoint extensions",
        default=None,
        alias="_measurementPoint",
    )
    country: Optional[List[CodeableConcept]] = Field(
        description="Where the strength range applies",
        default=None,
    )
    referenceStrength: Optional[List[IngredientSubstanceStrengthReferenceStrength]] = (
        Field(
            description="Strength expressed in terms of a reference substance",
            default=None,
        )
    )

    @property
    def presentation(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="presentation",
        )

    @property
    def concentration(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="concentration",
        )

    @field_validator(
        *(
            "referenceStrength",
            "country",
            "measurementPoint",
            "textConcentration",
            "textPresentation",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def presentation_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Ratio, RatioRange],
            field_name_base="presentation",
            required=False,
        )

    @model_validator(mode="after")
    def concentration_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Ratio, RatioRange],
            field_name_base="concentration",
            required=False,
        )


class IngredientSubstance(BackboneElement):
    """
    The substance that comprises this ingredient.
    """

    code: Optional[CodeableReference] = Field(
        description="A code or full resource that represents the ingredient substance",
        default=None,
    )
    strength: Optional[List[IngredientSubstanceStrength]] = Field(
        description="The quantity of substance, per presentation, or per volume or mass, and type of quantity",
        default=None,
    )

    @field_validator(
        *(
            "strength",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )


class Ingredient(DomainResource):
    """
    An ingredient of a manufactured item or pharmaceutical product.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Ingredient"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[Identifier] = Field(
        description="An identifier or code by which the ingredient can be referenced",
        default=None,
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    for_: Optional[List[Reference]] = Field(
        description="The product which this ingredient is a constituent part of",
        default=None,
    )
    role: Optional[CodeableConcept] = Field(
        description="Purpose of the ingredient within the product, e.g. active, inactive",
        default=None,
    )
    function: Optional[List[CodeableConcept]] = Field(
        description="Precise action within the drug product, e.g. antioxidant, alkalizing agent",
        default=None,
    )
    allergenicIndicator: Optional[Boolean] = Field(
        description="If the ingredient is a known or suspected allergen",
        default=None,
    )
    allergenicIndicator_ext: Optional[Element] = Field(
        description="Placeholder element for allergenicIndicator extensions",
        default=None,
        alias="_allergenicIndicator",
    )
    manufacturer: Optional[List[IngredientManufacturer]] = Field(
        description="An organization that manufactures this ingredient",
        default=None,
    )
    substance: Optional[IngredientSubstance] = Field(
        description="The substance that comprises this ingredient",
        default=None,
    )
    resourceType: Literal["Ingredient"] = Field(
        description=None,
        default="Ingredient",
    )

    @field_validator(
        *(
            "substance",
            "manufacturer",
            "allergenicIndicator",
            "function",
            "role",
            "for_",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_ing_1_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="(Ingredient.allergenicIndicator.where(value='true').count() + Ingredient.substance.code.reference.count())  < 2",
            human="If an ingredient is noted as an allergen (allergenicIndicator) then its substance should be a code. If the substance is a SubstanceDefinition, then the allegen information should be documented in that resource",
            key="ing-1",
            severity="error",
        )
