# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    DateTime,
    Markdown,
    Id,
    Canonical,
    Url,
)

from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    ContactDetail,
    UsageContext,
    CodeableConcept,
    BackboneElement,
    Reference,
    DomainResource,
)


class ImplementationGuideDependsOn(BackboneElement):
    """
    Another implementation guide that this implementation depends on. Typically, an implementation guide uses value sets, profiles etc.defined in other implementation guides.
    """

    uri: Optional[Canonical] = Field(
        description="Identity of the IG that this depends on",
        default=None,
    )
    uri_ext: Optional[Element] = Field(
        description="Placeholder element for uri extensions",
        default=None,
        alias="_uri",
    )
    packageId: Optional[Id] = Field(
        description="NPM Package name for IG this depends on",
        default=None,
    )
    packageId_ext: Optional[Element] = Field(
        description="Placeholder element for packageId extensions",
        default=None,
        alias="_packageId",
    )
    version: Optional[String] = Field(
        description="Version of the IG",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )

    @field_validator(
        *(
            "version",
            "packageId",
            "uri",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImplementationGuideGlobal(BackboneElement):
    """
    A set of profiles that all resources covered by this implementation guide must conform to.
    """

    type: Optional[Code] = Field(
        description="Type this profile applies to",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    profile: Optional[Canonical] = Field(
        description="Profile that all resources must conform to",
        default=None,
    )
    profile_ext: Optional[Element] = Field(
        description="Placeholder element for profile extensions",
        default=None,
        alias="_profile",
    )

    @field_validator(
        *(
            "profile",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImplementationGuideDefinitionGrouping(BackboneElement):
    """
    A logical group of resources. Logical groups can be used when building pages.
    """

    name: Optional[String] = Field(
        description="Descriptive name for the package",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    description: Optional[String] = Field(
        description="Human readable text describing the package",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )

    @field_validator(
        *(
            "description",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImplementationGuideDefinitionResource(BackboneElement):
    """
    A resource that is part of the implementation guide. Conformance resources (value set, structure definition, capability statements etc.) are obvious candidates for inclusion, but any kind of resource can be included as an example resource.
    """

    reference: Optional[Reference] = Field(
        description="Location of the resource",
        default=None,
    )
    fhirVersion: Optional[List[Code]] = Field(
        description="Versions this applies to (if different to IG)",
        default=None,
    )
    fhirVersion_ext: Optional[Element] = Field(
        description="Placeholder element for fhirVersion extensions",
        default=None,
        alias="_fhirVersion",
    )
    name: Optional[String] = Field(
        description="Human Name for the resource",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    description: Optional[String] = Field(
        description="Reason why included in guide",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    exampleBoolean: Optional[Boolean] = Field(
        description="Is an example/What is this an example of?",
        default=None,
    )
    exampleBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for exampleBoolean extensions",
        default=None,
        alias="_exampleBoolean",
    )
    exampleCanonical: Optional[Canonical] = Field(
        description="Is an example/What is this an example of?",
        default=None,
    )
    exampleCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for exampleCanonical extensions",
        default=None,
        alias="_exampleCanonical",
    )
    groupingId: Optional[Id] = Field(
        description="Grouping this is part of",
        default=None,
    )
    groupingId_ext: Optional[Element] = Field(
        description="Placeholder element for groupingId extensions",
        default=None,
        alias="_groupingId",
    )

    @property
    def example(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="example",
        )

    @field_validator(
        *(
            "groupingId",
            "description",
            "name",
            "fhirVersion",
            "reference",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def example_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Boolean, Canonical],
            field_name_base="example",
            required=False,
        )


class ImplementationGuideDefinitionPage(BackboneElement):
    """
    A page / section in the implementation guide. The root page is the implementation guide home page.
    """

    nameUrl: Optional[Url] = Field(
        description="Where to find that page",
        default=None,
    )
    nameUrl_ext: Optional[Element] = Field(
        description="Placeholder element for nameUrl extensions",
        default=None,
        alias="_nameUrl",
    )
    nameReference: Optional[Reference] = Field(
        description="Where to find that page",
        default=None,
    )
    title: Optional[String] = Field(
        description="Short title shown for navigational assistance",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    generation: Optional[Code] = Field(
        description="html | markdown | xml | generated",
        default=None,
    )
    generation_ext: Optional[Element] = Field(
        description="Placeholder element for generation extensions",
        default=None,
        alias="_generation",
    )
    page: Optional[List["ImplementationGuideDefinitionPage"]] = Field(
        description="Nested Pages / Sections",
        default=None,
    )

    @property
    def name(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="name",
        )

    @field_validator(
        *(
            "page",
            "generation",
            "title",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def name_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Url, Reference],
            field_name_base="name",
            required=True,
        )


class ImplementationGuideDefinitionParameter(BackboneElement):
    """
    Defines how IG is built by tools.
    """

    code: Optional[Code] = Field(
        description="apply | path-resource | path-pages | path-tx-cache | expansion-parameter | rule-broken-links | generate-xml | generate-json | generate-turtle | html-template",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    value: Optional[String] = Field(
        description="Value for named type",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImplementationGuideDefinitionTemplate(BackboneElement):
    """
    A template for building resources.
    """

    code: Optional[Code] = Field(
        description="Type of template specified",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    source: Optional[String] = Field(
        description="The source location for the template",
        default=None,
    )
    source_ext: Optional[Element] = Field(
        description="Placeholder element for source extensions",
        default=None,
        alias="_source",
    )
    scope: Optional[String] = Field(
        description="The scope in which the template applies",
        default=None,
    )
    scope_ext: Optional[Element] = Field(
        description="Placeholder element for scope extensions",
        default=None,
        alias="_scope",
    )

    @field_validator(
        *(
            "scope",
            "source",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImplementationGuideDefinition(BackboneElement):
    """
    The information needed by an IG publisher tool to publish the whole implementation guide.
    """

    grouping: Optional[List[ImplementationGuideDefinitionGrouping]] = Field(
        description="Grouping used to present related resources in the IG",
        default=None,
    )
    resource: Optional[List[ImplementationGuideDefinitionResource]] = Field(
        description="Resource in the implementation guide",
        default=None,
    )
    page: Optional[ImplementationGuideDefinitionPage] = Field(
        description="Page/Section in the Guide",
        default=None,
    )
    parameter: Optional[List[ImplementationGuideDefinitionParameter]] = Field(
        description="Defines how IG is built by tools",
        default=None,
    )
    template: Optional[List[ImplementationGuideDefinitionTemplate]] = Field(
        description="A template for building resources",
        default=None,
    )

    @field_validator(
        *(
            "template",
            "parameter",
            "page",
            "resource",
            "grouping",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )


class ImplementationGuideManifestResource(BackboneElement):
    """
    A resource that is part of the implementation guide. Conformance resources (value set, structure definition, capability statements etc.) are obvious candidates for inclusion, but any kind of resource can be included as an example resource.
    """

    reference: Optional[Reference] = Field(
        description="Location of the resource",
        default=None,
    )
    exampleBoolean: Optional[Boolean] = Field(
        description="Is an example/What is this an example of?",
        default=None,
    )
    exampleBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for exampleBoolean extensions",
        default=None,
        alias="_exampleBoolean",
    )
    exampleCanonical: Optional[Canonical] = Field(
        description="Is an example/What is this an example of?",
        default=None,
    )
    exampleCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for exampleCanonical extensions",
        default=None,
        alias="_exampleCanonical",
    )
    relativePath: Optional[Url] = Field(
        description="Relative path for page in IG",
        default=None,
    )
    relativePath_ext: Optional[Element] = Field(
        description="Placeholder element for relativePath extensions",
        default=None,
        alias="_relativePath",
    )

    @property
    def example(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="example",
        )

    @field_validator(
        *(
            "relativePath",
            "reference",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def example_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Boolean, Canonical],
            field_name_base="example",
            required=False,
        )


class ImplementationGuideManifestPage(BackboneElement):
    """
    Information about a page within the IG.
    """

    name: Optional[String] = Field(
        description="HTML page name",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Title of the page, for references",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    anchor: Optional[List[String]] = Field(
        description="Anchor available on the page",
        default=None,
    )
    anchor_ext: Optional[Element] = Field(
        description="Placeholder element for anchor extensions",
        default=None,
        alias="_anchor",
    )

    @field_validator(
        *(
            "anchor",
            "title",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImplementationGuideManifest(BackboneElement):
    """
    Information about an assembled implementation guide, created by the publication tooling.
    """

    rendering: Optional[Url] = Field(
        description="Location of rendered implementation guide",
        default=None,
    )
    rendering_ext: Optional[Element] = Field(
        description="Placeholder element for rendering extensions",
        default=None,
        alias="_rendering",
    )
    resource: Optional[List[ImplementationGuideManifestResource]] = Field(
        description="Resource in the implementation guide",
        default=None,
    )
    page: Optional[List[ImplementationGuideManifestPage]] = Field(
        description="HTML page within the parent IG",
        default=None,
    )
    image: Optional[List[String]] = Field(
        description="Image within the IG",
        default=None,
    )
    image_ext: Optional[Element] = Field(
        description="Placeholder element for image extensions",
        default=None,
        alias="_image",
    )
    other: Optional[List[String]] = Field(
        description="Additional linkable file in IG",
        default=None,
    )
    other_ext: Optional[Element] = Field(
        description="Placeholder element for other extensions",
        default=None,
        alias="_other",
    )

    @field_validator(
        *(
            "other",
            "image",
            "page",
            "resource",
            "rendering",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImplementationGuide(DomainResource):
    """
    A set of rules of how a particular interoperability or standards problem is solved - typically through the use of FHIR resources. This resource is used to gather all the parts of an implementation guide into a logical whole and to publish a computable definition of all the parts.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/ImplementationGuide"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this implementation guide, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    version: Optional[String] = Field(
        description="Business version of the implementation guide",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    name: Optional[String] = Field(
        description="Name for this implementation guide (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this implementation guide (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the implementation guide",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for implementation guide (if applicable)",
        default=None,
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    packageId: Optional[Id] = Field(
        description="NPM Package name for IG",
        default=None,
    )
    packageId_ext: Optional[Element] = Field(
        description="Placeholder element for packageId extensions",
        default=None,
        alias="_packageId",
    )
    license: Optional[Code] = Field(
        description="SPDX license code for this IG (or not-open-source)",
        default=None,
    )
    license_ext: Optional[Element] = Field(
        description="Placeholder element for license extensions",
        default=None,
        alias="_license",
    )
    fhirVersion: Optional[List[Code]] = Field(
        description="FHIR Version(s) this Implementation Guide targets",
        default=None,
    )
    fhirVersion_ext: Optional[Element] = Field(
        description="Placeholder element for fhirVersion extensions",
        default=None,
        alias="_fhirVersion",
    )
    dependsOn: Optional[List[ImplementationGuideDependsOn]] = Field(
        description="Another Implementation guide this depends on",
        default=None,
    )
    global_: Optional[List[ImplementationGuideGlobal]] = Field(
        description="Profiles that apply globally",
        default=None,
    )
    definition: Optional[ImplementationGuideDefinition] = Field(
        description="Information needed to build the IG",
        default=None,
    )
    manifest: Optional[ImplementationGuideManifest] = Field(
        description="Information about an assembled IG",
        default=None,
    )
    resourceType: Literal["ImplementationGuide"] = Field(
        description=None,
        default="ImplementationGuide",
    )

    @field_validator(
        *(
            "manifest",
            "definition",
            "global_",
            "dependsOn",
            "fhirVersion",
            "license",
            "packageId",
            "copyright",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "experimental",
            "status",
            "title",
            "name",
            "version",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("definition",), mode="after", check_fields=None)
    @classmethod
    def FHIR_ig_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="resource.groupingId.all(%context.grouping.id contains $this)",
            human="If a resource has a groupingId, it must refer to a grouping defined in the Implementation Guide",
            key="ig-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_ig_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.exists() implies name.matches('[A-Z]([A-Za-z0-9_]){0,254}')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="ig-0",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_ig_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="definition.resource.fhirVersion.all(%context.fhirVersion contains $this)",
            human="If a resource has a fhirVersion, it must be oe of the versions defined for the Implementation Guide",
            key="ig-2",
            severity="error",
        )
