# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    UnsignedInt,
    Id,
)

from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Coding,
    Reference,
    CodeableConcept,
    Annotation,
    BackboneElement,
    DomainResource,
)


class ImagingStudySeriesPerformer(BackboneElement):
    """
    Indicates who or what performed the series and how they were involved.
    """

    function: Optional[CodeableConcept] = Field(
        description="Type of performance",
        default=None,
    )
    actor: Optional[Reference] = Field(
        description="Who performed the series",
        default=None,
    )

    @field_validator(
        *(
            "actor",
            "function",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImagingStudySeriesInstance(BackboneElement):
    """
    A single SOP instance within the series, e.g. an image, or presentation state.
    """

    uid: Optional[Id] = Field(
        description="DICOM SOP Instance UID",
        default=None,
    )
    uid_ext: Optional[Element] = Field(
        description="Placeholder element for uid extensions",
        default=None,
        alias="_uid",
    )
    sopClass: Optional[Coding] = Field(
        description="DICOM class type",
        default=None,
    )
    number: Optional[UnsignedInt] = Field(
        description="The number of this instance in the series",
        default=None,
    )
    number_ext: Optional[Element] = Field(
        description="Placeholder element for number extensions",
        default=None,
        alias="_number",
    )
    title: Optional[String] = Field(
        description="Description of instance",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )

    @field_validator(
        *(
            "title",
            "number",
            "sopClass",
            "uid",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImagingStudySeries(BackboneElement):
    """
    Each study has one or more series of images or other content.
    """

    uid: Optional[Id] = Field(
        description="DICOM Series Instance UID for the series",
        default=None,
    )
    uid_ext: Optional[Element] = Field(
        description="Placeholder element for uid extensions",
        default=None,
        alias="_uid",
    )
    number: Optional[UnsignedInt] = Field(
        description="Numeric identifier of this series",
        default=None,
    )
    number_ext: Optional[Element] = Field(
        description="Placeholder element for number extensions",
        default=None,
        alias="_number",
    )
    modality: Optional[Coding] = Field(
        description="The modality of the instances in the series",
        default=None,
    )
    description: Optional[String] = Field(
        description="A short human readable summary of the series",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    numberOfInstances: Optional[UnsignedInt] = Field(
        description="Number of Series Related Instances",
        default=None,
    )
    numberOfInstances_ext: Optional[Element] = Field(
        description="Placeholder element for numberOfInstances extensions",
        default=None,
        alias="_numberOfInstances",
    )
    endpoint: Optional[List[Reference]] = Field(
        description="Series access endpoint",
        default=None,
    )
    bodySite: Optional[Coding] = Field(
        description="Body part examined",
        default=None,
    )
    laterality: Optional[Coding] = Field(
        description="Body part laterality",
        default=None,
    )
    specimen: Optional[List[Reference]] = Field(
        description="Specimen imaged",
        default=None,
    )
    started: Optional[DateTime] = Field(
        description="When the series started",
        default=None,
    )
    started_ext: Optional[Element] = Field(
        description="Placeholder element for started extensions",
        default=None,
        alias="_started",
    )
    performer: Optional[List[ImagingStudySeriesPerformer]] = Field(
        description="Who performed the series",
        default=None,
    )
    instance: Optional[List[ImagingStudySeriesInstance]] = Field(
        description="A single SOP instance from the series",
        default=None,
    )

    @field_validator(
        *(
            "instance",
            "performer",
            "started",
            "specimen",
            "laterality",
            "bodySite",
            "endpoint",
            "numberOfInstances",
            "description",
            "modality",
            "number",
            "uid",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )


class ImagingStudy(DomainResource):
    """
    Representation of the content produced in a DICOM imaging study. A study comprises a set of series, each of which includes a set of Service-Object Pair Instances (SOP Instances - images or other data) acquired or produced in a common context.  A series is of only one modality (e.g. X-ray, CT, MR, ultrasound), but a study may have multiple series of different modalities.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/ImagingStudy"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Identifiers for the whole study",
        default=None,
    )
    status: Optional[Code] = Field(
        description="registered | available | cancelled | entered-in-error | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    modality: Optional[List[Coding]] = Field(
        description="All series modality if actual acquisition modalities",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="Who or what is the subject of the study",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Encounter with which this imaging study is associated",
        default=None,
    )
    started: Optional[DateTime] = Field(
        description="When the study was started",
        default=None,
    )
    started_ext: Optional[Element] = Field(
        description="Placeholder element for started extensions",
        default=None,
        alias="_started",
    )
    basedOn: Optional[List[Reference]] = Field(
        description="Request fulfilled",
        default=None,
    )
    referrer: Optional[Reference] = Field(
        description="Referring physician",
        default=None,
    )
    interpreter: Optional[List[Reference]] = Field(
        description="Who interpreted images",
        default=None,
    )
    endpoint: Optional[List[Reference]] = Field(
        description="Study access endpoint",
        default=None,
    )
    numberOfSeries: Optional[UnsignedInt] = Field(
        description="Number of Study Related Series",
        default=None,
    )
    numberOfSeries_ext: Optional[Element] = Field(
        description="Placeholder element for numberOfSeries extensions",
        default=None,
        alias="_numberOfSeries",
    )
    numberOfInstances: Optional[UnsignedInt] = Field(
        description="Number of Study Related Instances",
        default=None,
    )
    numberOfInstances_ext: Optional[Element] = Field(
        description="Placeholder element for numberOfInstances extensions",
        default=None,
        alias="_numberOfInstances",
    )
    procedureReference: Optional[Reference] = Field(
        description="The performed Procedure reference",
        default=None,
    )
    procedureCode: Optional[List[CodeableConcept]] = Field(
        description="The performed procedure code",
        default=None,
    )
    location: Optional[Reference] = Field(
        description="Where ImagingStudy occurred",
        default=None,
    )
    reasonCode: Optional[List[CodeableConcept]] = Field(
        description="Why the study was requested",
        default=None,
    )
    reasonReference: Optional[List[Reference]] = Field(
        description="Why was study performed",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="User-defined comments",
        default=None,
    )
    description: Optional[String] = Field(
        description="Institution-generated description",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    series: Optional[List[ImagingStudySeries]] = Field(
        description="Each study has one or more series of instances",
        default=None,
    )
    resourceType: Literal["ImagingStudy"] = Field(
        description=None,
        default="ImagingStudy",
    )

    @field_validator(
        *(
            "series",
            "description",
            "note",
            "reasonReference",
            "reasonCode",
            "location",
            "procedureCode",
            "procedureReference",
            "numberOfInstances",
            "numberOfSeries",
            "endpoint",
            "interpreter",
            "referrer",
            "basedOn",
            "started",
            "encounter",
            "subject",
            "modality",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
