# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    Markdown,
    Time,
)

from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    CodeableConcept,
    Attachment,
    ContactPoint,
    BackboneElement,
    Period,
    DomainResource,
)


class HealthcareServiceEligibility(BackboneElement):
    """
    Does this service have specific eligibility requirements that need to be met in order to use the service?
    """

    code: Optional[CodeableConcept] = Field(
        description="Coded value for the eligibility",
        default=None,
    )
    comment: Optional[Markdown] = Field(
        description="Describes the eligibility conditions for the service",
        default=None,
    )
    comment_ext: Optional[Element] = Field(
        description="Placeholder element for comment extensions",
        default=None,
        alias="_comment",
    )

    @field_validator(
        *(
            "comment",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class HealthcareServiceAvailableTime(BackboneElement):
    """
    A collection of times that the Service Site is available.
    """

    daysOfWeek: Optional[List[Code]] = Field(
        description="mon | tue | wed | thu | fri | sat | sun",
        default=None,
    )
    daysOfWeek_ext: Optional[Element] = Field(
        description="Placeholder element for daysOfWeek extensions",
        default=None,
        alias="_daysOfWeek",
    )
    allDay: Optional[Boolean] = Field(
        description="Always available? e.g. 24 hour service",
        default=None,
    )
    allDay_ext: Optional[Element] = Field(
        description="Placeholder element for allDay extensions",
        default=None,
        alias="_allDay",
    )
    availableStartTime: Optional[Time] = Field(
        description="Opening time of day (ignored if allDay = true)",
        default=None,
    )
    availableStartTime_ext: Optional[Element] = Field(
        description="Placeholder element for availableStartTime extensions",
        default=None,
        alias="_availableStartTime",
    )
    availableEndTime: Optional[Time] = Field(
        description="Closing time of day (ignored if allDay = true)",
        default=None,
    )
    availableEndTime_ext: Optional[Element] = Field(
        description="Placeholder element for availableEndTime extensions",
        default=None,
        alias="_availableEndTime",
    )

    @field_validator(
        *(
            "availableEndTime",
            "availableStartTime",
            "allDay",
            "daysOfWeek",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class HealthcareServiceNotAvailable(BackboneElement):
    """
    The HealthcareService is not available during this period of time due to the provided reason.
    """

    description: Optional[String] = Field(
        description="Reason presented to the user explaining why time not available",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    during: Optional[Period] = Field(
        description="Service not available from this date",
        default=None,
    )

    @field_validator(
        *(
            "during",
            "description",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class HealthcareService(DomainResource):
    """
    The details of a healthcare service available at a location.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/HealthcareService"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="External identifiers for this item",
        default=None,
    )
    active: Optional[Boolean] = Field(
        description="Whether this HealthcareService record is in active use",
        default=None,
    )
    active_ext: Optional[Element] = Field(
        description="Placeholder element for active extensions",
        default=None,
        alias="_active",
    )
    providedBy: Optional[Reference] = Field(
        description="Organization that provides this service",
        default=None,
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Broad category of service being performed or delivered",
        default=None,
    )
    type: Optional[List[CodeableConcept]] = Field(
        description="Type of service that may be delivered or performed",
        default=None,
    )
    specialty: Optional[List[CodeableConcept]] = Field(
        description="Specialties handled by the HealthcareService",
        default=None,
    )
    location: Optional[List[Reference]] = Field(
        description="Location(s) where service may be provided",
        default=None,
    )
    name: Optional[String] = Field(
        description="Description of service as presented to a consumer while searching",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    comment: Optional[String] = Field(
        description="Additional description and/or any specific issues not covered elsewhere",
        default=None,
    )
    comment_ext: Optional[Element] = Field(
        description="Placeholder element for comment extensions",
        default=None,
        alias="_comment",
    )
    extraDetails: Optional[Markdown] = Field(
        description="Extra details about the service that can\u0027t be placed in the other fields",
        default=None,
    )
    extraDetails_ext: Optional[Element] = Field(
        description="Placeholder element for extraDetails extensions",
        default=None,
        alias="_extraDetails",
    )
    photo: Optional[Attachment] = Field(
        description="Facilitates quick identification of the service",
        default=None,
    )
    telecom: Optional[List[ContactPoint]] = Field(
        description="Contacts related to the healthcare service",
        default=None,
    )
    coverageArea: Optional[List[Reference]] = Field(
        description="Location(s) service is intended for/available to",
        default=None,
    )
    serviceProvisionCode: Optional[List[CodeableConcept]] = Field(
        description="Conditions under which service is available/offered",
        default=None,
    )
    eligibility: Optional[List[HealthcareServiceEligibility]] = Field(
        description="Specific eligibility requirements required to use the service",
        default=None,
    )
    program: Optional[List[CodeableConcept]] = Field(
        description="Programs that this service is applicable to",
        default=None,
    )
    characteristic: Optional[List[CodeableConcept]] = Field(
        description="Collection of characteristics (attributes)",
        default=None,
    )
    communication: Optional[List[CodeableConcept]] = Field(
        description="The language that this service is offered in",
        default=None,
    )
    referralMethod: Optional[List[CodeableConcept]] = Field(
        description="Ways that the service accepts referrals",
        default=None,
    )
    appointmentRequired: Optional[Boolean] = Field(
        description="If an appointment is required for access to this service",
        default=None,
    )
    appointmentRequired_ext: Optional[Element] = Field(
        description="Placeholder element for appointmentRequired extensions",
        default=None,
        alias="_appointmentRequired",
    )
    availableTime: Optional[List[HealthcareServiceAvailableTime]] = Field(
        description="Times the Service Site is available",
        default=None,
    )
    notAvailable: Optional[List[HealthcareServiceNotAvailable]] = Field(
        description="Not available during this time due to provided reason",
        default=None,
    )
    availabilityExceptions: Optional[String] = Field(
        description="Description of availability exceptions",
        default=None,
    )
    availabilityExceptions_ext: Optional[Element] = Field(
        description="Placeholder element for availabilityExceptions extensions",
        default=None,
        alias="_availabilityExceptions",
    )
    endpoint: Optional[List[Reference]] = Field(
        description="Technical endpoints providing access to electronic services operated for the healthcare service",
        default=None,
    )
    resourceType: Literal["HealthcareService"] = Field(
        description=None,
        default="HealthcareService",
    )

    @field_validator(
        *(
            "endpoint",
            "availabilityExceptions",
            "notAvailable",
            "availableTime",
            "appointmentRequired",
            "referralMethod",
            "communication",
            "characteristic",
            "program",
            "eligibility",
            "serviceProvisionCode",
            "coverageArea",
            "telecom",
            "photo",
            "extraDetails",
            "comment",
            "name",
            "location",
            "specialty",
            "type",
            "category",
            "providedBy",
            "active",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
