# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    DateTime,
    Markdown,
    Canonical,
)

from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    ContactDetail,
    UsageContext,
    CodeableConcept,
    BackboneElement,
    DomainResource,
)


class ExampleScenarioActor(BackboneElement):
    """
    Actor participating in the resource.
    """

    actorId: Optional[String] = Field(
        description="ID or acronym of the actor",
        default=None,
    )
    actorId_ext: Optional[Element] = Field(
        description="Placeholder element for actorId extensions",
        default=None,
        alias="_actorId",
    )
    type: Optional[Code] = Field(
        description="person | entity",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    name: Optional[String] = Field(
        description="The name of the actor as shown in the page",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    description: Optional[Markdown] = Field(
        description="The description of the actor",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )

    @field_validator(
        *(
            "description",
            "name",
            "type",
            "actorId",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExampleScenarioInstanceVersion(BackboneElement):
    """
    A specific version of the resource.
    """

    versionId: Optional[String] = Field(
        description="The identifier of a specific version of a resource",
        default=None,
    )
    versionId_ext: Optional[Element] = Field(
        description="Placeholder element for versionId extensions",
        default=None,
        alias="_versionId",
    )
    description: Optional[Markdown] = Field(
        description="The description of the resource version",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )

    @field_validator(
        *(
            "description",
            "versionId",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExampleScenarioInstanceContainedInstance(BackboneElement):
    """
    Resources contained in the instance (e.g. the observations contained in a bundle).
    """

    resourceId: Optional[String] = Field(
        description="Each resource contained in the instance",
        default=None,
    )
    resourceId_ext: Optional[Element] = Field(
        description="Placeholder element for resourceId extensions",
        default=None,
        alias="_resourceId",
    )
    versionId: Optional[String] = Field(
        description="A specific version of a resource contained in the instance",
        default=None,
    )
    versionId_ext: Optional[Element] = Field(
        description="Placeholder element for versionId extensions",
        default=None,
        alias="_versionId",
    )

    @field_validator(
        *(
            "versionId",
            "resourceId",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExampleScenarioInstance(BackboneElement):
    """
    Each resource and each version that is present in the workflow.
    """

    resourceId: Optional[String] = Field(
        description="The id of the resource for referencing",
        default=None,
    )
    resourceId_ext: Optional[Element] = Field(
        description="Placeholder element for resourceId extensions",
        default=None,
        alias="_resourceId",
    )
    resourceType: Optional[Code] = Field(
        description="The type of the resource",
        default=None,
    )
    resourceType_ext: Optional[Element] = Field(
        description="Placeholder element for resourceType extensions",
        default=None,
        alias="_resourceType",
    )
    name: Optional[String] = Field(
        description="A short name for the resource instance",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    description: Optional[Markdown] = Field(
        description="Human-friendly description of the resource instance",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    version: Optional[List[ExampleScenarioInstanceVersion]] = Field(
        description="A specific version of the resource",
        default=None,
    )
    containedInstance: Optional[List[ExampleScenarioInstanceContainedInstance]] = Field(
        description="Resources contained in the instance",
        default=None,
    )

    @field_validator(
        *(
            "containedInstance",
            "version",
            "description",
            "name",
            "resourceType",
            "resourceId",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )


class ExampleScenarioProcessStepOperationRequest(BackboneElement):
    """
    Each resource instance used by the initiator.
    """

    resourceId: Optional[String] = Field(
        description="Each resource contained in the instance",
        default=None,
    )
    resourceId_ext: Optional[Element] = Field(
        description="Placeholder element for resourceId extensions",
        default=None,
        alias="_resourceId",
    )
    versionId: Optional[String] = Field(
        description="A specific version of a resource contained in the instance",
        default=None,
    )
    versionId_ext: Optional[Element] = Field(
        description="Placeholder element for versionId extensions",
        default=None,
        alias="_versionId",
    )

    @field_validator(
        *(
            "versionId",
            "resourceId",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExampleScenarioProcessStepOperation(BackboneElement):
    """
    Each interaction or action.
    """

    number: Optional[String] = Field(
        description="The sequential number of the interaction",
        default=None,
    )
    number_ext: Optional[Element] = Field(
        description="Placeholder element for number extensions",
        default=None,
        alias="_number",
    )
    type: Optional[String] = Field(
        description="The type of operation - CRUD",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    name: Optional[String] = Field(
        description="The human-friendly name of the interaction",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    initiator: Optional[String] = Field(
        description="Who starts the transaction",
        default=None,
    )
    initiator_ext: Optional[Element] = Field(
        description="Placeholder element for initiator extensions",
        default=None,
        alias="_initiator",
    )
    receiver: Optional[String] = Field(
        description="Who receives the transaction",
        default=None,
    )
    receiver_ext: Optional[Element] = Field(
        description="Placeholder element for receiver extensions",
        default=None,
        alias="_receiver",
    )
    description: Optional[Markdown] = Field(
        description="A comment to be inserted in the diagram",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    initiatorActive: Optional[Boolean] = Field(
        description="Whether the initiator is deactivated right after the transaction",
        default=None,
    )
    initiatorActive_ext: Optional[Element] = Field(
        description="Placeholder element for initiatorActive extensions",
        default=None,
        alias="_initiatorActive",
    )
    receiverActive: Optional[Boolean] = Field(
        description="Whether the receiver is deactivated right after the transaction",
        default=None,
    )
    receiverActive_ext: Optional[Element] = Field(
        description="Placeholder element for receiverActive extensions",
        default=None,
        alias="_receiverActive",
    )
    request: Optional[ExampleScenarioProcessStepOperationRequest] = Field(
        description="Each resource instance used by the initiator",
        default=None,
    )
    response: Optional[ExampleScenarioInstanceContainedInstance] = Field(
        description="Each resource instance used by the responder",
        default=None,
    )

    @field_validator(
        *(
            "response",
            "request",
            "receiverActive",
            "initiatorActive",
            "description",
            "receiver",
            "initiator",
            "name",
            "type",
            "number",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExampleScenarioProcessStepAlternative(BackboneElement):
    """
    Indicates an alternative step that can be taken instead of the operations on the base step in exceptional/atypical circumstances.
    """

    title: Optional[String] = Field(
        description="Label for alternative",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    description: Optional[Markdown] = Field(
        description="A human-readable description of each option",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    step: Optional[List["ExampleScenarioProcessStep"]] = Field(
        description="What happens in each alternative option",
        default=None,
    )

    @field_validator(
        *(
            "step",
            "description",
            "title",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ExampleScenarioProcessStep(BackboneElement):
    """
    Each step of the process.
    """

    process: Optional[List["ExampleScenarioProcess"]] = Field(
        description="Nested process",
        default=None,
    )
    pause: Optional[Boolean] = Field(
        description="If there is a pause in the flow",
        default=None,
    )
    pause_ext: Optional[Element] = Field(
        description="Placeholder element for pause extensions",
        default=None,
        alias="_pause",
    )
    operation: Optional[ExampleScenarioProcessStepOperation] = Field(
        description="Each interaction or action",
        default=None,
    )
    alternative: Optional[List[ExampleScenarioProcessStepAlternative]] = Field(
        description="Alternate non-typical step action",
        default=None,
    )

    @field_validator(
        *(
            "alternative",
            "operation",
            "pause",
            "process",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )


class ExampleScenarioProcess(BackboneElement):
    """
    Each major process - a group of operations.
    """

    title: Optional[String] = Field(
        description="The diagram title of the group of operations",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    description: Optional[Markdown] = Field(
        description="A longer description of the group of operations",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    preConditions: Optional[Markdown] = Field(
        description="Description of initial status before the process starts",
        default=None,
    )
    preConditions_ext: Optional[Element] = Field(
        description="Placeholder element for preConditions extensions",
        default=None,
        alias="_preConditions",
    )
    postConditions: Optional[Markdown] = Field(
        description="Description of final status after the process ends",
        default=None,
    )
    postConditions_ext: Optional[Element] = Field(
        description="Placeholder element for postConditions extensions",
        default=None,
        alias="_postConditions",
    )
    step: Optional[List[ExampleScenarioProcessStep]] = Field(
        description="Each step of the process",
        default=None,
    )

    @field_validator(
        *(
            "step",
            "postConditions",
            "preConditions",
            "description",
            "title",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )


class ExampleScenario(DomainResource):
    """
    Example of workflow instance.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/ExampleScenario"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this example scenario, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifier for the example scenario",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the example scenario",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    name: Optional[String] = Field(
        description="Name for this example scenario (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for example scenario (if applicable)",
        default=None,
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    purpose: Optional[Markdown] = Field(
        description="The purpose of the example, e.g. to illustrate a scenario",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    actor: Optional[List[ExampleScenarioActor]] = Field(
        description="Actor participating in the resource",
        default=None,
    )
    instance: Optional[List[ExampleScenarioInstance]] = Field(
        description="Each resource and each version that is present in the workflow",
        default=None,
    )
    process: Optional[List[ExampleScenarioProcess]] = Field(
        description="Each major process - a group of operations",
        default=None,
    )
    workflow: Optional[List[Canonical]] = Field(
        description="Another nested workflow",
        default=None,
    )
    workflow_ext: Optional[Element] = Field(
        description="Placeholder element for workflow extensions",
        default=None,
        alias="_workflow",
    )
    resourceType: Literal["ExampleScenario"] = Field(
        description=None,
        default="ExampleScenario",
    )

    @field_validator(
        *(
            "workflow",
            "process",
            "instance",
            "actor",
            "purpose",
            "copyright",
            "jurisdiction",
            "useContext",
            "contact",
            "publisher",
            "date",
            "experimental",
            "status",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_esc_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.exists() implies name.matches('[A-Z]([A-Za-z0-9_]){0,254}')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="esc-0",
            severity="warning",
        )


ExampleScenarioProcessStepAlternative.model_rebuild()
ExampleScenarioProcessStep.model_rebuild()
ExampleScenarioProcess.model_rebuild()
ExampleScenario.model_rebuild()
