# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Date,
    DateTime,
    PositiveInt,
    Boolean,
)

from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    Period,
    BackboneElement,
    Quantity,
    Money,
    DomainResource,
)


class CoverageEligibilityRequestSupportingInfo(BackboneElement):
    """
    Additional information codes regarding exceptions, special considerations, the condition, situation, prior or concurrent issues.
    """

    sequence: Optional[PositiveInt] = Field(
        description="Information instance identifier",
        default=None,
    )
    sequence_ext: Optional[Element] = Field(
        description="Placeholder element for sequence extensions",
        default=None,
        alias="_sequence",
    )
    information: Optional[Reference] = Field(
        description="Data to be provided",
        default=None,
    )
    appliesToAll: Optional[Boolean] = Field(
        description="Applies to all items",
        default=None,
    )
    appliesToAll_ext: Optional[Element] = Field(
        description="Placeholder element for appliesToAll extensions",
        default=None,
        alias="_appliesToAll",
    )

    @field_validator(
        *(
            "appliesToAll",
            "information",
            "sequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CoverageEligibilityRequestInsurance(BackboneElement):
    """
    Financial instruments for reimbursement for the health care products and services.
    """

    focal: Optional[Boolean] = Field(
        description="Applicable coverage",
        default=None,
    )
    focal_ext: Optional[Element] = Field(
        description="Placeholder element for focal extensions",
        default=None,
        alias="_focal",
    )
    coverage: Optional[Reference] = Field(
        description="Insurance information",
        default=None,
    )
    businessArrangement: Optional[String] = Field(
        description="Additional provider contract number",
        default=None,
    )
    businessArrangement_ext: Optional[Element] = Field(
        description="Placeholder element for businessArrangement extensions",
        default=None,
        alias="_businessArrangement",
    )

    @field_validator(
        *(
            "businessArrangement",
            "coverage",
            "focal",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CoverageEligibilityRequestItemDiagnosis(BackboneElement):
    """
    Patient diagnosis for which care is sought.
    """

    diagnosisCodeableConcept: Optional[CodeableConcept] = Field(
        description="Nature of illness or problem",
        default=None,
    )
    diagnosisReference: Optional[Reference] = Field(
        description="Nature of illness or problem",
        default=None,
    )

    @property
    def diagnosis(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="diagnosis",
        )

    @model_validator(mode="after")
    def diagnosis_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Reference],
            field_name_base="diagnosis",
            required=False,
        )


class CoverageEligibilityRequestItem(BackboneElement):
    """
    Service categories or billable services for which benefit details and/or an authorization prior to service delivery may be required by the payor.
    """

    supportingInfoSequence: Optional[List[PositiveInt]] = Field(
        description="Applicable exception or supporting information",
        default=None,
    )
    supportingInfoSequence_ext: Optional[Element] = Field(
        description="Placeholder element for supportingInfoSequence extensions",
        default=None,
        alias="_supportingInfoSequence",
    )
    category: Optional[CodeableConcept] = Field(
        description="Benefit classification",
        default=None,
    )
    productOrService: Optional[CodeableConcept] = Field(
        description="Billing, service, product, or drug code",
        default=None,
    )
    modifier: Optional[List[CodeableConcept]] = Field(
        description="Product or service billing modifiers",
        default=None,
    )
    provider: Optional[Reference] = Field(
        description="Perfoming practitioner",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="Count of products or services",
        default=None,
    )
    unitPrice: Optional[Money] = Field(
        description="Fee, charge or cost per item",
        default=None,
    )
    facility: Optional[Reference] = Field(
        description="Servicing facility",
        default=None,
    )
    diagnosis: Optional[List[CoverageEligibilityRequestItemDiagnosis]] = Field(
        description="Applicable diagnosis",
        default=None,
    )
    detail: Optional[List[Reference]] = Field(
        description="Product or service details",
        default=None,
    )

    @field_validator(
        *(
            "detail",
            "diagnosis",
            "facility",
            "unitPrice",
            "quantity",
            "provider",
            "modifier",
            "productOrService",
            "category",
            "supportingInfoSequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CoverageEligibilityRequest(DomainResource):
    """
    The CoverageEligibilityRequest provides patient and insurance coverage information to an insurer for them to respond, in the form of an CoverageEligibilityResponse, with information regarding whether the stated coverage is valid and in-force and optionally to provide the insurance details of the policy.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=[
                "http://hl7.org/fhir/StructureDefinition/CoverageEligibilityRequest"
            ]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business Identifier for coverage eligiblity request",
        default=None,
    )
    status: Optional[Code] = Field(
        description="active | cancelled | draft | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    priority: Optional[CodeableConcept] = Field(
        description="Desired processing priority",
        default=None,
    )
    purpose: Optional[List[Code]] = Field(
        description="auth-requirements | benefits | discovery | validation",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    patient: Optional[Reference] = Field(
        description="Intended recipient of products and services",
        default=None,
    )
    servicedDate: Optional[Date] = Field(
        description="Estimated date or dates of service",
        default=None,
    )
    servicedDate_ext: Optional[Element] = Field(
        description="Placeholder element for servicedDate extensions",
        default=None,
        alias="_servicedDate",
    )
    servicedPeriod: Optional[Period] = Field(
        description="Estimated date or dates of service",
        default=None,
    )
    created: Optional[DateTime] = Field(
        description="Creation date",
        default=None,
    )
    created_ext: Optional[Element] = Field(
        description="Placeholder element for created extensions",
        default=None,
        alias="_created",
    )
    enterer: Optional[Reference] = Field(
        description="Author",
        default=None,
    )
    provider: Optional[Reference] = Field(
        description="Party responsible for the request",
        default=None,
    )
    insurer: Optional[Reference] = Field(
        description="Coverage issuer",
        default=None,
    )
    facility: Optional[Reference] = Field(
        description="Servicing facility",
        default=None,
    )
    supportingInfo: Optional[List[CoverageEligibilityRequestSupportingInfo]] = Field(
        description="Supporting information",
        default=None,
    )
    insurance: Optional[List[CoverageEligibilityRequestInsurance]] = Field(
        description="Patient insurance information",
        default=None,
    )
    item: Optional[List[CoverageEligibilityRequestItem]] = Field(
        description="Item to be evaluated for eligibiity",
        default=None,
    )
    resourceType: Literal["CoverageEligibilityRequest"] = Field(
        description=None,
        default="CoverageEligibilityRequest",
    )

    @property
    def serviced(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="serviced",
        )

    @field_validator(
        *(
            "item",
            "insurance",
            "supportingInfo",
            "facility",
            "insurer",
            "provider",
            "enterer",
            "created",
            "patient",
            "purpose",
            "priority",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def serviced_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Date, Period],
            field_name_base="serviced",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
