# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    Boolean,
)

from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    Attachment,
    BackboneElement,
    Period,
    Coding,
    DomainResource,
)


class ConsentPolicy(BackboneElement):
    """
    The references to the policies that are included in this consent scope. Policies may be organizational, but are often defined jurisdictionally, or in law.
    """

    authority: Optional[Uri] = Field(
        description="Enforcement source for policy",
        default=None,
    )
    authority_ext: Optional[Element] = Field(
        description="Placeholder element for authority extensions",
        default=None,
        alias="_authority",
    )
    uri: Optional[Uri] = Field(
        description="Specific policy covered by this consent",
        default=None,
    )
    uri_ext: Optional[Element] = Field(
        description="Placeholder element for uri extensions",
        default=None,
        alias="_uri",
    )

    @field_validator(
        *(
            "uri",
            "authority",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ConsentVerification(BackboneElement):
    """
    Whether a treatment instruction (e.g. artificial respiration yes or no) was verified with the patient, his/her family or another authorized person.
    """

    verified: Optional[Boolean] = Field(
        description="Has been verified",
        default=None,
    )
    verified_ext: Optional[Element] = Field(
        description="Placeholder element for verified extensions",
        default=None,
        alias="_verified",
    )
    verifiedWith: Optional[Reference] = Field(
        description="Person who verified",
        default=None,
    )
    verificationDate: Optional[DateTime] = Field(
        description="When consent verified",
        default=None,
    )
    verificationDate_ext: Optional[Element] = Field(
        description="Placeholder element for verificationDate extensions",
        default=None,
        alias="_verificationDate",
    )

    @field_validator(
        *(
            "verificationDate",
            "verifiedWith",
            "verified",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ConsentProvisionActor(BackboneElement):
    """
    Who or what is controlled by this rule. Use group to identify a set of actors by some property they share (e.g. 'admitting officers').
    """

    role: Optional[CodeableConcept] = Field(
        description="How the actor is involved",
        default=None,
    )
    reference: Optional[Reference] = Field(
        description="Resource for the actor (or group, by role)",
        default=None,
    )

    @field_validator(
        *(
            "reference",
            "role",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ConsentProvisionData(BackboneElement):
    """
    The resources controlled by this rule if specific resources are referenced.
    """

    meaning: Optional[Code] = Field(
        description="instance | related | dependents | authoredby",
        default=None,
    )
    meaning_ext: Optional[Element] = Field(
        description="Placeholder element for meaning extensions",
        default=None,
        alias="_meaning",
    )
    reference: Optional[Reference] = Field(
        description="The actual data reference",
        default=None,
    )

    @field_validator(
        *(
            "reference",
            "meaning",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ConsentProvision(BackboneElement):
    """
    An exception to the base policy of this consent. An exception can be an addition or removal of access permissions.
    """

    type: Optional[Code] = Field(
        description="deny | permit",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    period: Optional[Period] = Field(
        description="Timeframe for this rule",
        default=None,
    )
    actor: Optional[List[ConsentProvisionActor]] = Field(
        description="Who|what controlled by this rule (or group, by role)",
        default=None,
    )
    action: Optional[List[CodeableConcept]] = Field(
        description="Actions controlled by this rule",
        default=None,
    )
    securityLabel: Optional[List[Coding]] = Field(
        description="Security Labels that define affected resources",
        default=None,
    )
    purpose: Optional[List[Coding]] = Field(
        description="Context of activities covered by this rule",
        default=None,
    )
    class_: Optional[List[Coding]] = Field(
        description="e.g. Resource Type, Profile, CDA, etc.",
        default=None,
    )
    code: Optional[List[CodeableConcept]] = Field(
        description="e.g. LOINC or SNOMED CT code, etc. in the content",
        default=None,
    )
    dataPeriod: Optional[Period] = Field(
        description="Timeframe for data controlled by this rule",
        default=None,
    )
    data: Optional[List[ConsentProvisionData]] = Field(
        description="Data controlled by this rule",
        default=None,
    )
    provision: Optional[List["ConsentProvision"]] = Field(
        description="Nested Exception Rules",
        default=None,
    )

    @field_validator(
        *(
            "provision",
            "data",
            "dataPeriod",
            "code",
            "class_",
            "purpose",
            "securityLabel",
            "action",
            "actor",
            "period",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class Consent(DomainResource):
    """
    A record of a healthcare consumer’s  choices, which permits or denies identified recipient(s) or recipient role(s) to perform one or more actions within a given policy context, for specific purposes and periods of time.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Consent"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Identifier for this record (external references)",
        default=None,
    )
    status: Optional[Code] = Field(
        description="draft | proposed | active | rejected | inactive | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    scope: Optional[CodeableConcept] = Field(
        description="Which of the four areas this resource covers (extensible)",
        default=None,
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Classification of the consent statement - for indexing/retrieval",
        default=None,
    )
    patient: Optional[Reference] = Field(
        description="Who the consent applies to",
        default=None,
    )
    dateTime: Optional[DateTime] = Field(
        description="When this Consent was created or indexed",
        default=None,
    )
    dateTime_ext: Optional[Element] = Field(
        description="Placeholder element for dateTime extensions",
        default=None,
        alias="_dateTime",
    )
    performer: Optional[List[Reference]] = Field(
        description="Who is agreeing to the policy and rules",
        default=None,
    )
    organization: Optional[List[Reference]] = Field(
        description="Custodian of the consent",
        default=None,
    )
    sourceAttachment: Optional[Attachment] = Field(
        description="Source from which this consent is taken",
        default=None,
    )
    sourceReference: Optional[Reference] = Field(
        description="Source from which this consent is taken",
        default=None,
    )
    policy: Optional[List[ConsentPolicy]] = Field(
        description="Policies covered by this consent",
        default=None,
    )
    policyRule: Optional[CodeableConcept] = Field(
        description="Regulation that this consents to",
        default=None,
    )
    verification: Optional[List[ConsentVerification]] = Field(
        description="Consent Verified by patient or family",
        default=None,
    )
    provision: Optional[ConsentProvision] = Field(
        description="Constraints to the base Consent.policyRule",
        default=None,
    )
    resourceType: Literal["Consent"] = Field(
        description=None,
        default="Consent",
    )

    @property
    def source(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="source",
        )

    @field_validator(
        *(
            "provision",
            "verification",
            "policyRule",
            "policy",
            "organization",
            "performer",
            "dateTime",
            "patient",
            "category",
            "scope",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def source_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Attachment, Reference],
            field_name_base="source",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_ppc_1_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="policy.exists() or policyRule.exists()",
            human="Either a Policy or PolicyRule",
            key="ppc-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_ppc_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="patient.exists() or scope.coding.where(system='something' and code='patient-privacy').exists().not()",
            human="IF Scope=privacy, there must be a patient",
            key="ppc-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_ppc_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="patient.exists() or scope.coding.where(system='something' and code='research').exists().not()",
            human="IF Scope=research, there must be a patient",
            key="ppc-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_ppc_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="patient.exists() or scope.coding.where(system='something' and code='adr').exists().not()",
            human="IF Scope=adr, there must be a patient",
            key="ppc-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_ppc_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="patient.exists() or scope.coding.where(system='something' and code='treatment').exists().not()",
            human="IF Scope=treatment, there must be a patient",
            key="ppc-5",
            severity="error",
        )
