# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code, Markdown

from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    BackboneElement,
    CodeableReference,
    Range,
    DomainResource,
)


class ClinicalUseDefinitionContraindicationOtherTherapy(BackboneElement):
    """
    Information about the use of the medicinal product in relation to other therapies described as part of the contraindication.
    """

    relationshipType: Optional[CodeableConcept] = Field(
        description="The type of relationship between the product indication/contraindication and another therapy",
        default=None,
    )
    therapy: Optional[CodeableReference] = Field(
        description="Reference to a specific medication as part of an indication or contraindication",
        default=None,
    )

    @field_validator(
        *(
            "therapy",
            "relationshipType",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClinicalUseDefinitionContraindication(BackboneElement):
    """
    Specifics for when this is a contraindication.
    """

    diseaseSymptomProcedure: Optional[CodeableReference] = Field(
        description="The situation that is being documented as contraindicating against this item",
        default=None,
    )
    diseaseStatus: Optional[CodeableReference] = Field(
        description="The status of the disease or symptom for the contraindication",
        default=None,
    )
    comorbidity: Optional[List[CodeableReference]] = Field(
        description="A comorbidity (concurrent condition) or coinfection",
        default=None,
    )
    indication: Optional[List[Reference]] = Field(
        description="The indication which this is a contraidication for",
        default=None,
    )
    otherTherapy: Optional[List[ClinicalUseDefinitionContraindicationOtherTherapy]] = (
        Field(
            description="Information about use of the product in relation to other therapies described as part of the contraindication",
            default=None,
        )
    )

    @field_validator(
        *(
            "otherTherapy",
            "indication",
            "comorbidity",
            "diseaseStatus",
            "diseaseSymptomProcedure",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )


class ClinicalUseDefinitionIndicationOtherTherapy(BackboneElement):
    """
    Information about the use of the medicinal product in relation to other therapies described as part of the indication.
    """

    relationshipType: Optional[CodeableConcept] = Field(
        description="The type of relationship between the product indication/contraindication and another therapy",
        default=None,
    )
    therapy: Optional[CodeableReference] = Field(
        description="Reference to a specific medication as part of an indication or contraindication",
        default=None,
    )

    @field_validator(
        *(
            "therapy",
            "relationshipType",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClinicalUseDefinitionIndication(BackboneElement):
    """
    Specifics for when this is an indication.
    """

    diseaseSymptomProcedure: Optional[CodeableReference] = Field(
        description="The situation that is being documented as an indicaton for this item",
        default=None,
    )
    diseaseStatus: Optional[CodeableReference] = Field(
        description="The status of the disease or symptom for the indication",
        default=None,
    )
    comorbidity: Optional[List[CodeableReference]] = Field(
        description="A comorbidity or coinfection as part of the indication",
        default=None,
    )
    intendedEffect: Optional[CodeableReference] = Field(
        description="The intended effect, aim or strategy to be achieved",
        default=None,
    )
    durationRange: Optional[Range] = Field(
        description="Timing or duration information",
        default=None,
    )
    durationString: Optional[String] = Field(
        description="Timing or duration information",
        default=None,
    )
    durationString_ext: Optional[Element] = Field(
        description="Placeholder element for durationString extensions",
        default=None,
        alias="_durationString",
    )
    undesirableEffect: Optional[List[Reference]] = Field(
        description="An unwanted side effect or negative outcome of the subject of this resource when being used for this indication",
        default=None,
    )
    otherTherapy: Optional[List[ClinicalUseDefinitionIndicationOtherTherapy]] = Field(
        description="The use of the medicinal product in relation to other therapies described as part of the indication",
        default=None,
    )

    @property
    def duration(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="duration",
        )

    @field_validator(
        *(
            "otherTherapy",
            "undesirableEffect",
            "intendedEffect",
            "comorbidity",
            "diseaseStatus",
            "diseaseSymptomProcedure",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def duration_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Range, String],
            field_name_base="duration",
            required=False,
        )


class ClinicalUseDefinitionInteractionInteractant(BackboneElement):
    """
    The specific medication, food, substance or laboratory test that interacts.
    """

    itemReference: Optional[Reference] = Field(
        description="The specific medication, food or laboratory test that interacts",
        default=None,
    )
    itemCodeableConcept: Optional[CodeableConcept] = Field(
        description="The specific medication, food or laboratory test that interacts",
        default=None,
    )

    @property
    def item(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="item",
        )

    @model_validator(mode="after")
    def item_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Reference, CodeableConcept],
            field_name_base="item",
            required=True,
        )


class ClinicalUseDefinitionInteraction(BackboneElement):
    """
    Specifics for when this is an interaction.
    """

    interactant: Optional[List[ClinicalUseDefinitionInteractionInteractant]] = Field(
        description="The specific medication, food, substance or laboratory test that interacts",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="The type of the interaction e.g. drug-drug interaction, drug-lab test interaction",
        default=None,
    )
    effect: Optional[CodeableReference] = Field(
        description='The effect of the interaction, for example "reduced gastric absorption of primary medication"',
        default=None,
    )
    incidence: Optional[CodeableConcept] = Field(
        description="The incidence of the interaction, e.g. theoretical, observed",
        default=None,
    )
    management: Optional[List[CodeableConcept]] = Field(
        description="Actions for managing the interaction",
        default=None,
    )

    @field_validator(
        *(
            "management",
            "incidence",
            "effect",
            "type",
            "interactant",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClinicalUseDefinitionUndesirableEffect(BackboneElement):
    """
    Describe the possible undesirable effects (negative outcomes) from the use of the medicinal product as treatment.
    """

    symptomConditionEffect: Optional[CodeableReference] = Field(
        description="The situation in which the undesirable effect may manifest",
        default=None,
    )
    classification: Optional[CodeableConcept] = Field(
        description="High level classification of the effect",
        default=None,
    )
    frequencyOfOccurrence: Optional[CodeableConcept] = Field(
        description="How often the effect is seen",
        default=None,
    )

    @field_validator(
        *(
            "frequencyOfOccurrence",
            "classification",
            "symptomConditionEffect",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClinicalUseDefinitionWarning(BackboneElement):
    """
    A critical piece of information about environmental, health or physical risks or hazards that serve as caution to the user. For example 'Do not operate heavy machinery', 'May cause drowsiness', or 'Get medical advice/attention if you feel unwell'.
    """

    description: Optional[Markdown] = Field(
        description="A textual definition of this warning, with formatting",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    code: Optional[CodeableConcept] = Field(
        description="A coded or unformatted textual definition of this warning",
        default=None,
    )

    @field_validator(
        *(
            "code",
            "description",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClinicalUseDefinition(DomainResource):
    """
    A single issue - either an indication, contraindication, interaction or an undesirable effect for a medicinal product, medication, device or procedure.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/ClinicalUseDefinition"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business identifier for this issue",
        default=None,
    )
    type: Optional[Code] = Field(
        description="indication | contraindication | interaction | undesirable-effect | warning",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    category: Optional[List[CodeableConcept]] = Field(
        description='A categorisation of the issue, primarily for dividing warnings into subject heading areas such as "Pregnancy", "Overdose"',
        default=None,
    )
    subject: Optional[List[Reference]] = Field(
        description="The medication or procedure for which this is an indication",
        default=None,
    )
    status: Optional[CodeableConcept] = Field(
        description="Whether this is a current issue or one that has been retired etc",
        default=None,
    )
    contraindication: Optional[ClinicalUseDefinitionContraindication] = Field(
        description="Specifics for when this is a contraindication",
        default=None,
    )
    indication: Optional[ClinicalUseDefinitionIndication] = Field(
        description="Specifics for when this is an indication",
        default=None,
    )
    interaction: Optional[ClinicalUseDefinitionInteraction] = Field(
        description="Specifics for when this is an interaction",
        default=None,
    )
    population: Optional[List[Reference]] = Field(
        description="The population group to which this applies",
        default=None,
    )
    undesirableEffect: Optional[ClinicalUseDefinitionUndesirableEffect] = Field(
        description="A possible negative outcome from the use of this treatment",
        default=None,
    )
    warning: Optional[ClinicalUseDefinitionWarning] = Field(
        description="Critical environmental, health or physical risks or hazards. For example \u0027Do not operate heavy machinery\u0027, \u0027May cause drowsiness\u0027",
        default=None,
    )
    resourceType: Literal["ClinicalUseDefinition"] = Field(
        description=None,
        default="ClinicalUseDefinition",
    )

    @field_validator(
        *(
            "warning",
            "undesirableEffect",
            "population",
            "interaction",
            "indication",
            "contraindication",
            "status",
            "subject",
            "category",
            "type",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_cud_1_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="(ClinicalUseDefinition.indication.count() + ClinicalUseDefinition.contraindication.count() + ClinicalUseDefinition.interaction.count() + ClinicalUseDefinition.undesirableEffect.count() + ClinicalUseDefinition.warning.count())  < 2",
            human="Indication, Contraindication, Interaction, UndesirableEffect and Warning cannot be used in the same instance",
            key="cud-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
