# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    DateTime,
    Markdown,
    Date,
    PositiveInt,
)

from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    ContactDetail,
    UsageContext,
    CodeableConcept,
    Period,
    BackboneElement,
    Annotation,
    Reference,
    Attachment,
    HumanName,
    Address,
    ContactPoint,
    DomainResource,
)


class CitationSummary(BackboneElement):
    """
    A human-readable display of the citation.
    """

    style: Optional[CodeableConcept] = Field(
        description="Format for display of the citation",
        default=None,
    )
    text: Optional[Markdown] = Field(
        description="The human-readable display of the citation",
        default=None,
    )
    text_ext: Optional[Element] = Field(
        description="Placeholder element for text extensions",
        default=None,
        alias="_text",
    )

    @field_validator(
        *(
            "text",
            "style",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationClassification(BackboneElement):
    """
    The assignment to an organizing scheme.
    """

    type: Optional[CodeableConcept] = Field(
        description="The kind of classifier (e.g. publication type, keyword)",
        default=None,
    )
    classifier: Optional[List[CodeableConcept]] = Field(
        description="The specific classification value",
        default=None,
    )

    @field_validator(
        *(
            "classifier",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationStatusDate(BackboneElement):
    """
    An effective date or period for a status of the citation.
    """

    activity: Optional[CodeableConcept] = Field(
        description="Classification of the status",
        default=None,
    )
    actual: Optional[Boolean] = Field(
        description="Either occurred or expected",
        default=None,
    )
    actual_ext: Optional[Element] = Field(
        description="Placeholder element for actual extensions",
        default=None,
        alias="_actual",
    )
    period: Optional[Period] = Field(
        description="When the status started and/or ended",
        default=None,
    )

    @field_validator(
        *(
            "period",
            "actual",
            "activity",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationRelatesTo(BackboneElement):
    """
    Artifact related to the Citation Resource.
    """

    relationshipType: Optional[CodeableConcept] = Field(
        description="How the Citation resource relates to the target artifact",
        default=None,
    )
    targetClassifier: Optional[List[CodeableConcept]] = Field(
        description="The clasification of the related artifact",
        default=None,
    )
    targetUri: Optional[Uri] = Field(
        description="The article or artifact that the Citation Resource is related to",
        default=None,
    )
    targetUri_ext: Optional[Element] = Field(
        description="Placeholder element for targetUri extensions",
        default=None,
        alias="_targetUri",
    )
    targetIdentifier: Optional[Identifier] = Field(
        description="The article or artifact that the Citation Resource is related to",
        default=None,
    )
    targetReference: Optional[Reference] = Field(
        description="The article or artifact that the Citation Resource is related to",
        default=None,
    )
    targetAttachment: Optional[Attachment] = Field(
        description="The article or artifact that the Citation Resource is related to",
        default=None,
    )

    @property
    def target(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="target",
        )

    @field_validator(
        *(
            "targetClassifier",
            "relationshipType",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def target_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Uri, Identifier, Reference, Attachment],
            field_name_base="target",
            required=True,
        )


class CitationCitedArtifactVersion(BackboneElement):
    """
    The defined version of the cited artifact.
    """

    value: Optional[String] = Field(
        description="The version number or other version identifier",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )
    baseCitation: Optional[Reference] = Field(
        description="Citation for the main version of the cited artifact",
        default=None,
    )

    @field_validator(
        *(
            "baseCitation",
            "value",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactStatusDate(BackboneElement):
    """
    An effective date or period for a status of the cited artifact.
    """

    activity: Optional[CodeableConcept] = Field(
        description="Classification of the status",
        default=None,
    )
    actual: Optional[Boolean] = Field(
        description="Either occurred or expected",
        default=None,
    )
    actual_ext: Optional[Element] = Field(
        description="Placeholder element for actual extensions",
        default=None,
        alias="_actual",
    )
    period: Optional[Period] = Field(
        description="When the status started and/or ended",
        default=None,
    )

    @field_validator(
        *(
            "period",
            "actual",
            "activity",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactTitle(BackboneElement):
    """
    The title details of the article or artifact.
    """

    type: Optional[List[CodeableConcept]] = Field(
        description="The kind of title",
        default=None,
    )
    language: Optional[CodeableConcept] = Field(
        description="Used to express the specific language",
        default=None,
    )
    text: Optional[Markdown] = Field(
        description="The title of the article or artifact",
        default=None,
    )
    text_ext: Optional[Element] = Field(
        description="Placeholder element for text extensions",
        default=None,
        alias="_text",
    )

    @field_validator(
        *(
            "text",
            "language",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactAbstract(BackboneElement):
    """
    Summary of the article or artifact.
    """

    type: Optional[CodeableConcept] = Field(
        description="The kind of abstract",
        default=None,
    )
    language: Optional[CodeableConcept] = Field(
        description="Used to express the specific language",
        default=None,
    )
    text: Optional[Markdown] = Field(
        description="Abstract content",
        default=None,
    )
    text_ext: Optional[Element] = Field(
        description="Placeholder element for text extensions",
        default=None,
        alias="_text",
    )
    copyright: Optional[Markdown] = Field(
        description="Copyright notice for the abstract",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )

    @field_validator(
        *(
            "copyright",
            "text",
            "language",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactPart(BackboneElement):
    """
    The component of the article or artifact.
    """

    type: Optional[CodeableConcept] = Field(
        description="The kind of component",
        default=None,
    )
    value: Optional[String] = Field(
        description="The specification of the component",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )
    baseCitation: Optional[Reference] = Field(
        description="The citation for the full article or artifact",
        default=None,
    )

    @field_validator(
        *(
            "baseCitation",
            "value",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactRelatesTo(BackboneElement):
    """
    The artifact related to the cited artifact.
    """

    relationshipType: Optional[CodeableConcept] = Field(
        description="How the cited artifact relates to the target artifact",
        default=None,
    )
    targetClassifier: Optional[List[CodeableConcept]] = Field(
        description="The clasification of the related artifact",
        default=None,
    )
    targetUri: Optional[Uri] = Field(
        description="The article or artifact that the cited artifact is related to",
        default=None,
    )
    targetUri_ext: Optional[Element] = Field(
        description="Placeholder element for targetUri extensions",
        default=None,
        alias="_targetUri",
    )
    targetIdentifier: Optional[Identifier] = Field(
        description="The article or artifact that the cited artifact is related to",
        default=None,
    )
    targetReference: Optional[Reference] = Field(
        description="The article or artifact that the cited artifact is related to",
        default=None,
    )
    targetAttachment: Optional[Attachment] = Field(
        description="The article or artifact that the cited artifact is related to",
        default=None,
    )

    @property
    def target(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="target",
        )

    @field_validator(
        *(
            "targetClassifier",
            "relationshipType",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def target_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Uri, Identifier, Reference, Attachment],
            field_name_base="target",
            required=True,
        )


class CitationCitedArtifactPublicationFormPublishedIn(BackboneElement):
    """
    The collection the cited article or artifact is published in.
    """

    type: Optional[CodeableConcept] = Field(
        description="Kind of container (e.g. Periodical, database, or book)",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Journal identifiers include ISSN, ISO Abbreviation and NLMuniqueID; Book identifiers include ISBN",
        default=None,
    )
    title: Optional[String] = Field(
        description="Name of the database or title of the book or journal",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    publisher: Optional[Reference] = Field(
        description="Name of the publisher",
        default=None,
    )
    publisherLocation: Optional[String] = Field(
        description="Geographic location of the publisher",
        default=None,
    )
    publisherLocation_ext: Optional[Element] = Field(
        description="Placeholder element for publisherLocation extensions",
        default=None,
        alias="_publisherLocation",
    )

    @field_validator(
        *(
            "publisherLocation",
            "publisher",
            "title",
            "identifier",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactPublicationFormPeriodicReleaseDateOfPublication(
    BackboneElement
):
    """
    Defining the date on which the issue of the journal was published.
    """

    date: Optional[Date] = Field(
        description="Date on which the issue of the journal was published",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    year: Optional[String] = Field(
        description="Year on which the issue of the journal was published",
        default=None,
    )
    year_ext: Optional[Element] = Field(
        description="Placeholder element for year extensions",
        default=None,
        alias="_year",
    )
    month: Optional[String] = Field(
        description="Month on which the issue of the journal was published",
        default=None,
    )
    month_ext: Optional[Element] = Field(
        description="Placeholder element for month extensions",
        default=None,
        alias="_month",
    )
    day: Optional[String] = Field(
        description="Day on which the issue of the journal was published",
        default=None,
    )
    day_ext: Optional[Element] = Field(
        description="Placeholder element for day extensions",
        default=None,
        alias="_day",
    )
    season: Optional[String] = Field(
        description="Season on which the issue of the journal was published",
        default=None,
    )
    season_ext: Optional[Element] = Field(
        description="Placeholder element for season extensions",
        default=None,
        alias="_season",
    )
    text: Optional[String] = Field(
        description="Text representation of the date of which the issue of the journal was published",
        default=None,
    )
    text_ext: Optional[Element] = Field(
        description="Placeholder element for text extensions",
        default=None,
        alias="_text",
    )

    @field_validator(
        *(
            "text",
            "season",
            "day",
            "month",
            "year",
            "date",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactPublicationFormPeriodicRelease(BackboneElement):
    """
    The specific issue in which the cited article resides.
    """

    citedMedium: Optional[CodeableConcept] = Field(
        description="Internet or Print",
        default=None,
    )
    volume: Optional[String] = Field(
        description="Volume number of journal in which the article is published",
        default=None,
    )
    volume_ext: Optional[Element] = Field(
        description="Placeholder element for volume extensions",
        default=None,
        alias="_volume",
    )
    issue: Optional[String] = Field(
        description="Issue, part or supplement of journal in which the article is published",
        default=None,
    )
    issue_ext: Optional[Element] = Field(
        description="Placeholder element for issue extensions",
        default=None,
        alias="_issue",
    )
    dateOfPublication: Optional[
        CitationCitedArtifactPublicationFormPeriodicReleaseDateOfPublication
    ] = Field(
        description="Defining the date on which the issue of the journal was published",
        default=None,
    )

    @field_validator(
        *(
            "dateOfPublication",
            "issue",
            "volume",
            "citedMedium",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactPublicationForm(BackboneElement):
    """
    If multiple, used to represent alternative forms of the article that are not separate citations.
    """

    publishedIn: Optional[CitationCitedArtifactPublicationFormPublishedIn] = Field(
        description="The collection the cited article or artifact is published in",
        default=None,
    )
    periodicRelease: Optional[CitationCitedArtifactPublicationFormPeriodicRelease] = (
        Field(
            description="The specific issue in which the cited article resides",
            default=None,
        )
    )
    articleDate: Optional[DateTime] = Field(
        description="The date the article was added to the database, or the date the article was released",
        default=None,
    )
    articleDate_ext: Optional[Element] = Field(
        description="Placeholder element for articleDate extensions",
        default=None,
        alias="_articleDate",
    )
    lastRevisionDate: Optional[DateTime] = Field(
        description="The date the article was last revised or updated in the database",
        default=None,
    )
    lastRevisionDate_ext: Optional[Element] = Field(
        description="Placeholder element for lastRevisionDate extensions",
        default=None,
        alias="_lastRevisionDate",
    )
    language: Optional[List[CodeableConcept]] = Field(
        description="Language in which this form of the article is published",
        default=None,
    )
    accessionNumber: Optional[String] = Field(
        description="Entry number or identifier for inclusion in a database",
        default=None,
    )
    accessionNumber_ext: Optional[Element] = Field(
        description="Placeholder element for accessionNumber extensions",
        default=None,
        alias="_accessionNumber",
    )
    pageString: Optional[String] = Field(
        description="Used for full display of pagination",
        default=None,
    )
    pageString_ext: Optional[Element] = Field(
        description="Placeholder element for pageString extensions",
        default=None,
        alias="_pageString",
    )
    firstPage: Optional[String] = Field(
        description="Used for isolated representation of first page",
        default=None,
    )
    firstPage_ext: Optional[Element] = Field(
        description="Placeholder element for firstPage extensions",
        default=None,
        alias="_firstPage",
    )
    lastPage: Optional[String] = Field(
        description="Used for isolated representation of last page",
        default=None,
    )
    lastPage_ext: Optional[Element] = Field(
        description="Placeholder element for lastPage extensions",
        default=None,
        alias="_lastPage",
    )
    pageCount: Optional[String] = Field(
        description="Number of pages or screens",
        default=None,
    )
    pageCount_ext: Optional[Element] = Field(
        description="Placeholder element for pageCount extensions",
        default=None,
        alias="_pageCount",
    )
    copyright: Optional[Markdown] = Field(
        description="Copyright notice for the full article or artifact",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )

    @field_validator(
        *(
            "copyright",
            "pageCount",
            "lastPage",
            "firstPage",
            "pageString",
            "accessionNumber",
            "language",
            "lastRevisionDate",
            "articleDate",
            "periodicRelease",
            "publishedIn",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactWebLocation(BackboneElement):
    """
    Used for any URL for the article or artifact cited.
    """

    type: Optional[CodeableConcept] = Field(
        description="Code the reason for different URLs, e.g. abstract and full-text",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="The specific URL",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )

    @field_validator(
        *(
            "url",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactClassificationWhoClassified(BackboneElement):
    """
    Provenance and copyright of classification.
    """

    person: Optional[Reference] = Field(
        description="Person who created the classification",
        default=None,
    )
    organization: Optional[Reference] = Field(
        description="Organization who created the classification",
        default=None,
    )
    publisher: Optional[Reference] = Field(
        description="The publisher of the classification, not the publisher of the article or artifact being cited",
        default=None,
    )
    classifierCopyright: Optional[String] = Field(
        description="Rights management statement for the classification",
        default=None,
    )
    classifierCopyright_ext: Optional[Element] = Field(
        description="Placeholder element for classifierCopyright extensions",
        default=None,
        alias="_classifierCopyright",
    )
    freeToShare: Optional[Boolean] = Field(
        description="Acceptable to re-use the classification",
        default=None,
    )
    freeToShare_ext: Optional[Element] = Field(
        description="Placeholder element for freeToShare extensions",
        default=None,
        alias="_freeToShare",
    )

    @field_validator(
        *(
            "freeToShare",
            "classifierCopyright",
            "publisher",
            "organization",
            "person",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactClassification(BackboneElement):
    """
    The assignment to an organizing scheme.
    """

    type: Optional[CodeableConcept] = Field(
        description="The kind of classifier (e.g. publication type, keyword)",
        default=None,
    )
    classifier: Optional[List[CodeableConcept]] = Field(
        description="The specific classification value",
        default=None,
    )
    whoClassified: Optional[CitationCitedArtifactClassificationWhoClassified] = Field(
        description="Provenance and copyright of classification",
        default=None,
    )

    @field_validator(
        *(
            "whoClassified",
            "classifier",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactContributorshipEntryAffiliationInfo(BackboneElement):
    """
    Organization affiliated with the entity.
    """

    affiliation: Optional[String] = Field(
        description="Display for the organization",
        default=None,
    )
    affiliation_ext: Optional[Element] = Field(
        description="Placeholder element for affiliation extensions",
        default=None,
        alias="_affiliation",
    )
    role: Optional[String] = Field(
        description="Role within the organization, such as professional title",
        default=None,
    )
    role_ext: Optional[Element] = Field(
        description="Placeholder element for role extensions",
        default=None,
        alias="_role",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Identifier for the organization",
        default=None,
    )

    @field_validator(
        *(
            "identifier",
            "role",
            "affiliation",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactContributorshipEntryContributionInstance(BackboneElement):
    """
    Contributions with accounting for time or number.
    """

    type: Optional[CodeableConcept] = Field(
        description="The specific contribution",
        default=None,
    )
    time: Optional[DateTime] = Field(
        description="The time that the contribution was made",
        default=None,
    )
    time_ext: Optional[Element] = Field(
        description="Placeholder element for time extensions",
        default=None,
        alias="_time",
    )

    @field_validator(
        *(
            "time",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactContributorshipEntry(BackboneElement):
    """
    An individual entity named in the author list or contributor list.
    """

    name: Optional[HumanName] = Field(
        description="A name associated with the person",
        default=None,
    )
    initials: Optional[String] = Field(
        description="Initials for forename",
        default=None,
    )
    initials_ext: Optional[Element] = Field(
        description="Placeholder element for initials extensions",
        default=None,
        alias="_initials",
    )
    collectiveName: Optional[String] = Field(
        description="Used for collective or corporate name as an author",
        default=None,
    )
    collectiveName_ext: Optional[Element] = Field(
        description="Placeholder element for collectiveName extensions",
        default=None,
        alias="_collectiveName",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Author identifier, eg ORCID",
        default=None,
    )
    affiliationInfo: Optional[
        List[CitationCitedArtifactContributorshipEntryAffiliationInfo]
    ] = Field(
        description="Organizational affiliation",
        default=None,
    )
    address: Optional[List[Address]] = Field(
        description="Physical mailing address",
        default=None,
    )
    telecom: Optional[List[ContactPoint]] = Field(
        description="Email or telephone contact methods for the author or contributor",
        default=None,
    )
    contributionType: Optional[List[CodeableConcept]] = Field(
        description="The specific contribution",
        default=None,
    )
    role: Optional[CodeableConcept] = Field(
        description="The role of the contributor (e.g. author, editor, reviewer)",
        default=None,
    )
    contributionInstance: Optional[
        List[CitationCitedArtifactContributorshipEntryContributionInstance]
    ] = Field(
        description="Contributions with accounting for time or number",
        default=None,
    )
    correspondingContact: Optional[Boolean] = Field(
        description="Indication of which contributor is the corresponding contributor for the role",
        default=None,
    )
    correspondingContact_ext: Optional[Element] = Field(
        description="Placeholder element for correspondingContact extensions",
        default=None,
        alias="_correspondingContact",
    )
    listOrder: Optional[PositiveInt] = Field(
        description="Used to code order of authors",
        default=None,
    )
    listOrder_ext: Optional[Element] = Field(
        description="Placeholder element for listOrder extensions",
        default=None,
        alias="_listOrder",
    )

    @field_validator(
        *(
            "listOrder",
            "correspondingContact",
            "contributionInstance",
            "role",
            "contributionType",
            "telecom",
            "address",
            "affiliationInfo",
            "identifier",
            "collectiveName",
            "initials",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactContributorshipSummary(BackboneElement):
    """
    Used to record a display of the author/contributor list without separate coding for each list member.
    """

    type: Optional[CodeableConcept] = Field(
        description="Either authorList or contributorshipStatement",
        default=None,
    )
    style: Optional[CodeableConcept] = Field(
        description="The format for the display string",
        default=None,
    )
    source: Optional[CodeableConcept] = Field(
        description="Used to code the producer or rule for creating the display string",
        default=None,
    )
    value: Optional[Markdown] = Field(
        description="The display string for the author list, contributor list, or contributorship statement",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "source",
            "style",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifactContributorship(BackboneElement):
    """
    This element is used to list authors and other contributors, their contact information, specific contributions, and summary statements.
    """

    complete: Optional[Boolean] = Field(
        description="Indicates if the list includes all authors and/or contributors",
        default=None,
    )
    complete_ext: Optional[Element] = Field(
        description="Placeholder element for complete extensions",
        default=None,
        alias="_complete",
    )
    entry: Optional[List[CitationCitedArtifactContributorshipEntry]] = Field(
        description="An individual entity named in the list",
        default=None,
    )
    summary: Optional[List[CitationCitedArtifactContributorshipSummary]] = Field(
        description="Used to record a display of the author/contributor list without separate coding for each list member",
        default=None,
    )

    @field_validator(
        *(
            "summary",
            "entry",
            "complete",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )


class CitationCitedArtifact(BackboneElement):
    """
    The article or artifact being described.
    """

    identifier: Optional[List[Identifier]] = Field(
        description="May include DOI, PMID, PMCID, etc.",
        default=None,
    )
    relatedIdentifier: Optional[List[Identifier]] = Field(
        description="May include trial registry identifiers",
        default=None,
    )
    dateAccessed: Optional[DateTime] = Field(
        description="When the cited artifact was accessed",
        default=None,
    )
    dateAccessed_ext: Optional[Element] = Field(
        description="Placeholder element for dateAccessed extensions",
        default=None,
        alias="_dateAccessed",
    )
    version: Optional[CitationCitedArtifactVersion] = Field(
        description="The defined version of the cited artifact",
        default=None,
    )
    currentState: Optional[List[CodeableConcept]] = Field(
        description="The status of the cited artifact",
        default=None,
    )
    statusDate: Optional[List[CitationCitedArtifactStatusDate]] = Field(
        description="An effective date or period for a status of the cited artifact",
        default=None,
    )
    title: Optional[List[CitationCitedArtifactTitle]] = Field(
        description="The title details of the article or artifact",
        default=None,
    )
    abstract: Optional[List[CitationCitedArtifactAbstract]] = Field(
        description="Summary of the article or artifact",
        default=None,
    )
    part: Optional[CitationCitedArtifactPart] = Field(
        description="The component of the article or artifact",
        default=None,
    )
    relatesTo: Optional[List[CitationCitedArtifactRelatesTo]] = Field(
        description="The artifact related to the cited artifact",
        default=None,
    )
    publicationForm: Optional[List[CitationCitedArtifactPublicationForm]] = Field(
        description="If multiple, used to represent alternative forms of the article that are not separate citations",
        default=None,
    )
    webLocation: Optional[List[CitationCitedArtifactWebLocation]] = Field(
        description="Used for any URL for the article or artifact cited",
        default=None,
    )
    classification: Optional[List[CitationCitedArtifactClassification]] = Field(
        description="The assignment to an organizing scheme",
        default=None,
    )
    contributorship: Optional[CitationCitedArtifactContributorship] = Field(
        description="Attribution of authors and other contributors",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Any additional information or content for the article or artifact",
        default=None,
    )

    @field_validator(
        *(
            "note",
            "contributorship",
            "classification",
            "webLocation",
            "publicationForm",
            "relatesTo",
            "part",
            "abstract",
            "title",
            "statusDate",
            "currentState",
            "version",
            "dateAccessed",
            "relatedIdentifier",
            "identifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class Citation(DomainResource):
    """
    The Citation Resource enables reference to any knowledge artifact for purposes of identification and attribution. The Citation Resource supports existing reference structures and developing publication practices such as versioning, expressing complex contributorship roles, and referencing computable resources.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Citation"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this citation, represented as a globally unique URI",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Identifier for the Citation resource itself",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the citation",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    name: Optional[String] = Field(
        description="Name for this citation (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this citation (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="The publisher of the Citation, not the publisher of the article or artifact being cited",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher of the Citation Resource",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the citation",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the Citation Resource content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for citation (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this citation is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions for the Citation, not for the cited artifact",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    approvalDate: Optional[Date] = Field(
        description="When the citation was approved by publisher",
        default=None,
    )
    approvalDate_ext: Optional[Element] = Field(
        description="Placeholder element for approvalDate extensions",
        default=None,
        alias="_approvalDate",
    )
    lastReviewDate: Optional[Date] = Field(
        description="When the citation was last reviewed",
        default=None,
    )
    lastReviewDate_ext: Optional[Element] = Field(
        description="Placeholder element for lastReviewDate extensions",
        default=None,
        alias="_lastReviewDate",
    )
    effectivePeriod: Optional[Period] = Field(
        description="When the citation is expected to be used",
        default=None,
    )
    author: Optional[List[ContactDetail]] = Field(
        description="Who authored the Citation",
        default=None,
    )
    editor: Optional[List[ContactDetail]] = Field(
        description="Who edited the Citation",
        default=None,
    )
    reviewer: Optional[List[ContactDetail]] = Field(
        description="Who reviewed the Citation",
        default=None,
    )
    endorser: Optional[List[ContactDetail]] = Field(
        description="Who endorsed the Citation",
        default=None,
    )
    summary: Optional[List[CitationSummary]] = Field(
        description="A human-readable display of the citation",
        default=None,
    )
    classification: Optional[List[CitationClassification]] = Field(
        description="The assignment to an organizing scheme",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Used for general notes and annotations not coded elsewhere",
        default=None,
    )
    currentState: Optional[List[CodeableConcept]] = Field(
        description="The status of the citation",
        default=None,
    )
    statusDate: Optional[List[CitationStatusDate]] = Field(
        description="An effective date or period for a status of the citation",
        default=None,
    )
    relatesTo: Optional[List[CitationRelatesTo]] = Field(
        description="Artifact related to the Citation Resource",
        default=None,
    )
    citedArtifact: Optional[CitationCitedArtifact] = Field(
        description="The article or artifact being described",
        default=None,
    )
    resourceType: Literal["Citation"] = Field(
        description=None,
        default="Citation",
    )

    @field_validator(
        *(
            "citedArtifact",
            "relatesTo",
            "statusDate",
            "currentState",
            "note",
            "classification",
            "summary",
            "endorser",
            "reviewer",
            "editor",
            "author",
            "effectivePeriod",
            "lastReviewDate",
            "approvalDate",
            "copyright",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "experimental",
            "status",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count()) or $this is Parameters",
            human="All FHIR elements must have a @value or children unless an empty Parameters resource",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_cnl_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.exists() implies name.matches('[A-Z]([A-Za-z0-9_]){0,254}')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="cnl-0",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
