# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    UnsignedInt,
    Instant,
    PositiveInt,
    DateTime,
)

from fhircraft.fhir.resources.datatypes.R4B.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    BackboneElement,
    Period,
    DomainResource,
)


class AppointmentParticipant(BackboneElement):
    """
    List of participants involved in the appointment.
    """

    type: Optional[List[CodeableConcept]] = Field(
        description="Role of participant in the appointment",
        default=None,
    )
    actor: Optional[Reference] = Field(
        description="Person, Location/HealthcareService or Device",
        default=None,
    )
    required: Optional[Code] = Field(
        description="required | optional | information-only",
        default=None,
    )
    required_ext: Optional[Element] = Field(
        description="Placeholder element for required extensions",
        default=None,
        alias="_required",
    )
    status: Optional[Code] = Field(
        description="accepted | declined | tentative | needs-action",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    period: Optional[Period] = Field(
        description="Participation period of the actor",
        default=None,
    )

    @field_validator(
        *(
            "period",
            "status",
            "required",
            "actor",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class Appointment(DomainResource):
    """
    A booking of a healthcare event among patient(s), practitioner(s), related person(s) and/or device(s) for a specific date/time. This may result in one or more Encounter(s).
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Appointment"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="External Ids for this item",
        default=None,
    )
    status: Optional[Code] = Field(
        description="proposed | pending | booked | arrived | fulfilled | cancelled | noshow | entered-in-error | checked-in | waitlist",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    cancelationReason: Optional[CodeableConcept] = Field(
        description="The coded reason for the appointment being cancelled",
        default=None,
    )
    serviceCategory: Optional[List[CodeableConcept]] = Field(
        description="A broad categorization of the service that is to be performed during this appointment",
        default=None,
    )
    serviceType: Optional[List[CodeableConcept]] = Field(
        description="The specific service that is to be performed during this appointment",
        default=None,
    )
    specialty: Optional[List[CodeableConcept]] = Field(
        description="The specialty of a practitioner that would be required to perform the service requested in this appointment",
        default=None,
    )
    appointmentType: Optional[CodeableConcept] = Field(
        description="The style of appointment or patient that has been booked in the slot (not service type)",
        default=None,
    )
    reasonCode: Optional[List[CodeableConcept]] = Field(
        description="Coded reason this appointment is scheduled",
        default=None,
    )
    reasonReference: Optional[List[Reference]] = Field(
        description="Reason the appointment is to take place (resource)",
        default=None,
    )
    priority: Optional[UnsignedInt] = Field(
        description="Used to make informed decisions if needing to re-prioritize",
        default=None,
    )
    priority_ext: Optional[Element] = Field(
        description="Placeholder element for priority extensions",
        default=None,
        alias="_priority",
    )
    description: Optional[String] = Field(
        description="Shown on a subject line in a meeting request, or appointment list",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    supportingInformation: Optional[List[Reference]] = Field(
        description="Additional information to support the appointment",
        default=None,
    )
    start: Optional[Instant] = Field(
        description="When appointment is to take place",
        default=None,
    )
    start_ext: Optional[Element] = Field(
        description="Placeholder element for start extensions",
        default=None,
        alias="_start",
    )
    end: Optional[Instant] = Field(
        description="When appointment is to conclude",
        default=None,
    )
    end_ext: Optional[Element] = Field(
        description="Placeholder element for end extensions",
        default=None,
        alias="_end",
    )
    minutesDuration: Optional[PositiveInt] = Field(
        description="Can be less than start/end (e.g. estimate)",
        default=None,
    )
    minutesDuration_ext: Optional[Element] = Field(
        description="Placeholder element for minutesDuration extensions",
        default=None,
        alias="_minutesDuration",
    )
    slot: Optional[List[Reference]] = Field(
        description="The slots that this appointment is filling",
        default=None,
    )
    created: Optional[DateTime] = Field(
        description="The date that this appointment was initially created",
        default=None,
    )
    created_ext: Optional[Element] = Field(
        description="Placeholder element for created extensions",
        default=None,
        alias="_created",
    )
    comment: Optional[String] = Field(
        description="Additional comments",
        default=None,
    )
    comment_ext: Optional[Element] = Field(
        description="Placeholder element for comment extensions",
        default=None,
        alias="_comment",
    )
    patientInstruction: Optional[String] = Field(
        description="Detailed information and instructions for the patient",
        default=None,
    )
    patientInstruction_ext: Optional[Element] = Field(
        description="Placeholder element for patientInstruction extensions",
        default=None,
        alias="_patientInstruction",
    )
    basedOn: Optional[List[Reference]] = Field(
        description="The service request this appointment is allocated to assess",
        default=None,
    )
    participant: Optional[List[AppointmentParticipant]] = Field(
        description="Participants involved in appointment",
        default=None,
    )
    requestedPeriod: Optional[List[Period]] = Field(
        description="Potential date/time interval(s) requested to allocate the appointment within",
        default=None,
    )
    resourceType: Literal["Appointment"] = Field(
        description=None,
        default="Appointment",
    )

    @field_validator(
        *(
            "requestedPeriod",
            "participant",
            "basedOn",
            "patientInstruction",
            "comment",
            "created",
            "slot",
            "minutesDuration",
            "end",
            "start",
            "supportingInformation",
            "description",
            "priority",
            "reasonReference",
            "reasonCode",
            "appointmentType",
            "specialty",
            "serviceType",
            "serviceCategory",
            "cancelationReason",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contained",), mode="after", check_fields=None)
    @classmethod
    def FHIR_dom_r4b_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="($this is Citation or $this is Evidence or $this is EvidenceReport or $this is EvidenceVariable or $this is MedicinalProductDefinition or $this is PackagedProductDefinition or $this is AdministrableProductDefinition or $this is Ingredient or $this is ClinicalUseDefinition or $this is RegulatedAuthorization or $this is SubstanceDefinition or $this is SubscriptionStatus or $this is SubscriptionTopic) implies (%resource is Citation or %resource is Evidence or %resource is EvidenceReport or %resource is EvidenceVariable or %resource is MedicinalProductDefinition or %resource is PackagedProductDefinition or %resource is AdministrableProductDefinition or %resource is Ingredient or %resource is ClinicalUseDefinition or %resource is RegulatedAuthorization or %resource is SubstanceDefinition or %resource is SubscriptionStatus or %resource is SubscriptionTopic)",
            human="Containing new R4B resources within R4 resources may cause interoperability issues if instances are shared with R4 systems",
            key="dom-r4b",
            severity="warning",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("participant",), mode="after", check_fields=None)
    @classmethod
    def FHIR_app_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="type.exists() or actor.exists()",
            human="Either the type or actor on the participant SHALL be specified",
            key="app-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_app_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="start.exists() = end.exists()",
            human="Either start and end are specified, or neither",
            key="app-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_app_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="(start.exists() and end.exists()) or (status in ('proposed' | 'cancelled' | 'waitlist'))",
            human="Only proposed or cancelled appointments can be missing start/end dates",
            key="app-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_app_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="Appointment.cancelationReason.exists() implies (Appointment.status='no-show' or Appointment.status='cancelled')",
            human="Cancelation reason is only used for appointments that have been cancelled, or no-show",
            key="app-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where(((id.exists() and ('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url)))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(uri) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
