# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    Decimal,
    Integer,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    BackboneElement,
    CodeableConcept,
    Quantity,
    Annotation,
    DomainResource,
)


class VisionPrescriptionLensSpecificationPrism(BackboneElement):
    """
    Allows for adjustment on two axis.
    """

    amount: Optional[Decimal] = Field(
        description="Amount of adjustment",
        default=None,
    )
    amount_ext: Optional[Element] = Field(
        description="Placeholder element for amount extensions",
        default=None,
        alias="_amount",
    )
    base: Optional[Code] = Field(
        description="up | down | in | out",
        default=None,
    )
    base_ext: Optional[Element] = Field(
        description="Placeholder element for base extensions",
        default=None,
        alias="_base",
    )

    @field_validator(
        *(
            "base",
            "amount",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class VisionPrescriptionLensSpecification(BackboneElement):
    """
    Contain the details of  the individual lens specifications and serves as the authorization for the fullfillment by certified professionals.
    """

    product: Optional[CodeableConcept] = Field(
        description="Product to be supplied",
        default=None,
    )
    eye: Optional[Code] = Field(
        description="right | left",
        default=None,
    )
    eye_ext: Optional[Element] = Field(
        description="Placeholder element for eye extensions",
        default=None,
        alias="_eye",
    )
    sphere: Optional[Decimal] = Field(
        description="Power of the lens",
        default=None,
    )
    sphere_ext: Optional[Element] = Field(
        description="Placeholder element for sphere extensions",
        default=None,
        alias="_sphere",
    )
    cylinder: Optional[Decimal] = Field(
        description="Lens power for astigmatism",
        default=None,
    )
    cylinder_ext: Optional[Element] = Field(
        description="Placeholder element for cylinder extensions",
        default=None,
        alias="_cylinder",
    )
    axis: Optional[Integer] = Field(
        description="Lens meridian which contain no power for astigmatism",
        default=None,
    )
    axis_ext: Optional[Element] = Field(
        description="Placeholder element for axis extensions",
        default=None,
        alias="_axis",
    )
    prism: Optional[List[VisionPrescriptionLensSpecificationPrism]] = Field(
        description="Eye alignment compensation",
        default=None,
    )
    add: Optional[Decimal] = Field(
        description="Added power for multifocal levels",
        default=None,
    )
    add_ext: Optional[Element] = Field(
        description="Placeholder element for add extensions",
        default=None,
        alias="_add",
    )
    power: Optional[Decimal] = Field(
        description="Contact lens power",
        default=None,
    )
    power_ext: Optional[Element] = Field(
        description="Placeholder element for power extensions",
        default=None,
        alias="_power",
    )
    backCurve: Optional[Decimal] = Field(
        description="Contact lens back curvature",
        default=None,
    )
    backCurve_ext: Optional[Element] = Field(
        description="Placeholder element for backCurve extensions",
        default=None,
        alias="_backCurve",
    )
    diameter: Optional[Decimal] = Field(
        description="Contact lens diameter",
        default=None,
    )
    diameter_ext: Optional[Element] = Field(
        description="Placeholder element for diameter extensions",
        default=None,
        alias="_diameter",
    )
    duration: Optional[Quantity] = Field(
        description="Lens wear duration",
        default=None,
    )
    color: Optional[String] = Field(
        description="Color required",
        default=None,
    )
    color_ext: Optional[Element] = Field(
        description="Placeholder element for color extensions",
        default=None,
        alias="_color",
    )
    brand: Optional[String] = Field(
        description="Brand required",
        default=None,
    )
    brand_ext: Optional[Element] = Field(
        description="Placeholder element for brand extensions",
        default=None,
        alias="_brand",
    )
    note: Optional[List[Annotation]] = Field(
        description="Notes for coatings",
        default=None,
    )

    @field_validator(
        *(
            "note",
            "brand",
            "color",
            "duration",
            "diameter",
            "backCurve",
            "power",
            "add",
            "prism",
            "axis",
            "cylinder",
            "sphere",
            "eye",
            "product",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class VisionPrescription(DomainResource):
    """
    An authorization for the provision of glasses and/or contact lenses to a patient.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/VisionPrescription"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business Identifier for vision prescription",
        default=None,
    )
    status: Optional[Code] = Field(
        description="active | cancelled | draft | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    created: Optional[DateTime] = Field(
        description="Response creation date",
        default=None,
    )
    created_ext: Optional[Element] = Field(
        description="Placeholder element for created extensions",
        default=None,
        alias="_created",
    )
    patient: Optional[Reference] = Field(
        description="Who prescription is for",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Created during encounter / admission / stay",
        default=None,
    )
    dateWritten: Optional[DateTime] = Field(
        description="When prescription was authorized",
        default=None,
    )
    dateWritten_ext: Optional[Element] = Field(
        description="Placeholder element for dateWritten extensions",
        default=None,
        alias="_dateWritten",
    )
    prescriber: Optional[Reference] = Field(
        description="Who authorized the vision prescription",
        default=None,
    )
    lensSpecification: Optional[List[VisionPrescriptionLensSpecification]] = Field(
        description="Vision lens authorization",
        default=None,
    )
    resourceType: Literal["VisionPrescription"] = Field(
        description=None,
        default="VisionPrescription",
    )

    @field_validator(
        *(
            "lensSpecification",
            "prescriber",
            "dateWritten",
            "encounter",
            "patient",
            "created",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
