# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    Date,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Reference,
    CodeableConcept,
    Timing,
    BackboneElement,
    Signature,
    DomainResource,
)


class VerificationResultPrimarySource(BackboneElement):
    """
    Information about the primary source(s) involved in validation.
    """

    who: Optional[Reference] = Field(
        description="Reference to the primary source",
        default=None,
    )
    type: Optional[List[CodeableConcept]] = Field(
        description="Type of primary source (License Board; Primary Education; Continuing Education; Postal Service; Relationship owner; Registration Authority; legal source; issuing source; authoritative source)",
        default=None,
    )
    communicationMethod: Optional[List[CodeableConcept]] = Field(
        description="Method for exchanging information with the primary source",
        default=None,
    )
    validationStatus: Optional[CodeableConcept] = Field(
        description="successful | failed | unknown",
        default=None,
    )
    validationDate: Optional[DateTime] = Field(
        description="When the target was validated against the primary source",
        default=None,
    )
    validationDate_ext: Optional[Element] = Field(
        description="Placeholder element for validationDate extensions",
        default=None,
        alias="_validationDate",
    )
    canPushUpdates: Optional[CodeableConcept] = Field(
        description="yes | no | undetermined",
        default=None,
    )
    pushTypeAvailable: Optional[List[CodeableConcept]] = Field(
        description="specific | any | source",
        default=None,
    )

    @field_validator(
        *(
            "pushTypeAvailable",
            "canPushUpdates",
            "validationDate",
            "validationStatus",
            "communicationMethod",
            "type",
            "who",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class VerificationResultAttestation(BackboneElement):
    """
    Information about the entity attesting to information.
    """

    who: Optional[Reference] = Field(
        description="The individual or organization attesting to information",
        default=None,
    )
    onBehalfOf: Optional[Reference] = Field(
        description="When the who is asserting on behalf of another (organization or individual)",
        default=None,
    )
    communicationMethod: Optional[CodeableConcept] = Field(
        description="The method by which attested information was submitted/retrieved",
        default=None,
    )
    date: Optional[Date] = Field(
        description="The date the information was attested to",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    sourceIdentityCertificate: Optional[String] = Field(
        description="A digital identity certificate associated with the attestation source",
        default=None,
    )
    sourceIdentityCertificate_ext: Optional[Element] = Field(
        description="Placeholder element for sourceIdentityCertificate extensions",
        default=None,
        alias="_sourceIdentityCertificate",
    )
    proxyIdentityCertificate: Optional[String] = Field(
        description="A digital identity certificate associated with the proxy entity submitting attested information on behalf of the attestation source",
        default=None,
    )
    proxyIdentityCertificate_ext: Optional[Element] = Field(
        description="Placeholder element for proxyIdentityCertificate extensions",
        default=None,
        alias="_proxyIdentityCertificate",
    )
    proxySignature: Optional[Signature] = Field(
        description="Proxy signature",
        default=None,
    )
    sourceSignature: Optional[Signature] = Field(
        description="Attester signature",
        default=None,
    )

    @field_validator(
        *(
            "sourceSignature",
            "proxySignature",
            "proxyIdentityCertificate",
            "sourceIdentityCertificate",
            "date",
            "communicationMethod",
            "onBehalfOf",
            "who",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class VerificationResultValidator(BackboneElement):
    """
    Information about the entity validating information.
    """

    organization: Optional[Reference] = Field(
        description="Reference to the organization validating information",
        default=None,
    )
    identityCertificate: Optional[String] = Field(
        description="A digital identity certificate associated with the validator",
        default=None,
    )
    identityCertificate_ext: Optional[Element] = Field(
        description="Placeholder element for identityCertificate extensions",
        default=None,
        alias="_identityCertificate",
    )
    attestationSignature: Optional[Signature] = Field(
        description="Validator signature",
        default=None,
    )

    @field_validator(
        *(
            "attestationSignature",
            "identityCertificate",
            "organization",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class VerificationResult(DomainResource):
    """
    Describes validation requirements, source(s), status and dates for one or more elements.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/VerificationResult"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    target: Optional[List[Reference]] = Field(
        description="A resource that was validated",
        default=None,
    )
    targetLocation: Optional[List[String]] = Field(
        description="The fhirpath location(s) within the resource that was validated",
        default=None,
    )
    targetLocation_ext: Optional[Element] = Field(
        description="Placeholder element for targetLocation extensions",
        default=None,
        alias="_targetLocation",
    )
    need: Optional[CodeableConcept] = Field(
        description="none | initial | periodic",
        default=None,
    )
    status: Optional[Code] = Field(
        description="attested | validated | in-process | req-revalid | val-fail | reval-fail",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    statusDate: Optional[DateTime] = Field(
        description="When the validation status was updated",
        default=None,
    )
    statusDate_ext: Optional[Element] = Field(
        description="Placeholder element for statusDate extensions",
        default=None,
        alias="_statusDate",
    )
    validationType: Optional[CodeableConcept] = Field(
        description="nothing | primary | multiple",
        default=None,
    )
    validationProcess: Optional[List[CodeableConcept]] = Field(
        description="The primary process by which the target is validated (edit check; value set; primary source; multiple sources; standalone; in context)",
        default=None,
    )
    frequency: Optional[Timing] = Field(
        description="Frequency of revalidation",
        default=None,
    )
    lastPerformed: Optional[DateTime] = Field(
        description="The date/time validation was last completed (including failed validations)",
        default=None,
    )
    lastPerformed_ext: Optional[Element] = Field(
        description="Placeholder element for lastPerformed extensions",
        default=None,
        alias="_lastPerformed",
    )
    nextScheduled: Optional[Date] = Field(
        description="The date when target is next validated, if appropriate",
        default=None,
    )
    nextScheduled_ext: Optional[Element] = Field(
        description="Placeholder element for nextScheduled extensions",
        default=None,
        alias="_nextScheduled",
    )
    failureAction: Optional[CodeableConcept] = Field(
        description="fatal | warn | rec-only | none",
        default=None,
    )
    primarySource: Optional[List[VerificationResultPrimarySource]] = Field(
        description="Information about the primary source(s) involved in validation",
        default=None,
    )
    attestation: Optional[VerificationResultAttestation] = Field(
        description="Information about the entity attesting to information",
        default=None,
    )
    validator: Optional[List[VerificationResultValidator]] = Field(
        description="Information about the entity validating information",
        default=None,
    )
    resourceType: Literal["VerificationResult"] = Field(
        description=None,
        default="VerificationResult",
    )

    @field_validator(
        *(
            "validator",
            "attestation",
            "primarySource",
            "failureAction",
            "nextScheduled",
            "lastPerformed",
            "frequency",
            "validationProcess",
            "validationType",
            "statusDate",
            "status",
            "need",
            "targetLocation",
            "target",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
