# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    DateTime,
    Markdown,
    Date,
    Canonical,
    Integer,
    Decimal,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    ContactDetail,
    UsageContext,
    CodeableConcept,
    BackboneElement,
    Coding,
    DomainResource,
)


class ValueSetComposeIncludeConceptDesignation(BackboneElement):
    """
    Additional representations for this concept when used in this value set - other languages, aliases, specialized purposes, used for particular purposes, etc.
    """

    language: Optional[Code] = Field(
        description="Human language of the designation",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    use: Optional[Coding] = Field(
        description="Types of uses of designations",
        default=None,
    )
    value: Optional[String] = Field(
        description="The text value for this designation",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "use",
            "language",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ValueSetComposeIncludeConcept(BackboneElement):
    """
    Specifies a concept to be included or excluded.
    """

    code: Optional[Code] = Field(
        description="Code or expression from system",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    display: Optional[String] = Field(
        description="Text to display for this code for this value set in this valueset",
        default=None,
    )
    display_ext: Optional[Element] = Field(
        description="Placeholder element for display extensions",
        default=None,
        alias="_display",
    )
    designation: Optional[List[ValueSetComposeIncludeConceptDesignation]] = Field(
        description="Additional representations for this concept",
        default=None,
    )

    @field_validator(
        *(
            "designation",
            "display",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ValueSetComposeIncludeFilter(BackboneElement):
    """
    Select concepts by specify a matching criterion based on the properties (including relationships) defined by the system, or on filters defined by the system. If multiple filters are specified, they SHALL all be true.
    """

    property_: Optional[Code] = Field(
        description="A property/filter defined by the code system",
        default=None,
    )
    property_ext: Optional[Element] = Field(
        description="Placeholder element for property extensions",
        default=None,
        alias="_property",
    )
    op: Optional[Code] = Field(
        description="= | is-a | descendent-of | is-not-a | regex | in | not-in | generalizes | exists",
        default=None,
    )
    op_ext: Optional[Element] = Field(
        description="Placeholder element for op extensions",
        default=None,
        alias="_op",
    )
    value: Optional[String] = Field(
        description="Code from the system, or regex criteria, or boolean value for exists",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "op",
            "property_",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ValueSetComposeInclude(BackboneElement):
    """
    Include one or more codes from a code system or other value set(s).
    """

    system: Optional[Uri] = Field(
        description="The system the codes come from",
        default=None,
    )
    system_ext: Optional[Element] = Field(
        description="Placeholder element for system extensions",
        default=None,
        alias="_system",
    )
    version: Optional[String] = Field(
        description="Specific version of the code system referred to",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    concept: Optional[List[ValueSetComposeIncludeConcept]] = Field(
        description="A concept defined in the system",
        default=None,
    )
    filter: Optional[List[ValueSetComposeIncludeFilter]] = Field(
        description="Select codes/concepts by their properties (including relationships)",
        default=None,
    )
    valueSet: Optional[List[Canonical]] = Field(
        description="Select the contents included in this value set",
        default=None,
    )
    valueSet_ext: Optional[Element] = Field(
        description="Placeholder element for valueSet extensions",
        default=None,
        alias="_valueSet",
    )

    @field_validator(
        *(
            "valueSet",
            "filter",
            "concept",
            "version",
            "system",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ValueSetComposeIncludeConceptDesignation(BackboneElement):
    """
    Additional representations for this concept when used in this value set - other languages, aliases, specialized purposes, used for particular purposes, etc.
    """

    language: Optional[Code] = Field(
        description="Human language of the designation",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    use: Optional[Coding] = Field(
        description="Types of uses of designations",
        default=None,
    )
    value: Optional[String] = Field(
        description="The text value for this designation",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "use",
            "language",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ValueSetComposeIncludeConcept(BackboneElement):
    """
    Specifies a concept to be included or excluded.
    """

    code: Optional[Code] = Field(
        description="Code or expression from system",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    display: Optional[String] = Field(
        description="Text to display for this code for this value set in this valueset",
        default=None,
    )
    display_ext: Optional[Element] = Field(
        description="Placeholder element for display extensions",
        default=None,
        alias="_display",
    )
    designation: Optional[List[ValueSetComposeIncludeConceptDesignation]] = Field(
        description="Additional representations for this concept",
        default=None,
    )

    @field_validator(
        *(
            "designation",
            "display",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ValueSetComposeIncludeFilter(BackboneElement):
    """
    Select concepts by specify a matching criterion based on the properties (including relationships) defined by the system, or on filters defined by the system. If multiple filters are specified, they SHALL all be true.
    """

    property_: Optional[Code] = Field(
        description="A property/filter defined by the code system",
        default=None,
    )
    property_ext: Optional[Element] = Field(
        description="Placeholder element for property extensions",
        default=None,
        alias="_property",
    )
    op: Optional[Code] = Field(
        description="= | is-a | descendent-of | is-not-a | regex | in | not-in | generalizes | exists",
        default=None,
    )
    op_ext: Optional[Element] = Field(
        description="Placeholder element for op extensions",
        default=None,
        alias="_op",
    )
    value: Optional[String] = Field(
        description="Code from the system, or regex criteria, or boolean value for exists",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "op",
            "property_",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ValueSetComposeExclude(BackboneElement):
    """
    Exclude one or more codes from the value set based on code system filters and/or other value sets.
    """

    system: Optional[Uri] = Field(
        description="The system the codes come from",
        default=None,
    )
    system_ext: Optional[Element] = Field(
        description="Placeholder element for system extensions",
        default=None,
        alias="_system",
    )
    version: Optional[String] = Field(
        description="Specific version of the code system referred to",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    concept: Optional[List[ValueSetComposeIncludeConcept]] = Field(
        description="A concept defined in the system",
        default=None,
    )
    filter: Optional[List[ValueSetComposeIncludeFilter]] = Field(
        description="Select codes/concepts by their properties (including relationships)",
        default=None,
    )
    valueSet: Optional[List[Canonical]] = Field(
        description="Select the contents included in this value set",
        default=None,
    )
    valueSet_ext: Optional[Element] = Field(
        description="Placeholder element for valueSet extensions",
        default=None,
        alias="_valueSet",
    )

    @field_validator(
        *(
            "valueSet",
            "filter",
            "concept",
            "version",
            "system",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ValueSetCompose(BackboneElement):
    """
    A set of criteria that define the contents of the value set by including or excluding codes selected from the specified code system(s) that the value set draws from. This is also known as the Content Logical Definition (CLD).
    """

    lockedDate: Optional[Date] = Field(
        description="Fixed date for references with no specified version (transitive)",
        default=None,
    )
    lockedDate_ext: Optional[Element] = Field(
        description="Placeholder element for lockedDate extensions",
        default=None,
        alias="_lockedDate",
    )
    inactive: Optional[Boolean] = Field(
        description="Whether inactive codes are in the value set",
        default=None,
    )
    inactive_ext: Optional[Element] = Field(
        description="Placeholder element for inactive extensions",
        default=None,
        alias="_inactive",
    )
    include: Optional[List[ValueSetComposeInclude]] = Field(
        description="Include one or more codes from a code system or other value set(s)",
        default=None,
    )
    exclude: Optional[List[ValueSetComposeExclude]] = Field(
        description="Explicitly exclude codes from a code system or other value sets",
        default=None,
    )

    @field_validator(
        *(
            "exclude",
            "include",
            "inactive",
            "lockedDate",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("include",), mode="after", check_fields=None)
    @classmethod
    def FHIR_vsd_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="valueSet.exists() or system.exists()",
            human="A value set include/exclude SHALL have a value set or a system",
            key="vsd-1",
            severity="error",
        )

    @field_validator(*("include",), mode="after", check_fields=None)
    @classmethod
    def FHIR_vsd_2_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(concept.exists() or filter.exists()) implies system.exists()",
            human="A value set with concepts or filters SHALL include a system",
            key="vsd-2",
            severity="error",
        )

    @field_validator(*("include",), mode="after", check_fields=None)
    @classmethod
    def FHIR_vsd_3_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="concept.empty() or filter.empty()",
            human="Cannot have both concept and filter",
            key="vsd-3",
            severity="error",
        )


class ValueSetExpansionParameter(BackboneElement):
    """
    A parameter that controlled the expansion process. These parameters may be used by users of expanded value sets to check whether the expansion is suitable for a particular purpose, or to pick the correct expansion.
    """

    name: Optional[String] = Field(
        description="Name as assigned by the client or server",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    valueString: Optional[String] = Field(
        description="Value of the named parameter",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueBoolean: Optional[Boolean] = Field(
        description="Value of the named parameter",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueInteger: Optional[Integer] = Field(
        description="Value of the named parameter",
        default=None,
    )
    valueInteger_ext: Optional[Element] = Field(
        description="Placeholder element for valueInteger extensions",
        default=None,
        alias="_valueInteger",
    )
    valueDecimal: Optional[Decimal] = Field(
        description="Value of the named parameter",
        default=None,
    )
    valueDecimal_ext: Optional[Element] = Field(
        description="Placeholder element for valueDecimal extensions",
        default=None,
        alias="_valueDecimal",
    )
    valueUri: Optional[Uri] = Field(
        description="Value of the named parameter",
        default=None,
    )
    valueUri_ext: Optional[Element] = Field(
        description="Placeholder element for valueUri extensions",
        default=None,
        alias="_valueUri",
    )
    valueCode: Optional[Code] = Field(
        description="Value of the named parameter",
        default=None,
    )
    valueCode_ext: Optional[Element] = Field(
        description="Placeholder element for valueCode extensions",
        default=None,
        alias="_valueCode",
    )
    valueDateTime: Optional[DateTime] = Field(
        description="Value of the named parameter",
        default=None,
    )
    valueDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for valueDateTime extensions",
        default=None,
        alias="_valueDateTime",
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("name", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[String, Boolean, Integer, Decimal, Uri, Code, DateTime],
            field_name_base="value",
            required=False,
        )


class ValueSetExpansionContainsDesignation(BackboneElement):
    """
    Additional representations for this item - other languages, aliases, specialized purposes, used for particular purposes, etc. These are relevant when the conditions of the expansion do not fix to a single correct representation.
    """

    language: Optional[Code] = Field(
        description="Human language of the designation",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    use: Optional[Coding] = Field(
        description="Types of uses of designations",
        default=None,
    )
    value: Optional[String] = Field(
        description="The text value for this designation",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "use",
            "language",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ValueSetExpansionContains(BackboneElement):
    """
    The codes that are contained in the value set expansion.
    """

    system: Optional[Uri] = Field(
        description="System value for the code",
        default=None,
    )
    system_ext: Optional[Element] = Field(
        description="Placeholder element for system extensions",
        default=None,
        alias="_system",
    )
    abstract: Optional[Boolean] = Field(
        description="If user cannot select this entry",
        default=None,
    )
    abstract_ext: Optional[Element] = Field(
        description="Placeholder element for abstract extensions",
        default=None,
        alias="_abstract",
    )
    inactive: Optional[Boolean] = Field(
        description="If concept is inactive in the code system",
        default=None,
    )
    inactive_ext: Optional[Element] = Field(
        description="Placeholder element for inactive extensions",
        default=None,
        alias="_inactive",
    )
    version: Optional[String] = Field(
        description="Version in which this code/display is defined",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    code: Optional[Code] = Field(
        description="Code - if blank, this is not a selectable code",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    display: Optional[String] = Field(
        description="User display for the concept",
        default=None,
    )
    display_ext: Optional[Element] = Field(
        description="Placeholder element for display extensions",
        default=None,
        alias="_display",
    )
    designation: Optional[List[ValueSetExpansionContainsDesignation]] = Field(
        description="Additional representations for this item",
        default=None,
    )
    contains: Optional[List["ValueSetExpansionContains"]] = Field(
        description="Codes contained under this entry",
        default=None,
    )

    @field_validator(
        *(
            "contains",
            "designation",
            "display",
            "code",
            "version",
            "inactive",
            "abstract",
            "system",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ValueSetExpansion(BackboneElement):
    """
    A value set can also be "expanded", where the value set is turned into a simple collection of enumerated codes. This element holds the expansion, if it has been performed.
    """

    identifier: Optional[Uri] = Field(
        description="Identifies the value set expansion (business identifier)",
        default=None,
    )
    identifier_ext: Optional[Element] = Field(
        description="Placeholder element for identifier extensions",
        default=None,
        alias="_identifier",
    )
    timestamp: Optional[DateTime] = Field(
        description="Time ValueSet expansion happened",
        default=None,
    )
    timestamp_ext: Optional[Element] = Field(
        description="Placeholder element for timestamp extensions",
        default=None,
        alias="_timestamp",
    )
    total: Optional[Integer] = Field(
        description="Total number of codes in the expansion",
        default=None,
    )
    total_ext: Optional[Element] = Field(
        description="Placeholder element for total extensions",
        default=None,
        alias="_total",
    )
    offset: Optional[Integer] = Field(
        description="Offset at which this resource starts",
        default=None,
    )
    offset_ext: Optional[Element] = Field(
        description="Placeholder element for offset extensions",
        default=None,
        alias="_offset",
    )
    parameter: Optional[List[ValueSetExpansionParameter]] = Field(
        description="Parameter that controlled the expansion process",
        default=None,
    )
    contains: Optional[List[ValueSetExpansionContains]] = Field(
        description="Codes in the value set",
        default=None,
    )

    @field_validator(
        *(
            "contains",
            "parameter",
            "offset",
            "total",
            "timestamp",
            "identifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("contains",), mode="after", check_fields=None)
    @classmethod
    def FHIR_vsd_6_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="code.exists() or display.exists()",
            human="SHALL have a code or a display",
            key="vsd-6",
            severity="error",
        )

    @field_validator(*("contains",), mode="after", check_fields=None)
    @classmethod
    def FHIR_vsd_9_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="code.exists() or abstract = true",
            human="Must have a code if not abstract",
            key="vsd-9",
            severity="error",
        )

    @field_validator(*("contains",), mode="after", check_fields=None)
    @classmethod
    def FHIR_vsd_10_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="code.empty() or system.exists()",
            human="Must have a system if a code is present",
            key="vsd-10",
            severity="error",
        )


class ValueSet(DomainResource):
    """
    A ValueSet resource instance specifies a set of codes drawn from one or more code systems, intended for use in a particular context. Value sets link between [[[CodeSystem]]] definitions and their use in [coded elements](terminologies.html).
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/ValueSet"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this value set, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifier for the value set (business identifier)",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the value set",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    name: Optional[String] = Field(
        description="Name for this value set (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this value set (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the value set",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for value set (if applicable)",
        default=None,
    )
    immutable: Optional[Boolean] = Field(
        description="Indicates whether or not any change to the content logical definition may occur",
        default=None,
    )
    immutable_ext: Optional[Element] = Field(
        description="Placeholder element for immutable extensions",
        default=None,
        alias="_immutable",
    )
    purpose: Optional[Markdown] = Field(
        description="Why this value set is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    compose: Optional[ValueSetCompose] = Field(
        description="Content logical definition of the value set (CLD)",
        default=None,
    )
    expansion: Optional[ValueSetExpansion] = Field(
        description='Used when the value set is "expanded"',
        default=None,
    )
    resourceType: Literal["ValueSet"] = Field(
        description=None,
        default="ValueSet",
    )

    @field_validator(
        *(
            "expansion",
            "compose",
            "copyright",
            "purpose",
            "immutable",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "experimental",
            "status",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_vsd_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.matches('[A-Z]([A-Za-z0-9_]){0,254}')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="vsd-0",
            severity="warning",
        )
