# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    DateTime,
    Markdown,
    Integer,
    Canonical,
    Id,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    ContactDetail,
    UsageContext,
    CodeableConcept,
    BackboneElement,
    Coding,
    Reference,
    DomainResource,
)


class TestScriptOrigin(BackboneElement):
    """
    An abstract server used in operations within this test script in the origin element.
    """

    index: Optional[Integer] = Field(
        description="The index of the abstract origin server starting at 1",
        default=None,
    )
    index_ext: Optional[Element] = Field(
        description="Placeholder element for index extensions",
        default=None,
        alias="_index",
    )
    profile: Optional[Coding] = Field(
        description="FHIR-Client | FHIR-SDC-FormFiller",
        default=None,
    )

    @field_validator(
        *(
            "profile",
            "index",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestScriptDestination(BackboneElement):
    """
    An abstract server used in operations within this test script in the destination element.
    """

    index: Optional[Integer] = Field(
        description="The index of the abstract destination server starting at 1",
        default=None,
    )
    index_ext: Optional[Element] = Field(
        description="Placeholder element for index extensions",
        default=None,
        alias="_index",
    )
    profile: Optional[Coding] = Field(
        description="FHIR-Server | FHIR-SDC-FormManager | FHIR-SDC-FormReceiver | FHIR-SDC-FormProcessor",
        default=None,
    )

    @field_validator(
        *(
            "profile",
            "index",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestScriptMetadataLink(BackboneElement):
    """
    A link to the FHIR specification that this test is covering.
    """

    url: Optional[Uri] = Field(
        description="URL to the specification",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    description: Optional[String] = Field(
        description="Short description",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )

    @field_validator(
        *(
            "description",
            "url",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestScriptMetadataCapability(BackboneElement):
    """
    Capabilities that must exist and are assumed to function correctly on the FHIR server being tested.
    """

    required: Optional[Boolean] = Field(
        description="Are the capabilities required?",
        default=None,
    )
    required_ext: Optional[Element] = Field(
        description="Placeholder element for required extensions",
        default=None,
        alias="_required",
    )
    validated: Optional[Boolean] = Field(
        description="Are the capabilities validated?",
        default=None,
    )
    validated_ext: Optional[Element] = Field(
        description="Placeholder element for validated extensions",
        default=None,
        alias="_validated",
    )
    description: Optional[String] = Field(
        description="The expected capabilities of the server",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    origin: Optional[List[Integer]] = Field(
        description="Which origin server these requirements apply to",
        default=None,
    )
    origin_ext: Optional[Element] = Field(
        description="Placeholder element for origin extensions",
        default=None,
        alias="_origin",
    )
    destination: Optional[Integer] = Field(
        description="Which server these requirements apply to",
        default=None,
    )
    destination_ext: Optional[Element] = Field(
        description="Placeholder element for destination extensions",
        default=None,
        alias="_destination",
    )
    link: Optional[List[Uri]] = Field(
        description="Links to the FHIR specification",
        default=None,
    )
    link_ext: Optional[Element] = Field(
        description="Placeholder element for link extensions",
        default=None,
        alias="_link",
    )
    capabilities: Optional[Canonical] = Field(
        description="Required Capability Statement",
        default=None,
    )
    capabilities_ext: Optional[Element] = Field(
        description="Placeholder element for capabilities extensions",
        default=None,
        alias="_capabilities",
    )

    @field_validator(
        *(
            "capabilities",
            "link",
            "destination",
            "origin",
            "description",
            "validated",
            "required",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestScriptMetadata(BackboneElement):
    """
    The required capability must exist and are assumed to function correctly on the FHIR server being tested.
    """

    link: Optional[List[TestScriptMetadataLink]] = Field(
        description="Links to the FHIR specification",
        default=None,
    )
    capability: Optional[List[TestScriptMetadataCapability]] = Field(
        description="Capabilities  that are assumed to function correctly on the FHIR server being tested",
        default=None,
    )

    @field_validator(
        *(
            "capability",
            "link",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestScriptFixture(BackboneElement):
    """
    Fixture in the test script - by reference (uri). All fixtures are required for the test script to execute.
    """

    autocreate: Optional[Boolean] = Field(
        description="Whether or not to implicitly create the fixture during setup",
        default=None,
    )
    autocreate_ext: Optional[Element] = Field(
        description="Placeholder element for autocreate extensions",
        default=None,
        alias="_autocreate",
    )
    autodelete: Optional[Boolean] = Field(
        description="Whether or not to implicitly delete the fixture during teardown",
        default=None,
    )
    autodelete_ext: Optional[Element] = Field(
        description="Placeholder element for autodelete extensions",
        default=None,
        alias="_autodelete",
    )
    resource: Optional[Reference] = Field(
        description="Reference of the resource",
        default=None,
    )

    @field_validator(
        *(
            "resource",
            "autodelete",
            "autocreate",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestScriptVariable(BackboneElement):
    """
    Variable is set based either on element value in response body or on header field value in the response headers.
    """

    name: Optional[String] = Field(
        description="Descriptive name for this variable",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    defaultValue: Optional[String] = Field(
        description="Default, hard-coded, or user-defined value for this variable",
        default=None,
    )
    defaultValue_ext: Optional[Element] = Field(
        description="Placeholder element for defaultValue extensions",
        default=None,
        alias="_defaultValue",
    )
    description: Optional[String] = Field(
        description="Natural language description of the variable",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    expression: Optional[String] = Field(
        description="The FHIRPath expression against the fixture body",
        default=None,
    )
    expression_ext: Optional[Element] = Field(
        description="Placeholder element for expression extensions",
        default=None,
        alias="_expression",
    )
    headerField: Optional[String] = Field(
        description="HTTP header field name for source",
        default=None,
    )
    headerField_ext: Optional[Element] = Field(
        description="Placeholder element for headerField extensions",
        default=None,
        alias="_headerField",
    )
    hint: Optional[String] = Field(
        description="Hint help text for default value to enter",
        default=None,
    )
    hint_ext: Optional[Element] = Field(
        description="Placeholder element for hint extensions",
        default=None,
        alias="_hint",
    )
    path: Optional[String] = Field(
        description="XPath or JSONPath against the fixture body",
        default=None,
    )
    path_ext: Optional[Element] = Field(
        description="Placeholder element for path extensions",
        default=None,
        alias="_path",
    )
    sourceId: Optional[Id] = Field(
        description="Fixture Id of source expression or headerField within this variable",
        default=None,
    )
    sourceId_ext: Optional[Element] = Field(
        description="Placeholder element for sourceId extensions",
        default=None,
        alias="_sourceId",
    )

    @field_validator(
        *(
            "sourceId",
            "path",
            "hint",
            "headerField",
            "expression",
            "description",
            "defaultValue",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestScriptSetupActionOperationRequestHeader(BackboneElement):
    """
    Header elements would be used to set HTTP headers.
    """

    field: Optional[String] = Field(
        description="HTTP header field name",
        default=None,
    )
    field_ext: Optional[Element] = Field(
        description="Placeholder element for field extensions",
        default=None,
        alias="_field",
    )
    value: Optional[String] = Field(
        description="HTTP headerfield value",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "field",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestScriptSetupActionOperation(BackboneElement):
    """
    The operation to perform.
    """

    type: Optional[Coding] = Field(
        description="The operation code type that will be executed",
        default=None,
    )
    resource: Optional[Code] = Field(
        description="Resource type",
        default=None,
    )
    resource_ext: Optional[Element] = Field(
        description="Placeholder element for resource extensions",
        default=None,
        alias="_resource",
    )
    label: Optional[String] = Field(
        description="Tracking/logging operation label",
        default=None,
    )
    label_ext: Optional[Element] = Field(
        description="Placeholder element for label extensions",
        default=None,
        alias="_label",
    )
    description: Optional[String] = Field(
        description="Tracking/reporting operation description",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    accept: Optional[Code] = Field(
        description="Mime type to accept in the payload of the response, with charset etc.",
        default=None,
    )
    accept_ext: Optional[Element] = Field(
        description="Placeholder element for accept extensions",
        default=None,
        alias="_accept",
    )
    contentType: Optional[Code] = Field(
        description="Mime type of the request payload contents, with charset etc.",
        default=None,
    )
    contentType_ext: Optional[Element] = Field(
        description="Placeholder element for contentType extensions",
        default=None,
        alias="_contentType",
    )
    destination: Optional[Integer] = Field(
        description="Server responding to the request",
        default=None,
    )
    destination_ext: Optional[Element] = Field(
        description="Placeholder element for destination extensions",
        default=None,
        alias="_destination",
    )
    encodeRequestUrl: Optional[Boolean] = Field(
        description="Whether or not to send the request url in encoded format",
        default=None,
    )
    encodeRequestUrl_ext: Optional[Element] = Field(
        description="Placeholder element for encodeRequestUrl extensions",
        default=None,
        alias="_encodeRequestUrl",
    )
    method: Optional[Code] = Field(
        description="delete | get | options | patch | post | put | head",
        default=None,
    )
    method_ext: Optional[Element] = Field(
        description="Placeholder element for method extensions",
        default=None,
        alias="_method",
    )
    origin: Optional[Integer] = Field(
        description="Server initiating the request",
        default=None,
    )
    origin_ext: Optional[Element] = Field(
        description="Placeholder element for origin extensions",
        default=None,
        alias="_origin",
    )
    params: Optional[String] = Field(
        description="Explicitly defined path parameters",
        default=None,
    )
    params_ext: Optional[Element] = Field(
        description="Placeholder element for params extensions",
        default=None,
        alias="_params",
    )
    requestHeader: Optional[List[TestScriptSetupActionOperationRequestHeader]] = Field(
        description="Each operation can have one or more header elements",
        default=None,
    )
    requestId: Optional[Id] = Field(
        description="Fixture Id of mapped request",
        default=None,
    )
    requestId_ext: Optional[Element] = Field(
        description="Placeholder element for requestId extensions",
        default=None,
        alias="_requestId",
    )
    responseId: Optional[Id] = Field(
        description="Fixture Id of mapped response",
        default=None,
    )
    responseId_ext: Optional[Element] = Field(
        description="Placeholder element for responseId extensions",
        default=None,
        alias="_responseId",
    )
    sourceId: Optional[Id] = Field(
        description="Fixture Id of body for PUT and POST requests",
        default=None,
    )
    sourceId_ext: Optional[Element] = Field(
        description="Placeholder element for sourceId extensions",
        default=None,
        alias="_sourceId",
    )
    targetId: Optional[Id] = Field(
        description="Id of fixture used for extracting the [id],  [type], and [vid] for GET requests",
        default=None,
    )
    targetId_ext: Optional[Element] = Field(
        description="Placeholder element for targetId extensions",
        default=None,
        alias="_targetId",
    )
    url: Optional[String] = Field(
        description="Request URL",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )

    @field_validator(
        *(
            "url",
            "targetId",
            "sourceId",
            "responseId",
            "requestId",
            "requestHeader",
            "params",
            "origin",
            "method",
            "encodeRequestUrl",
            "destination",
            "contentType",
            "accept",
            "description",
            "label",
            "resource",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestScriptSetupActionAssert(BackboneElement):
    """
    Evaluates the results of previous operations to determine if the server under test behaves appropriately.
    """

    label: Optional[String] = Field(
        description="Tracking/logging assertion label",
        default=None,
    )
    label_ext: Optional[Element] = Field(
        description="Placeholder element for label extensions",
        default=None,
        alias="_label",
    )
    description: Optional[String] = Field(
        description="Tracking/reporting assertion description",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    direction: Optional[Code] = Field(
        description="response | request",
        default=None,
    )
    direction_ext: Optional[Element] = Field(
        description="Placeholder element for direction extensions",
        default=None,
        alias="_direction",
    )
    compareToSourceId: Optional[String] = Field(
        description="Id of the source fixture to be evaluated",
        default=None,
    )
    compareToSourceId_ext: Optional[Element] = Field(
        description="Placeholder element for compareToSourceId extensions",
        default=None,
        alias="_compareToSourceId",
    )
    compareToSourceExpression: Optional[String] = Field(
        description="The FHIRPath expression to evaluate against the source fixture",
        default=None,
    )
    compareToSourceExpression_ext: Optional[Element] = Field(
        description="Placeholder element for compareToSourceExpression extensions",
        default=None,
        alias="_compareToSourceExpression",
    )
    compareToSourcePath: Optional[String] = Field(
        description="XPath or JSONPath expression to evaluate against the source fixture",
        default=None,
    )
    compareToSourcePath_ext: Optional[Element] = Field(
        description="Placeholder element for compareToSourcePath extensions",
        default=None,
        alias="_compareToSourcePath",
    )
    contentType: Optional[Code] = Field(
        description="Mime type to compare against the \u0027Content-Type\u0027 header",
        default=None,
    )
    contentType_ext: Optional[Element] = Field(
        description="Placeholder element for contentType extensions",
        default=None,
        alias="_contentType",
    )
    expression: Optional[String] = Field(
        description="The FHIRPath expression to be evaluated",
        default=None,
    )
    expression_ext: Optional[Element] = Field(
        description="Placeholder element for expression extensions",
        default=None,
        alias="_expression",
    )
    headerField: Optional[String] = Field(
        description="HTTP header field name",
        default=None,
    )
    headerField_ext: Optional[Element] = Field(
        description="Placeholder element for headerField extensions",
        default=None,
        alias="_headerField",
    )
    minimumId: Optional[String] = Field(
        description="Fixture Id of minimum content resource",
        default=None,
    )
    minimumId_ext: Optional[Element] = Field(
        description="Placeholder element for minimumId extensions",
        default=None,
        alias="_minimumId",
    )
    navigationLinks: Optional[Boolean] = Field(
        description="Perform validation on navigation links?",
        default=None,
    )
    navigationLinks_ext: Optional[Element] = Field(
        description="Placeholder element for navigationLinks extensions",
        default=None,
        alias="_navigationLinks",
    )
    operator: Optional[Code] = Field(
        description="equals | notEquals | in | notIn | greaterThan | lessThan | empty | notEmpty | contains | notContains | eval",
        default=None,
    )
    operator_ext: Optional[Element] = Field(
        description="Placeholder element for operator extensions",
        default=None,
        alias="_operator",
    )
    path: Optional[String] = Field(
        description="XPath or JSONPath expression",
        default=None,
    )
    path_ext: Optional[Element] = Field(
        description="Placeholder element for path extensions",
        default=None,
        alias="_path",
    )
    requestMethod: Optional[Code] = Field(
        description="delete | get | options | patch | post | put | head",
        default=None,
    )
    requestMethod_ext: Optional[Element] = Field(
        description="Placeholder element for requestMethod extensions",
        default=None,
        alias="_requestMethod",
    )
    requestURL: Optional[String] = Field(
        description="Request URL comparison value",
        default=None,
    )
    requestURL_ext: Optional[Element] = Field(
        description="Placeholder element for requestURL extensions",
        default=None,
        alias="_requestURL",
    )
    resource: Optional[Code] = Field(
        description="Resource type",
        default=None,
    )
    resource_ext: Optional[Element] = Field(
        description="Placeholder element for resource extensions",
        default=None,
        alias="_resource",
    )
    response: Optional[Code] = Field(
        description="okay | created | noContent | notModified | bad | forbidden | notFound | methodNotAllowed | conflict | gone | preconditionFailed | unprocessable",
        default=None,
    )
    response_ext: Optional[Element] = Field(
        description="Placeholder element for response extensions",
        default=None,
        alias="_response",
    )
    responseCode: Optional[String] = Field(
        description="HTTP response code to test",
        default=None,
    )
    responseCode_ext: Optional[Element] = Field(
        description="Placeholder element for responseCode extensions",
        default=None,
        alias="_responseCode",
    )
    sourceId: Optional[Id] = Field(
        description="Fixture Id of source expression or headerField",
        default=None,
    )
    sourceId_ext: Optional[Element] = Field(
        description="Placeholder element for sourceId extensions",
        default=None,
        alias="_sourceId",
    )
    validateProfileId: Optional[Id] = Field(
        description="Profile Id of validation profile reference",
        default=None,
    )
    validateProfileId_ext: Optional[Element] = Field(
        description="Placeholder element for validateProfileId extensions",
        default=None,
        alias="_validateProfileId",
    )
    value: Optional[String] = Field(
        description="The value to compare to",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )
    warningOnly: Optional[Boolean] = Field(
        description="Will this assert produce a warning only on error?",
        default=None,
    )
    warningOnly_ext: Optional[Element] = Field(
        description="Placeholder element for warningOnly extensions",
        default=None,
        alias="_warningOnly",
    )

    @field_validator(
        *(
            "warningOnly",
            "value",
            "validateProfileId",
            "sourceId",
            "responseCode",
            "response",
            "resource",
            "requestURL",
            "requestMethod",
            "path",
            "operator",
            "navigationLinks",
            "minimumId",
            "headerField",
            "expression",
            "contentType",
            "compareToSourcePath",
            "compareToSourceExpression",
            "compareToSourceId",
            "direction",
            "description",
            "label",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestScriptSetupAction(BackboneElement):
    """
    Action would contain either an operation or an assertion.
    """

    operation: Optional[TestScriptSetupActionOperation] = Field(
        description="The setup operation to perform",
        default=None,
    )
    assert_: Optional[TestScriptSetupActionAssert] = Field(
        description="The assertion to perform",
        default=None,
    )

    @field_validator(
        *(
            "assert_",
            "operation",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("operation",), mode="after", check_fields=None)
    @classmethod
    def FHIR_tst_7_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="sourceId.exists() or (targetId.count() + url.count() + params.count() = 1) or (type.code in ('capabilities' |'search' | 'transaction' | 'history'))",
            human="Setup operation SHALL contain either sourceId or targetId or params or url.",
            key="tst-7",
            severity="error",
        )

    @field_validator(*("assert_",), mode="after", check_fields=None)
    @classmethod
    def FHIR_tst_5_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() or (contentType.count() + expression.count() + headerField.count() + minimumId.count() + navigationLinks.count() + path.count() + requestMethod.count() + resource.count() + responseCode.count() + response.count()  + validateProfileId.count() <=1)",
            human="Only a single assertion SHALL be present within setup action assert element.",
            key="tst-5",
            severity="error",
        )

    @field_validator(*("assert_",), mode="after", check_fields=None)
    @classmethod
    def FHIR_tst_10_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="compareToSourceId.empty() xor (compareToSourceExpression.exists() or compareToSourcePath.exists())",
            human="Setup action assert SHALL contain either compareToSourceId and compareToSourceExpression, compareToSourceId and compareToSourcePath or neither.",
            key="tst-10",
            severity="error",
        )

    @field_validator(*("assert_",), mode="after", check_fields=None)
    @classmethod
    def FHIR_tst_12_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(response.empty() and responseCode.empty() and direction = 'request') or direction.empty() or direction = 'response'",
            human="Setup action assert response and responseCode SHALL be empty when direction equals request",
            key="tst-12",
            severity="error",
        )


class TestScriptSetup(BackboneElement):
    """
    A series of required setup operations before tests are executed.
    """

    action: Optional[List[TestScriptSetupAction]] = Field(
        description="A setup operation or assert to perform",
        default=None,
    )

    @field_validator(
        *("action", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("action",), mode="after", check_fields=None)
    @classmethod
    def FHIR_tst_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="operation.exists() xor assert.exists()",
            human="Setup action SHALL contain either an operation or assert but not both.",
            key="tst-1",
            severity="error",
        )


class TestScriptSetupActionOperationRequestHeader(BackboneElement):
    """
    Header elements would be used to set HTTP headers.
    """

    field: Optional[String] = Field(
        description="HTTP header field name",
        default=None,
    )
    field_ext: Optional[Element] = Field(
        description="Placeholder element for field extensions",
        default=None,
        alias="_field",
    )
    value: Optional[String] = Field(
        description="HTTP headerfield value",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "field",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestScriptTestActionOperation(BackboneElement):
    """
    An operation would involve a REST request to a server.
    """

    type: Optional[Coding] = Field(
        description="The operation code type that will be executed",
        default=None,
    )
    resource: Optional[Code] = Field(
        description="Resource type",
        default=None,
    )
    resource_ext: Optional[Element] = Field(
        description="Placeholder element for resource extensions",
        default=None,
        alias="_resource",
    )
    label: Optional[String] = Field(
        description="Tracking/logging operation label",
        default=None,
    )
    label_ext: Optional[Element] = Field(
        description="Placeholder element for label extensions",
        default=None,
        alias="_label",
    )
    description: Optional[String] = Field(
        description="Tracking/reporting operation description",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    accept: Optional[Code] = Field(
        description="Mime type to accept in the payload of the response, with charset etc.",
        default=None,
    )
    accept_ext: Optional[Element] = Field(
        description="Placeholder element for accept extensions",
        default=None,
        alias="_accept",
    )
    contentType: Optional[Code] = Field(
        description="Mime type of the request payload contents, with charset etc.",
        default=None,
    )
    contentType_ext: Optional[Element] = Field(
        description="Placeholder element for contentType extensions",
        default=None,
        alias="_contentType",
    )
    destination: Optional[Integer] = Field(
        description="Server responding to the request",
        default=None,
    )
    destination_ext: Optional[Element] = Field(
        description="Placeholder element for destination extensions",
        default=None,
        alias="_destination",
    )
    encodeRequestUrl: Optional[Boolean] = Field(
        description="Whether or not to send the request url in encoded format",
        default=None,
    )
    encodeRequestUrl_ext: Optional[Element] = Field(
        description="Placeholder element for encodeRequestUrl extensions",
        default=None,
        alias="_encodeRequestUrl",
    )
    method: Optional[Code] = Field(
        description="delete | get | options | patch | post | put | head",
        default=None,
    )
    method_ext: Optional[Element] = Field(
        description="Placeholder element for method extensions",
        default=None,
        alias="_method",
    )
    origin: Optional[Integer] = Field(
        description="Server initiating the request",
        default=None,
    )
    origin_ext: Optional[Element] = Field(
        description="Placeholder element for origin extensions",
        default=None,
        alias="_origin",
    )
    params: Optional[String] = Field(
        description="Explicitly defined path parameters",
        default=None,
    )
    params_ext: Optional[Element] = Field(
        description="Placeholder element for params extensions",
        default=None,
        alias="_params",
    )
    requestHeader: Optional[List[TestScriptSetupActionOperationRequestHeader]] = Field(
        description="Each operation can have one or more header elements",
        default=None,
    )
    requestId: Optional[Id] = Field(
        description="Fixture Id of mapped request",
        default=None,
    )
    requestId_ext: Optional[Element] = Field(
        description="Placeholder element for requestId extensions",
        default=None,
        alias="_requestId",
    )
    responseId: Optional[Id] = Field(
        description="Fixture Id of mapped response",
        default=None,
    )
    responseId_ext: Optional[Element] = Field(
        description="Placeholder element for responseId extensions",
        default=None,
        alias="_responseId",
    )
    sourceId: Optional[Id] = Field(
        description="Fixture Id of body for PUT and POST requests",
        default=None,
    )
    sourceId_ext: Optional[Element] = Field(
        description="Placeholder element for sourceId extensions",
        default=None,
        alias="_sourceId",
    )
    targetId: Optional[Id] = Field(
        description="Id of fixture used for extracting the [id],  [type], and [vid] for GET requests",
        default=None,
    )
    targetId_ext: Optional[Element] = Field(
        description="Placeholder element for targetId extensions",
        default=None,
        alias="_targetId",
    )
    url: Optional[String] = Field(
        description="Request URL",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )

    @field_validator(
        *(
            "url",
            "targetId",
            "sourceId",
            "responseId",
            "requestId",
            "requestHeader",
            "params",
            "origin",
            "method",
            "encodeRequestUrl",
            "destination",
            "contentType",
            "accept",
            "description",
            "label",
            "resource",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestScriptTestActionAssert(BackboneElement):
    """
    Evaluates the results of previous operations to determine if the server under test behaves appropriately.
    """

    label: Optional[String] = Field(
        description="Tracking/logging assertion label",
        default=None,
    )
    label_ext: Optional[Element] = Field(
        description="Placeholder element for label extensions",
        default=None,
        alias="_label",
    )
    description: Optional[String] = Field(
        description="Tracking/reporting assertion description",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    direction: Optional[Code] = Field(
        description="response | request",
        default=None,
    )
    direction_ext: Optional[Element] = Field(
        description="Placeholder element for direction extensions",
        default=None,
        alias="_direction",
    )
    compareToSourceId: Optional[String] = Field(
        description="Id of the source fixture to be evaluated",
        default=None,
    )
    compareToSourceId_ext: Optional[Element] = Field(
        description="Placeholder element for compareToSourceId extensions",
        default=None,
        alias="_compareToSourceId",
    )
    compareToSourceExpression: Optional[String] = Field(
        description="The FHIRPath expression to evaluate against the source fixture",
        default=None,
    )
    compareToSourceExpression_ext: Optional[Element] = Field(
        description="Placeholder element for compareToSourceExpression extensions",
        default=None,
        alias="_compareToSourceExpression",
    )
    compareToSourcePath: Optional[String] = Field(
        description="XPath or JSONPath expression to evaluate against the source fixture",
        default=None,
    )
    compareToSourcePath_ext: Optional[Element] = Field(
        description="Placeholder element for compareToSourcePath extensions",
        default=None,
        alias="_compareToSourcePath",
    )
    contentType: Optional[Code] = Field(
        description="Mime type to compare against the \u0027Content-Type\u0027 header",
        default=None,
    )
    contentType_ext: Optional[Element] = Field(
        description="Placeholder element for contentType extensions",
        default=None,
        alias="_contentType",
    )
    expression: Optional[String] = Field(
        description="The FHIRPath expression to be evaluated",
        default=None,
    )
    expression_ext: Optional[Element] = Field(
        description="Placeholder element for expression extensions",
        default=None,
        alias="_expression",
    )
    headerField: Optional[String] = Field(
        description="HTTP header field name",
        default=None,
    )
    headerField_ext: Optional[Element] = Field(
        description="Placeholder element for headerField extensions",
        default=None,
        alias="_headerField",
    )
    minimumId: Optional[String] = Field(
        description="Fixture Id of minimum content resource",
        default=None,
    )
    minimumId_ext: Optional[Element] = Field(
        description="Placeholder element for minimumId extensions",
        default=None,
        alias="_minimumId",
    )
    navigationLinks: Optional[Boolean] = Field(
        description="Perform validation on navigation links?",
        default=None,
    )
    navigationLinks_ext: Optional[Element] = Field(
        description="Placeholder element for navigationLinks extensions",
        default=None,
        alias="_navigationLinks",
    )
    operator: Optional[Code] = Field(
        description="equals | notEquals | in | notIn | greaterThan | lessThan | empty | notEmpty | contains | notContains | eval",
        default=None,
    )
    operator_ext: Optional[Element] = Field(
        description="Placeholder element for operator extensions",
        default=None,
        alias="_operator",
    )
    path: Optional[String] = Field(
        description="XPath or JSONPath expression",
        default=None,
    )
    path_ext: Optional[Element] = Field(
        description="Placeholder element for path extensions",
        default=None,
        alias="_path",
    )
    requestMethod: Optional[Code] = Field(
        description="delete | get | options | patch | post | put | head",
        default=None,
    )
    requestMethod_ext: Optional[Element] = Field(
        description="Placeholder element for requestMethod extensions",
        default=None,
        alias="_requestMethod",
    )
    requestURL: Optional[String] = Field(
        description="Request URL comparison value",
        default=None,
    )
    requestURL_ext: Optional[Element] = Field(
        description="Placeholder element for requestURL extensions",
        default=None,
        alias="_requestURL",
    )
    resource: Optional[Code] = Field(
        description="Resource type",
        default=None,
    )
    resource_ext: Optional[Element] = Field(
        description="Placeholder element for resource extensions",
        default=None,
        alias="_resource",
    )
    response: Optional[Code] = Field(
        description="okay | created | noContent | notModified | bad | forbidden | notFound | methodNotAllowed | conflict | gone | preconditionFailed | unprocessable",
        default=None,
    )
    response_ext: Optional[Element] = Field(
        description="Placeholder element for response extensions",
        default=None,
        alias="_response",
    )
    responseCode: Optional[String] = Field(
        description="HTTP response code to test",
        default=None,
    )
    responseCode_ext: Optional[Element] = Field(
        description="Placeholder element for responseCode extensions",
        default=None,
        alias="_responseCode",
    )
    sourceId: Optional[Id] = Field(
        description="Fixture Id of source expression or headerField",
        default=None,
    )
    sourceId_ext: Optional[Element] = Field(
        description="Placeholder element for sourceId extensions",
        default=None,
        alias="_sourceId",
    )
    validateProfileId: Optional[Id] = Field(
        description="Profile Id of validation profile reference",
        default=None,
    )
    validateProfileId_ext: Optional[Element] = Field(
        description="Placeholder element for validateProfileId extensions",
        default=None,
        alias="_validateProfileId",
    )
    value: Optional[String] = Field(
        description="The value to compare to",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )
    warningOnly: Optional[Boolean] = Field(
        description="Will this assert produce a warning only on error?",
        default=None,
    )
    warningOnly_ext: Optional[Element] = Field(
        description="Placeholder element for warningOnly extensions",
        default=None,
        alias="_warningOnly",
    )

    @field_validator(
        *(
            "warningOnly",
            "value",
            "validateProfileId",
            "sourceId",
            "responseCode",
            "response",
            "resource",
            "requestURL",
            "requestMethod",
            "path",
            "operator",
            "navigationLinks",
            "minimumId",
            "headerField",
            "expression",
            "contentType",
            "compareToSourcePath",
            "compareToSourceExpression",
            "compareToSourceId",
            "direction",
            "description",
            "label",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestScriptTestAction(BackboneElement):
    """
    Action would contain either an operation or an assertion.
    """

    operation: Optional[TestScriptTestActionOperation] = Field(
        description="The setup operation to perform",
        default=None,
    )
    assert_: Optional[TestScriptTestActionAssert] = Field(
        description="The setup assertion to perform",
        default=None,
    )

    @field_validator(
        *(
            "assert_",
            "operation",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("operation",), mode="after", check_fields=None)
    @classmethod
    def FHIR_tst_8_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="sourceId.exists() or (targetId.count() + url.count() + params.count() = 1) or (type.code in ('capabilities' | 'search' | 'transaction' | 'history'))",
            human="Test operation SHALL contain either sourceId or targetId or params or url.",
            key="tst-8",
            severity="error",
        )

    @field_validator(*("assert_",), mode="after", check_fields=None)
    @classmethod
    def FHIR_tst_6_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() or (contentType.count() + expression.count() + headerField.count() + minimumId.count() + navigationLinks.count() + path.count() + requestMethod.count() + resource.count() + responseCode.count() + response.count() + validateProfileId.count() <=1)",
            human="Only a single assertion SHALL be present within test action assert element.",
            key="tst-6",
            severity="error",
        )

    @field_validator(*("assert_",), mode="after", check_fields=None)
    @classmethod
    def FHIR_tst_11_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="compareToSourceId.empty() xor (compareToSourceExpression.exists() or compareToSourcePath.exists())",
            human="Test action assert SHALL contain either compareToSourceId and compareToSourceExpression, compareToSourceId and compareToSourcePath or neither.",
            key="tst-11",
            severity="error",
        )

    @field_validator(*("assert_",), mode="after", check_fields=None)
    @classmethod
    def FHIR_tst_13_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(response.empty() and responseCode.empty() and direction = 'request') or direction.empty() or direction = 'response'",
            human="Test action assert response and response and responseCode SHALL be empty when direction equals request",
            key="tst-13",
            severity="error",
        )


class TestScriptTest(BackboneElement):
    """
    A test in this script.
    """

    name: Optional[String] = Field(
        description="Tracking/logging name of this test",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    description: Optional[String] = Field(
        description="Tracking/reporting short description of the test",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    action: Optional[List[TestScriptTestAction]] = Field(
        description="A test operation or assert to perform",
        default=None,
    )

    @field_validator(
        *(
            "action",
            "description",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("action",), mode="after", check_fields=None)
    @classmethod
    def FHIR_tst_2_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="operation.exists() xor assert.exists()",
            human="Test action SHALL contain either an operation or assert but not both.",
            key="tst-2",
            severity="error",
        )


class TestScriptTeardownAction(BackboneElement):
    """
    The teardown action will only contain an operation.
    """

    operation: Optional[TestScriptSetupActionOperation] = Field(
        description="The teardown operation to perform",
        default=None,
    )

    @field_validator(
        *("operation", "modifierExtension", "extension"),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("operation",), mode="after", check_fields=None)
    @classmethod
    def FHIR_tst_9_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="sourceId.exists() or (targetId.count() + url.count() + params.count() = 1) or (type.code in ('capabilities' | 'search' | 'transaction' | 'history'))",
            human="Teardown operation SHALL contain either sourceId or targetId or params or url.",
            key="tst-9",
            severity="error",
        )


class TestScriptTeardown(BackboneElement):
    """
    A series of operations required to clean up after all the tests are executed (successfully or otherwise).
    """

    action: Optional[List[TestScriptTeardownAction]] = Field(
        description="One or more teardown operations to perform",
        default=None,
    )

    @field_validator(
        *("action", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestScript(DomainResource):
    """
    A structured set of tests against a FHIR server or client implementation to determine compliance against the FHIR specification.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/TestScript"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this test script, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[Identifier] = Field(
        description="Additional identifier for the test script",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the test script",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    name: Optional[String] = Field(
        description="Name for this test script (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this test script (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the test script",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for test script (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this test script is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    origin: Optional[List[TestScriptOrigin]] = Field(
        description="An abstract server representing a client or sender in a message exchange",
        default=None,
    )
    destination: Optional[List[TestScriptDestination]] = Field(
        description="An abstract server representing a destination or receiver in a message exchange",
        default=None,
    )
    metadata: Optional[TestScriptMetadata] = Field(
        description="Required capability that is assumed to function correctly on the FHIR server being tested",
        default=None,
    )
    fixture: Optional[List[TestScriptFixture]] = Field(
        description="Fixture in the test script - by reference (uri)",
        default=None,
    )
    profile: Optional[List[Reference]] = Field(
        description="Reference of the validation profile",
        default=None,
    )
    variable: Optional[List[TestScriptVariable]] = Field(
        description="Placeholder for evaluated elements",
        default=None,
    )
    setup: Optional[TestScriptSetup] = Field(
        description="A series of required setup operations before tests are executed",
        default=None,
    )
    test: Optional[List[TestScriptTest]] = Field(
        description="A test in this script",
        default=None,
    )
    teardown: Optional[TestScriptTeardown] = Field(
        description="A series of required clean up steps",
        default=None,
    )
    resourceType: Literal["TestScript"] = Field(
        description=None,
        default="TestScript",
    )

    @field_validator(
        *(
            "teardown",
            "test",
            "setup",
            "variable",
            "profile",
            "fixture",
            "metadata",
            "destination",
            "origin",
            "copyright",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "experimental",
            "status",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("metadata",), mode="after", check_fields=None)
    @classmethod
    def FHIR_tst_4_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="capability.required.exists() or capability.validated.exists()",
            human="TestScript metadata capability SHALL contain required or validated or both.",
            key="tst-4",
            severity="error",
        )

    @field_validator(*("variable",), mode="after", check_fields=None)
    @classmethod
    def FHIR_tst_3_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="expression.empty() or headerField.empty() or path.empty()",
            human="Variable can only contain one of expression, headerField or path.",
            key="tst-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_tst_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.matches('[A-Z]([A-Za-z0-9_]){0,254}')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="tst-0",
            severity="warning",
        )
