# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    CodeableConcept,
    Identifier,
    BackboneElement,
    DomainResource,
)


class SubstanceSourceMaterialFractionDescription(BackboneElement):
    """
    Many complex materials are fractions of parts of plants, animals, or minerals. Fraction elements are often necessary to define both Substances and Specified Group 1 Substances. For substances derived from Plants, fraction information will be captured at the Substance information level ( . Oils, Juices and Exudates). Additional information for Extracts, such as extraction solvent composition, will be captured at the Specified Substance Group 1 information level. For plasma-derived products fraction information will be captured at the Substance and the Specified Substance Group 1 levels.
    """

    fraction: Optional[String] = Field(
        description="This element is capturing information about the fraction of a plant part, or human plasma for fractionation",
        default=None,
    )
    fraction_ext: Optional[Element] = Field(
        description="Placeholder element for fraction extensions",
        default=None,
        alias="_fraction",
    )
    materialType: Optional[CodeableConcept] = Field(
        description="The specific type of the material constituting the component. For Herbal preparations the particulars of the extracts (liquid/dry) is described in Specified Substance Group 1",
        default=None,
    )

    @field_validator(
        *(
            "materialType",
            "fraction",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubstanceSourceMaterialOrganismAuthor(BackboneElement):
    """
    4.9.13.6.1 Author type (Conditional).
    """

    authorType: Optional[CodeableConcept] = Field(
        description="The type of author of an organism species shall be specified. The parenthetical author of an organism species refers to the first author who published the plant/animal name (of any rank). The primary author of an organism species refers to the first author(s), who validly published the plant/animal name",
        default=None,
    )
    authorDescription: Optional[String] = Field(
        description="The author of an organism species shall be specified. The author year of an organism shall also be specified when applicable; refers to the year in which the first author(s) published the infraspecific plant/animal name (of any rank)",
        default=None,
    )
    authorDescription_ext: Optional[Element] = Field(
        description="Placeholder element for authorDescription extensions",
        default=None,
        alias="_authorDescription",
    )

    @field_validator(
        *(
            "authorDescription",
            "authorType",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubstanceSourceMaterialOrganismHybrid(BackboneElement):
    """
    4.9.13.8.1 Hybrid species maternal organism ID (Optional).
    """

    maternalOrganismId: Optional[String] = Field(
        description="The identifier of the maternal species constituting the hybrid organism shall be specified based on a controlled vocabulary. For plants, the parents aren\u2019t always known, and it is unlikely that it will be known which is maternal and which is paternal",
        default=None,
    )
    maternalOrganismId_ext: Optional[Element] = Field(
        description="Placeholder element for maternalOrganismId extensions",
        default=None,
        alias="_maternalOrganismId",
    )
    maternalOrganismName: Optional[String] = Field(
        description="The name of the maternal species constituting the hybrid organism shall be specified. For plants, the parents aren\u2019t always known, and it is unlikely that it will be known which is maternal and which is paternal",
        default=None,
    )
    maternalOrganismName_ext: Optional[Element] = Field(
        description="Placeholder element for maternalOrganismName extensions",
        default=None,
        alias="_maternalOrganismName",
    )
    paternalOrganismId: Optional[String] = Field(
        description="The identifier of the paternal species constituting the hybrid organism shall be specified based on a controlled vocabulary",
        default=None,
    )
    paternalOrganismId_ext: Optional[Element] = Field(
        description="Placeholder element for paternalOrganismId extensions",
        default=None,
        alias="_paternalOrganismId",
    )
    paternalOrganismName: Optional[String] = Field(
        description="The name of the paternal species constituting the hybrid organism shall be specified",
        default=None,
    )
    paternalOrganismName_ext: Optional[Element] = Field(
        description="Placeholder element for paternalOrganismName extensions",
        default=None,
        alias="_paternalOrganismName",
    )
    hybridType: Optional[CodeableConcept] = Field(
        description="The hybrid type of an organism shall be specified",
        default=None,
    )

    @field_validator(
        *(
            "hybridType",
            "paternalOrganismName",
            "paternalOrganismId",
            "maternalOrganismName",
            "maternalOrganismId",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubstanceSourceMaterialOrganismOrganismGeneral(BackboneElement):
    """
    4.9.13.7.1 Kingdom (Conditional).
    """

    kingdom: Optional[CodeableConcept] = Field(
        description="The kingdom of an organism shall be specified",
        default=None,
    )
    phylum: Optional[CodeableConcept] = Field(
        description="The phylum of an organism shall be specified",
        default=None,
    )
    class_: Optional[CodeableConcept] = Field(
        description="The class of an organism shall be specified",
        default=None,
    )
    order: Optional[CodeableConcept] = Field(
        description="The order of an organism shall be specified,",
        default=None,
    )

    @field_validator(
        *(
            "order",
            "class_",
            "phylum",
            "kingdom",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubstanceSourceMaterialOrganism(BackboneElement):
    """
    This subclause describes the organism which the substance is derived from. For vaccines, the parent organism shall be specified based on these subclause elements. As an example, full taxonomy will be described for the Substance Name: ., Leaf.
    """

    family: Optional[CodeableConcept] = Field(
        description="The family of an organism shall be specified",
        default=None,
    )
    genus: Optional[CodeableConcept] = Field(
        description="The genus of an organism shall be specified; refers to the Latin epithet of the genus element of the plant/animal scientific name; it is present in names for genera, species and infraspecies",
        default=None,
    )
    species: Optional[CodeableConcept] = Field(
        description="The species of an organism shall be specified; refers to the Latin epithet of the species of the plant/animal; it is present in names for species and infraspecies",
        default=None,
    )
    intraspecificType: Optional[CodeableConcept] = Field(
        description="The Intraspecific type of an organism shall be specified",
        default=None,
    )
    intraspecificDescription: Optional[String] = Field(
        description="The intraspecific description of an organism shall be specified based on a controlled vocabulary. For Influenza Vaccine, the intraspecific description shall contain the syntax of the antigen in line with the WHO convention",
        default=None,
    )
    intraspecificDescription_ext: Optional[Element] = Field(
        description="Placeholder element for intraspecificDescription extensions",
        default=None,
        alias="_intraspecificDescription",
    )
    author: Optional[List[SubstanceSourceMaterialOrganismAuthor]] = Field(
        description="4.9.13.6.1 Author type (Conditional)",
        default=None,
    )
    hybrid: Optional[SubstanceSourceMaterialOrganismHybrid] = Field(
        description="4.9.13.8.1 Hybrid species maternal organism ID (Optional)",
        default=None,
    )
    organismGeneral: Optional[SubstanceSourceMaterialOrganismOrganismGeneral] = Field(
        description="4.9.13.7.1 Kingdom (Conditional)",
        default=None,
    )

    @field_validator(
        *(
            "organismGeneral",
            "hybrid",
            "author",
            "intraspecificDescription",
            "intraspecificType",
            "species",
            "genus",
            "family",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubstanceSourceMaterialPartDescription(BackboneElement):
    """
    To do.
    """

    part: Optional[CodeableConcept] = Field(
        description="Entity of anatomical origin of source material within an organism",
        default=None,
    )
    partLocation: Optional[CodeableConcept] = Field(
        description="The detailed anatomic location when the part can be extracted from different anatomical locations of the organism. Multiple alternative locations may apply",
        default=None,
    )

    @field_validator(
        *(
            "partLocation",
            "part",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubstanceSourceMaterial(DomainResource):
    """
    Source material shall capture information on the taxonomic and anatomical origins as well as the fraction of a material that can result in or can be modified to form a substance. This set of data elements shall be used to define polymer substances isolated from biological matrices. Taxonomic and anatomical origins shall be described using a controlled vocabulary as required. This information is captured for naturally derived polymers ( . starch) and structurally diverse substances. For Organisms belonging to the Kingdom Plantae the Substance level defines the fresh material of a single species or infraspecies, the Herbal Drug and the Herbal preparation. For Herbal preparations, the fraction information will be captured at the Substance information level and additional information for herbal extracts will be captured at the Specified Substance Group 1 information level. See for further explanation the Substance Class: Structurally Diverse and the herbal annex.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/SubstanceSourceMaterial"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    sourceMaterialClass: Optional[CodeableConcept] = Field(
        description="General high level classification of the source material specific to the origin of the material",
        default=None,
    )
    sourceMaterialType: Optional[CodeableConcept] = Field(
        description="The type of the source material shall be specified based on a controlled vocabulary. For vaccines, this subclause refers to the class of infectious agent",
        default=None,
    )
    sourceMaterialState: Optional[CodeableConcept] = Field(
        description="The state of the source material when extracted",
        default=None,
    )
    organismId: Optional[Identifier] = Field(
        description="The unique identifier associated with the source material parent organism shall be specified",
        default=None,
    )
    organismName: Optional[String] = Field(
        description="The organism accepted Scientific name shall be provided based on the organism taxonomy",
        default=None,
    )
    organismName_ext: Optional[Element] = Field(
        description="Placeholder element for organismName extensions",
        default=None,
        alias="_organismName",
    )
    parentSubstanceId: Optional[List[Identifier]] = Field(
        description="The parent of the herbal drug Ginkgo biloba, Leaf is the substance ID of the substance (fresh) of Ginkgo biloba L. or Ginkgo biloba L. (Whole plant)",
        default=None,
    )
    parentSubstanceName: Optional[List[String]] = Field(
        description="The parent substance of the Herbal Drug, or Herbal preparation",
        default=None,
    )
    parentSubstanceName_ext: Optional[Element] = Field(
        description="Placeholder element for parentSubstanceName extensions",
        default=None,
        alias="_parentSubstanceName",
    )
    countryOfOrigin: Optional[List[CodeableConcept]] = Field(
        description="The country where the plant material is harvested or the countries where the plasma is sourced from as laid down in accordance with the Plasma Master File. For \u201cPlasma-derived substances\u201d the attribute country of origin provides information about the countries used for the manufacturing of the Cryopoor plama or Crioprecipitate",
        default=None,
    )
    geographicalLocation: Optional[List[String]] = Field(
        description="The place/region where the plant is harvested or the places/regions where the animal source material has its habitat",
        default=None,
    )
    geographicalLocation_ext: Optional[Element] = Field(
        description="Placeholder element for geographicalLocation extensions",
        default=None,
        alias="_geographicalLocation",
    )
    developmentStage: Optional[CodeableConcept] = Field(
        description="Stage of life for animals, plants, insects and microorganisms. This information shall be provided only when the substance is significantly different in these stages (e.g. foetal bovine serum)",
        default=None,
    )
    fractionDescription: Optional[List[SubstanceSourceMaterialFractionDescription]] = (
        Field(
            description="Many complex materials are fractions of parts of plants, animals, or minerals. Fraction elements are often necessary to define both Substances and Specified Group 1 Substances. For substances derived from Plants, fraction information will be captured at the Substance information level ( . Oils, Juices and Exudates). Additional information for Extracts, such as extraction solvent composition, will be captured at the Specified Substance Group 1 information level. For plasma-derived products fraction information will be captured at the Substance and the Specified Substance Group 1 levels",
            default=None,
        )
    )
    organism: Optional[SubstanceSourceMaterialOrganism] = Field(
        description="This subclause describes the organism which the substance is derived from. For vaccines, the parent organism shall be specified based on these subclause elements. As an example, full taxonomy will be described for the Substance Name: ., Leaf",
        default=None,
    )
    partDescription: Optional[List[SubstanceSourceMaterialPartDescription]] = Field(
        description="To do",
        default=None,
    )
    resourceType: Literal["SubstanceSourceMaterial"] = Field(
        description=None,
        default="SubstanceSourceMaterial",
    )

    @field_validator(
        *(
            "partDescription",
            "organism",
            "fractionDescription",
            "developmentStage",
            "geographicalLocation",
            "countryOfOrigin",
            "parentSubstanceName",
            "parentSubstanceId",
            "organismName",
            "organismId",
            "sourceMaterialState",
            "sourceMaterialType",
            "sourceMaterialClass",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
