# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    DateTime,
    Markdown,
    Canonical,
    Id,
    Integer,
    Base64Binary,
    Date,
    Decimal,
    Instant,
    Oid,
    PositiveInt,
    Time,
    UnsignedInt,
    Url,
    Uuid,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    ContactDetail,
    UsageContext,
    CodeableConcept,
    BackboneElement,
    Address,
    Age,
    Annotation,
    Attachment,
    Coding,
    ContactPoint,
    Count,
    Distance,
    Duration,
    HumanName,
    Money,
    Period,
    Quantity,
    Range,
    Ratio,
    Reference,
    SampledData,
    Signature,
    Timing,
    Contributor,
    DataRequirement,
    Expression,
    ParameterDefinition,
    RelatedArtifact,
    TriggerDefinition,
    Dosage,
    DomainResource,
)


class StructureMapStructure(BackboneElement):
    """
    A structure definition used by this map. The structure definition may describe instances that are converted, or the instances that are produced.
    """

    url: Optional[Canonical] = Field(
        description="Canonical reference to structure definition",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    mode: Optional[Code] = Field(
        description="source | queried | target | produced",
        default=None,
    )
    mode_ext: Optional[Element] = Field(
        description="Placeholder element for mode extensions",
        default=None,
        alias="_mode",
    )
    alias: Optional[String] = Field(
        description="Name for type in this map",
        default=None,
    )
    alias_ext: Optional[Element] = Field(
        description="Placeholder element for alias extensions",
        default=None,
        alias="_alias",
    )
    documentation: Optional[String] = Field(
        description="Documentation on use of structure",
        default=None,
    )
    documentation_ext: Optional[Element] = Field(
        description="Placeholder element for documentation extensions",
        default=None,
        alias="_documentation",
    )

    @field_validator(
        *(
            "documentation",
            "alias",
            "mode",
            "url",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class StructureMapGroupInput(BackboneElement):
    """
    A name assigned to an instance of data. The instance must be provided when the mapping is invoked.
    """

    name: Optional[Id] = Field(
        description="Name for this instance of data",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    type: Optional[String] = Field(
        description="Type for this instance of data",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    mode: Optional[Code] = Field(
        description="source | target",
        default=None,
    )
    mode_ext: Optional[Element] = Field(
        description="Placeholder element for mode extensions",
        default=None,
        alias="_mode",
    )
    documentation: Optional[String] = Field(
        description="Documentation for this instance of data",
        default=None,
    )
    documentation_ext: Optional[Element] = Field(
        description="Placeholder element for documentation extensions",
        default=None,
        alias="_documentation",
    )

    @field_validator(
        *(
            "documentation",
            "mode",
            "type",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class StructureMapGroupRuleSource(BackboneElement):
    """
    Source inputs to the mapping.
    """

    context: Optional[Id] = Field(
        description="Type or variable this rule applies to",
        default=None,
    )
    context_ext: Optional[Element] = Field(
        description="Placeholder element for context extensions",
        default=None,
        alias="_context",
    )
    min: Optional[Integer] = Field(
        description="Specified minimum cardinality",
        default=None,
    )
    min_ext: Optional[Element] = Field(
        description="Placeholder element for min extensions",
        default=None,
        alias="_min",
    )
    max: Optional[String] = Field(
        description="Specified maximum cardinality (number or *)",
        default=None,
    )
    max_ext: Optional[Element] = Field(
        description="Placeholder element for max extensions",
        default=None,
        alias="_max",
    )
    type: Optional[String] = Field(
        description="Rule only applies if source has this type",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    defaultValueBase64Binary: Optional[Base64Binary] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueBase64Binary_ext: Optional[Element] = Field(
        description="Placeholder element for defaultValueBase64Binary extensions",
        default=None,
        alias="_defaultValueBase64Binary",
    )
    defaultValueBoolean: Optional[Boolean] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for defaultValueBoolean extensions",
        default=None,
        alias="_defaultValueBoolean",
    )
    defaultValueCanonical: Optional[Canonical] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for defaultValueCanonical extensions",
        default=None,
        alias="_defaultValueCanonical",
    )
    defaultValueCode: Optional[Code] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueCode_ext: Optional[Element] = Field(
        description="Placeholder element for defaultValueCode extensions",
        default=None,
        alias="_defaultValueCode",
    )
    defaultValueDate: Optional[Date] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueDate_ext: Optional[Element] = Field(
        description="Placeholder element for defaultValueDate extensions",
        default=None,
        alias="_defaultValueDate",
    )
    defaultValueDateTime: Optional[DateTime] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for defaultValueDateTime extensions",
        default=None,
        alias="_defaultValueDateTime",
    )
    defaultValueDecimal: Optional[Decimal] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueDecimal_ext: Optional[Element] = Field(
        description="Placeholder element for defaultValueDecimal extensions",
        default=None,
        alias="_defaultValueDecimal",
    )
    defaultValueId: Optional[Id] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueId_ext: Optional[Element] = Field(
        description="Placeholder element for defaultValueId extensions",
        default=None,
        alias="_defaultValueId",
    )
    defaultValueInstant: Optional[Instant] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueInstant_ext: Optional[Element] = Field(
        description="Placeholder element for defaultValueInstant extensions",
        default=None,
        alias="_defaultValueInstant",
    )
    defaultValueInteger: Optional[Integer] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueInteger_ext: Optional[Element] = Field(
        description="Placeholder element for defaultValueInteger extensions",
        default=None,
        alias="_defaultValueInteger",
    )
    defaultValueMarkdown: Optional[Markdown] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueMarkdown_ext: Optional[Element] = Field(
        description="Placeholder element for defaultValueMarkdown extensions",
        default=None,
        alias="_defaultValueMarkdown",
    )
    defaultValueOid: Optional[Oid] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueOid_ext: Optional[Element] = Field(
        description="Placeholder element for defaultValueOid extensions",
        default=None,
        alias="_defaultValueOid",
    )
    defaultValuePositiveInt: Optional[PositiveInt] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValuePositiveInt_ext: Optional[Element] = Field(
        description="Placeholder element for defaultValuePositiveInt extensions",
        default=None,
        alias="_defaultValuePositiveInt",
    )
    defaultValueString: Optional[String] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueString_ext: Optional[Element] = Field(
        description="Placeholder element for defaultValueString extensions",
        default=None,
        alias="_defaultValueString",
    )
    defaultValueTime: Optional[Time] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueTime_ext: Optional[Element] = Field(
        description="Placeholder element for defaultValueTime extensions",
        default=None,
        alias="_defaultValueTime",
    )
    defaultValueUnsignedInt: Optional[UnsignedInt] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueUnsignedInt_ext: Optional[Element] = Field(
        description="Placeholder element for defaultValueUnsignedInt extensions",
        default=None,
        alias="_defaultValueUnsignedInt",
    )
    defaultValueUri: Optional[Uri] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueUri_ext: Optional[Element] = Field(
        description="Placeholder element for defaultValueUri extensions",
        default=None,
        alias="_defaultValueUri",
    )
    defaultValueUrl: Optional[Url] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueUrl_ext: Optional[Element] = Field(
        description="Placeholder element for defaultValueUrl extensions",
        default=None,
        alias="_defaultValueUrl",
    )
    defaultValueUuid: Optional[Uuid] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueUuid_ext: Optional[Element] = Field(
        description="Placeholder element for defaultValueUuid extensions",
        default=None,
        alias="_defaultValueUuid",
    )
    defaultValueAddress: Optional[Address] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueAge: Optional[Age] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueAnnotation: Optional[Annotation] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueAttachment: Optional[Attachment] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueCodeableConcept: Optional[CodeableConcept] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueCoding: Optional[Coding] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueContactPoint: Optional[ContactPoint] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueCount: Optional[Count] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueDistance: Optional[Distance] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueDuration: Optional[Duration] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueHumanName: Optional[HumanName] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueIdentifier: Optional[Identifier] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueMoney: Optional[Money] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValuePeriod: Optional[Period] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueQuantity: Optional[Quantity] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueRange: Optional[Range] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueRatio: Optional[Ratio] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueReference: Optional[Reference] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueSampledData: Optional[SampledData] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueSignature: Optional[Signature] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueTiming: Optional[Timing] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueContactDetail: Optional[ContactDetail] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueContributor: Optional[Contributor] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueDataRequirement: Optional[DataRequirement] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueExpression: Optional[Expression] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueParameterDefinition: Optional[ParameterDefinition] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueRelatedArtifact: Optional[RelatedArtifact] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueTriggerDefinition: Optional[TriggerDefinition] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueUsageContext: Optional[UsageContext] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueDosage: Optional[Dosage] = Field(
        description="Default value if no value exists",
        default=None,
    )
    defaultValueMeta: Optional[Meta] = Field(
        description="Default value if no value exists",
        default=None,
    )
    element: Optional[String] = Field(
        description="Optional field for this source",
        default=None,
    )
    element_ext: Optional[Element] = Field(
        description="Placeholder element for element extensions",
        default=None,
        alias="_element",
    )
    listMode: Optional[Code] = Field(
        description="first | not_first | last | not_last | only_one",
        default=None,
    )
    listMode_ext: Optional[Element] = Field(
        description="Placeholder element for listMode extensions",
        default=None,
        alias="_listMode",
    )
    variable: Optional[Id] = Field(
        description="Named context for field, if a field is specified",
        default=None,
    )
    variable_ext: Optional[Element] = Field(
        description="Placeholder element for variable extensions",
        default=None,
        alias="_variable",
    )
    condition: Optional[String] = Field(
        description="FHIRPath expression  - must be true or the rule does not apply",
        default=None,
    )
    condition_ext: Optional[Element] = Field(
        description="Placeholder element for condition extensions",
        default=None,
        alias="_condition",
    )
    check: Optional[String] = Field(
        description="FHIRPath expression  - must be true or the mapping engine throws an error instead of completing",
        default=None,
    )
    check_ext: Optional[Element] = Field(
        description="Placeholder element for check extensions",
        default=None,
        alias="_check",
    )
    logMessage: Optional[String] = Field(
        description="Message to put in log if source exists (FHIRPath)",
        default=None,
    )
    logMessage_ext: Optional[Element] = Field(
        description="Placeholder element for logMessage extensions",
        default=None,
        alias="_logMessage",
    )

    @property
    def defaultValue(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="defaultValue",
        )

    @field_validator(
        *(
            "logMessage",
            "check",
            "condition",
            "variable",
            "listMode",
            "element",
            "type",
            "max",
            "min",
            "context",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def defaultValue_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[
                Base64Binary,
                Boolean,
                Canonical,
                Code,
                Date,
                DateTime,
                Decimal,
                Id,
                Instant,
                Integer,
                Markdown,
                Oid,
                PositiveInt,
                String,
                Time,
                UnsignedInt,
                Uri,
                Url,
                Uuid,
                Address,
                Age,
                Annotation,
                Attachment,
                CodeableConcept,
                Coding,
                ContactPoint,
                Count,
                Distance,
                Duration,
                HumanName,
                Identifier,
                Money,
                Period,
                Quantity,
                Range,
                Ratio,
                Reference,
                SampledData,
                Signature,
                Timing,
                ContactDetail,
                Contributor,
                DataRequirement,
                Expression,
                ParameterDefinition,
                RelatedArtifact,
                TriggerDefinition,
                UsageContext,
                Dosage,
                Meta,
            ],
            field_name_base="defaultValue",
            required=False,
        )


class StructureMapGroupRuleTargetParameter(BackboneElement):
    """
    Parameters to the transform.
    """

    valueId: Optional[Id] = Field(
        description="Parameter value - variable or literal",
        default=None,
    )
    valueId_ext: Optional[Element] = Field(
        description="Placeholder element for valueId extensions",
        default=None,
        alias="_valueId",
    )
    valueString: Optional[String] = Field(
        description="Parameter value - variable or literal",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueBoolean: Optional[Boolean] = Field(
        description="Parameter value - variable or literal",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueInteger: Optional[Integer] = Field(
        description="Parameter value - variable or literal",
        default=None,
    )
    valueInteger_ext: Optional[Element] = Field(
        description="Placeholder element for valueInteger extensions",
        default=None,
        alias="_valueInteger",
    )
    valueDecimal: Optional[Decimal] = Field(
        description="Parameter value - variable or literal",
        default=None,
    )
    valueDecimal_ext: Optional[Element] = Field(
        description="Placeholder element for valueDecimal extensions",
        default=None,
        alias="_valueDecimal",
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Id, String, Boolean, Integer, Decimal],
            field_name_base="value",
            required=True,
        )


class StructureMapGroupRuleTarget(BackboneElement):
    """
    Content to create because of this mapping rule.
    """

    context: Optional[Id] = Field(
        description="Type or variable this rule applies to",
        default=None,
    )
    context_ext: Optional[Element] = Field(
        description="Placeholder element for context extensions",
        default=None,
        alias="_context",
    )
    contextType: Optional[Code] = Field(
        description="type | variable",
        default=None,
    )
    contextType_ext: Optional[Element] = Field(
        description="Placeholder element for contextType extensions",
        default=None,
        alias="_contextType",
    )
    element: Optional[String] = Field(
        description="Field to create in the context",
        default=None,
    )
    element_ext: Optional[Element] = Field(
        description="Placeholder element for element extensions",
        default=None,
        alias="_element",
    )
    variable: Optional[Id] = Field(
        description="Named context for field, if desired, and a field is specified",
        default=None,
    )
    variable_ext: Optional[Element] = Field(
        description="Placeholder element for variable extensions",
        default=None,
        alias="_variable",
    )
    listMode: Optional[List[Code]] = Field(
        description="first | share | last | collate",
        default=None,
    )
    listMode_ext: Optional[Element] = Field(
        description="Placeholder element for listMode extensions",
        default=None,
        alias="_listMode",
    )
    listRuleId: Optional[Id] = Field(
        description="Internal rule reference for shared list items",
        default=None,
    )
    listRuleId_ext: Optional[Element] = Field(
        description="Placeholder element for listRuleId extensions",
        default=None,
        alias="_listRuleId",
    )
    transform: Optional[Code] = Field(
        description="create | copy +",
        default=None,
    )
    transform_ext: Optional[Element] = Field(
        description="Placeholder element for transform extensions",
        default=None,
        alias="_transform",
    )
    parameter: Optional[List[StructureMapGroupRuleTargetParameter]] = Field(
        description="Parameters to the transform",
        default=None,
    )

    @field_validator(
        *(
            "parameter",
            "transform",
            "listRuleId",
            "listMode",
            "variable",
            "element",
            "contextType",
            "context",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class StructureMapGroupRuleDependent(BackboneElement):
    """
    Which other rules to apply in the context of this rule.
    """

    name: Optional[Id] = Field(
        description="Name of a rule or group to apply",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    variable: Optional[List[String]] = Field(
        description="Variable to pass to the rule or group",
        default=None,
    )
    variable_ext: Optional[Element] = Field(
        description="Placeholder element for variable extensions",
        default=None,
        alias="_variable",
    )

    @field_validator(
        *(
            "variable",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class StructureMapGroupRule(BackboneElement):
    """
    Transform Rule from source to target.
    """

    name: Optional[Id] = Field(
        description="Name of the rule for internal references",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    source: Optional[List[StructureMapGroupRuleSource]] = Field(
        description="Source inputs to the mapping",
        default=None,
    )
    target: Optional[List[StructureMapGroupRuleTarget]] = Field(
        description="Content to create because of this mapping rule",
        default=None,
    )
    rule: Optional[List["StructureMapGroupRule"]] = Field(
        description="Rules contained in this rule",
        default=None,
    )
    dependent: Optional[List[StructureMapGroupRuleDependent]] = Field(
        description="Which other rules to apply in the context of this rule",
        default=None,
    )
    documentation: Optional[String] = Field(
        description="Documentation for this instance of data",
        default=None,
    )
    documentation_ext: Optional[Element] = Field(
        description="Placeholder element for documentation extensions",
        default=None,
        alias="_documentation",
    )

    @field_validator(
        *(
            "documentation",
            "dependent",
            "rule",
            "target",
            "source",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("target",), mode="after", check_fields=None)
    @classmethod
    def FHIR_smp_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="element.exists() implies context.exists()",
            human="Can only have an element if you have a context",
            key="smp-1",
            severity="error",
        )

    @field_validator(*("target",), mode="after", check_fields=None)
    @classmethod
    def FHIR_smp_2_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="context.exists() implies contextType.exists()",
            human="Must have a contextType if you have a context",
            key="smp-2",
            severity="error",
        )


class StructureMapGroup(BackboneElement):
    """
    Organizes the mapping into manageable chunks for human review/ease of maintenance.
    """

    name: Optional[Id] = Field(
        description="Human-readable label",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    extends: Optional[Id] = Field(
        description="Another group that this group adds rules to",
        default=None,
    )
    extends_ext: Optional[Element] = Field(
        description="Placeholder element for extends extensions",
        default=None,
        alias="_extends",
    )
    typeMode: Optional[Code] = Field(
        description="none | types | type-and-types",
        default=None,
    )
    typeMode_ext: Optional[Element] = Field(
        description="Placeholder element for typeMode extensions",
        default=None,
        alias="_typeMode",
    )
    documentation: Optional[String] = Field(
        description="Additional description/explanation for group",
        default=None,
    )
    documentation_ext: Optional[Element] = Field(
        description="Placeholder element for documentation extensions",
        default=None,
        alias="_documentation",
    )
    input: Optional[List[StructureMapGroupInput]] = Field(
        description="Named instance provided when invoking the map",
        default=None,
    )
    rule: Optional[List[StructureMapGroupRule]] = Field(
        description="Transform Rule from source to target",
        default=None,
    )

    @field_validator(
        *(
            "rule",
            "input",
            "documentation",
            "typeMode",
            "extends",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class StructureMap(DomainResource):
    """
    A Map of relationships between 2 structures that can be used to transform data.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/StructureMap"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this structure map, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifier for the structure map",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the structure map",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    name: Optional[String] = Field(
        description="Name for this structure map (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this structure map (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the structure map",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for structure map (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this structure map is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    structure: Optional[List[StructureMapStructure]] = Field(
        description="Structure Definition used by this map",
        default=None,
    )
    import_: Optional[List[Canonical]] = Field(
        description="Other maps used by this map (canonical URLs)",
        default=None,
    )
    import_ext: Optional[Element] = Field(
        description="Placeholder element for import extensions",
        default=None,
        alias="_import",
    )
    group: Optional[List[StructureMapGroup]] = Field(
        description="Named sections for reader convenience",
        default=None,
    )
    resourceType: Literal["StructureMap"] = Field(
        description=None,
        default="StructureMap",
    )

    @field_validator(
        *(
            "group",
            "import_",
            "structure",
            "copyright",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "experimental",
            "status",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_smp_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.matches('[A-Z]([A-Za-z0-9_]){0,254}')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="smp-0",
            severity="warning",
        )
