# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    DateTime,
    Markdown,
    Canonical,
    Integer,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    ContactDetail,
    UsageContext,
    CodeableConcept,
    BackboneElement,
    DomainResource,
)


class OperationDefinitionParameterBinding(BackboneElement):
    """
    Binds to a value set if this parameter is coded (code, Coding, CodeableConcept).
    """

    strength: Optional[Code] = Field(
        description="required | extensible | preferred | example",
        default=None,
    )
    strength_ext: Optional[Element] = Field(
        description="Placeholder element for strength extensions",
        default=None,
        alias="_strength",
    )
    valueSet: Optional[Canonical] = Field(
        description="Source of value set",
        default=None,
    )
    valueSet_ext: Optional[Element] = Field(
        description="Placeholder element for valueSet extensions",
        default=None,
        alias="_valueSet",
    )

    @field_validator(
        *(
            "valueSet",
            "strength",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class OperationDefinitionParameterReferencedFrom(BackboneElement):
    """
    Identifies other resource parameters within the operation invocation that are expected to resolve to this resource.
    """

    source: Optional[String] = Field(
        description="Referencing parameter",
        default=None,
    )
    source_ext: Optional[Element] = Field(
        description="Placeholder element for source extensions",
        default=None,
        alias="_source",
    )
    sourceId: Optional[String] = Field(
        description="Element id of reference",
        default=None,
    )
    sourceId_ext: Optional[Element] = Field(
        description="Placeholder element for sourceId extensions",
        default=None,
        alias="_sourceId",
    )

    @field_validator(
        *(
            "sourceId",
            "source",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class OperationDefinitionParameter(BackboneElement):
    """
    The parameters for the operation/query.
    """

    name: Optional[Code] = Field(
        description="Name in Parameters.parameter.name or in URL",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    use: Optional[Code] = Field(
        description="in | out",
        default=None,
    )
    use_ext: Optional[Element] = Field(
        description="Placeholder element for use extensions",
        default=None,
        alias="_use",
    )
    min: Optional[Integer] = Field(
        description="Minimum Cardinality",
        default=None,
    )
    min_ext: Optional[Element] = Field(
        description="Placeholder element for min extensions",
        default=None,
        alias="_min",
    )
    max: Optional[String] = Field(
        description="Maximum Cardinality (a number or *)",
        default=None,
    )
    max_ext: Optional[Element] = Field(
        description="Placeholder element for max extensions",
        default=None,
        alias="_max",
    )
    documentation: Optional[String] = Field(
        description="Description of meaning/use",
        default=None,
    )
    documentation_ext: Optional[Element] = Field(
        description="Placeholder element for documentation extensions",
        default=None,
        alias="_documentation",
    )
    type: Optional[Code] = Field(
        description="What type this parameter has",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    targetProfile: Optional[List[Canonical]] = Field(
        description="If type is Reference | canonical, allowed targets",
        default=None,
    )
    targetProfile_ext: Optional[Element] = Field(
        description="Placeholder element for targetProfile extensions",
        default=None,
        alias="_targetProfile",
    )
    searchType: Optional[Code] = Field(
        description="number | date | string | token | reference | composite | quantity | uri | special",
        default=None,
    )
    searchType_ext: Optional[Element] = Field(
        description="Placeholder element for searchType extensions",
        default=None,
        alias="_searchType",
    )
    binding: Optional[OperationDefinitionParameterBinding] = Field(
        description="ValueSet details if this is coded",
        default=None,
    )
    referencedFrom: Optional[List[OperationDefinitionParameterReferencedFrom]] = Field(
        description="References to this parameter",
        default=None,
    )
    part: Optional[List["OperationDefinitionParameter"]] = Field(
        description="Parts of a nested Parameter",
        default=None,
    )

    @field_validator(
        *(
            "part",
            "referencedFrom",
            "binding",
            "searchType",
            "targetProfile",
            "type",
            "documentation",
            "max",
            "min",
            "use",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class OperationDefinitionOverload(BackboneElement):
    """
    Defines an appropriate combination of parameters to use when invoking this operation, to help code generators when generating overloaded parameter sets for this operation.
    """

    parameterName: Optional[List[String]] = Field(
        description="Name of parameter to include in overload",
        default=None,
    )
    parameterName_ext: Optional[Element] = Field(
        description="Placeholder element for parameterName extensions",
        default=None,
        alias="_parameterName",
    )
    comment: Optional[String] = Field(
        description="Comments to go on overload",
        default=None,
    )
    comment_ext: Optional[Element] = Field(
        description="Placeholder element for comment extensions",
        default=None,
        alias="_comment",
    )

    @field_validator(
        *(
            "comment",
            "parameterName",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class OperationDefinition(DomainResource):
    """
    A formal computable definition of an operation (on the RESTful interface) or a named query (using the search interaction).
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/OperationDefinition"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this operation definition, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    version: Optional[String] = Field(
        description="Business version of the operation definition",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    name: Optional[String] = Field(
        description="Name for this operation definition (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this operation definition (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    kind: Optional[Code] = Field(
        description="operation | query",
        default=None,
    )
    kind_ext: Optional[Element] = Field(
        description="Placeholder element for kind extensions",
        default=None,
        alias="_kind",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the operation definition",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for operation definition (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this operation definition is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    affectsState: Optional[Boolean] = Field(
        description="Whether content is changed by the operation",
        default=None,
    )
    affectsState_ext: Optional[Element] = Field(
        description="Placeholder element for affectsState extensions",
        default=None,
        alias="_affectsState",
    )
    code: Optional[Code] = Field(
        description="Name used to invoke the operation",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    comment: Optional[Markdown] = Field(
        description="Additional information about use",
        default=None,
    )
    comment_ext: Optional[Element] = Field(
        description="Placeholder element for comment extensions",
        default=None,
        alias="_comment",
    )
    base: Optional[Canonical] = Field(
        description="Marks this as a profile of the base",
        default=None,
    )
    base_ext: Optional[Element] = Field(
        description="Placeholder element for base extensions",
        default=None,
        alias="_base",
    )
    resource: Optional[List[Code]] = Field(
        description="Types this operation applies to",
        default=None,
    )
    resource_ext: Optional[Element] = Field(
        description="Placeholder element for resource extensions",
        default=None,
        alias="_resource",
    )
    system: Optional[Boolean] = Field(
        description="Invoke at the system level?",
        default=None,
    )
    system_ext: Optional[Element] = Field(
        description="Placeholder element for system extensions",
        default=None,
        alias="_system",
    )
    type: Optional[Boolean] = Field(
        description="Invoke at the type level?",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    instance: Optional[Boolean] = Field(
        description="Invoke on an instance?",
        default=None,
    )
    instance_ext: Optional[Element] = Field(
        description="Placeholder element for instance extensions",
        default=None,
        alias="_instance",
    )
    inputProfile: Optional[Canonical] = Field(
        description="Validation information for in parameters",
        default=None,
    )
    inputProfile_ext: Optional[Element] = Field(
        description="Placeholder element for inputProfile extensions",
        default=None,
        alias="_inputProfile",
    )
    outputProfile: Optional[Canonical] = Field(
        description="Validation information for out parameters",
        default=None,
    )
    outputProfile_ext: Optional[Element] = Field(
        description="Placeholder element for outputProfile extensions",
        default=None,
        alias="_outputProfile",
    )
    parameter: Optional[List[OperationDefinitionParameter]] = Field(
        description="Parameters for the operation/query",
        default=None,
    )
    overload: Optional[List[OperationDefinitionOverload]] = Field(
        description="Define overloaded variants for when  generating code",
        default=None,
    )
    resourceType: Literal["OperationDefinition"] = Field(
        description=None,
        default="OperationDefinition",
    )

    @field_validator(
        *(
            "overload",
            "parameter",
            "outputProfile",
            "inputProfile",
            "instance",
            "type",
            "system",
            "resource",
            "base",
            "comment",
            "code",
            "affectsState",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "experimental",
            "kind",
            "status",
            "title",
            "name",
            "version",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("parameter",), mode="after", check_fields=None)
    @classmethod
    def FHIR_opd_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="type.exists() or part.exists()",
            human="Either a type must be provided, or parts",
            key="opd-1",
            severity="error",
        )

    @field_validator(*("parameter",), mode="after", check_fields=None)
    @classmethod
    def FHIR_opd_2_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="searchType.exists() implies type = 'string'",
            human="A search type can only be specified for parameters of type string",
            key="opd-2",
            severity="error",
        )

    @field_validator(*("parameter",), mode="after", check_fields=None)
    @classmethod
    def FHIR_opd_3_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="targetProfile.exists() implies (type = 'Reference' or type = 'canonical')",
            human="A targetProfile can only be specified for parameters of type Reference or Canonical",
            key="opd-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_opd_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.matches('[A-Z]([A-Za-z0-9_]){0,254}')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="opd-0",
            severity="warning",
        )
