# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    Decimal,
    Integer,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    CodeableConcept,
    Identifier,
    BackboneElement,
    Range,
    Reference,
    DomainResource,
)


class ObservationDefinitionQuantitativeDetails(BackboneElement):
    """
    Characteristics for quantitative results of this observation.
    """

    customaryUnit: Optional[CodeableConcept] = Field(
        description="Customary unit for quantitative results",
        default=None,
    )
    unit: Optional[CodeableConcept] = Field(
        description="SI unit for quantitative results",
        default=None,
    )
    conversionFactor: Optional[Decimal] = Field(
        description="SI to Customary unit conversion factor",
        default=None,
    )
    conversionFactor_ext: Optional[Element] = Field(
        description="Placeholder element for conversionFactor extensions",
        default=None,
        alias="_conversionFactor",
    )
    decimalPrecision: Optional[Integer] = Field(
        description="Decimal precision of observation quantitative results",
        default=None,
    )
    decimalPrecision_ext: Optional[Element] = Field(
        description="Placeholder element for decimalPrecision extensions",
        default=None,
        alias="_decimalPrecision",
    )

    @field_validator(
        *(
            "decimalPrecision",
            "conversionFactor",
            "unit",
            "customaryUnit",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ObservationDefinitionQualifiedInterval(BackboneElement):
    """
    Multiple  ranges of results qualified by different contexts for ordinal or continuous observations conforming to this ObservationDefinition.
    """

    category: Optional[Code] = Field(
        description="reference | critical | absolute",
        default=None,
    )
    category_ext: Optional[Element] = Field(
        description="Placeholder element for category extensions",
        default=None,
        alias="_category",
    )
    range: Optional[Range] = Field(
        description="The interval itself, for continuous or ordinal observations",
        default=None,
    )
    context: Optional[CodeableConcept] = Field(
        description="Range context qualifier",
        default=None,
    )
    appliesTo: Optional[List[CodeableConcept]] = Field(
        description="Targetted population of the range",
        default=None,
    )
    gender: Optional[Code] = Field(
        description="male | female | other | unknown",
        default=None,
    )
    gender_ext: Optional[Element] = Field(
        description="Placeholder element for gender extensions",
        default=None,
        alias="_gender",
    )
    age: Optional[Range] = Field(
        description="Applicable age range, if relevant",
        default=None,
    )
    gestationalAge: Optional[Range] = Field(
        description="Applicable gestational age range, if relevant",
        default=None,
    )
    condition: Optional[String] = Field(
        description="Condition associated with the reference range",
        default=None,
    )
    condition_ext: Optional[Element] = Field(
        description="Placeholder element for condition extensions",
        default=None,
        alias="_condition",
    )

    @field_validator(
        *(
            "condition",
            "gestationalAge",
            "age",
            "gender",
            "appliesTo",
            "context",
            "range",
            "category",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ObservationDefinition(DomainResource):
    """
    Set of definitional characteristics for a kind of observation or measurement produced or consumed by an orderable health care service.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/ObservationDefinition"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Category of observation",
        default=None,
    )
    code: Optional[CodeableConcept] = Field(
        description="Type of observation (code / type)",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business identifier for this ObservationDefinition instance",
        default=None,
    )
    permittedDataType: Optional[List[Code]] = Field(
        description="Quantity | CodeableConcept | string | boolean | integer | Range | Ratio | SampledData | time | dateTime | Period",
        default=None,
    )
    permittedDataType_ext: Optional[Element] = Field(
        description="Placeholder element for permittedDataType extensions",
        default=None,
        alias="_permittedDataType",
    )
    multipleResultsAllowed: Optional[Boolean] = Field(
        description="Multiple results allowed",
        default=None,
    )
    multipleResultsAllowed_ext: Optional[Element] = Field(
        description="Placeholder element for multipleResultsAllowed extensions",
        default=None,
        alias="_multipleResultsAllowed",
    )
    method: Optional[CodeableConcept] = Field(
        description="Method used to produce the observation",
        default=None,
    )
    preferredReportName: Optional[String] = Field(
        description="Preferred report name",
        default=None,
    )
    preferredReportName_ext: Optional[Element] = Field(
        description="Placeholder element for preferredReportName extensions",
        default=None,
        alias="_preferredReportName",
    )
    quantitativeDetails: Optional[ObservationDefinitionQuantitativeDetails] = Field(
        description="Characteristics of quantitative results",
        default=None,
    )
    qualifiedInterval: Optional[List[ObservationDefinitionQualifiedInterval]] = Field(
        description="Qualified range for continuous and ordinal observation results",
        default=None,
    )
    validCodedValueSet: Optional[Reference] = Field(
        description="Value set of valid coded values for the observations conforming to this ObservationDefinition",
        default=None,
    )
    normalCodedValueSet: Optional[Reference] = Field(
        description="Value set of normal coded values for the observations conforming to this ObservationDefinition",
        default=None,
    )
    abnormalCodedValueSet: Optional[Reference] = Field(
        description="Value set of abnormal coded values for the observations conforming to this ObservationDefinition",
        default=None,
    )
    criticalCodedValueSet: Optional[Reference] = Field(
        description="Value set of critical coded values for the observations conforming to this ObservationDefinition",
        default=None,
    )
    resourceType: Literal["ObservationDefinition"] = Field(
        description=None,
        default="ObservationDefinition",
    )

    @field_validator(
        *(
            "criticalCodedValueSet",
            "abnormalCodedValueSet",
            "normalCodedValueSet",
            "validCodedValueSet",
            "qualifiedInterval",
            "quantitativeDetails",
            "preferredReportName",
            "method",
            "multipleResultsAllowed",
            "permittedDataType",
            "identifier",
            "code",
            "category",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
