# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Canonical,
    DateTime,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    CodeableConcept,
    BackboneElement,
    Timing,
    Quantity,
    Ratio,
    Annotation,
    DomainResource,
)


class NutritionOrderOralDietNutrient(BackboneElement):
    """
    Class that defines the quantity and type of nutrient modifications (for example carbohydrate, fiber or sodium) required for the oral diet.
    """

    modifier: Optional[CodeableConcept] = Field(
        description="Type of nutrient that is being modified",
        default=None,
    )
    amount: Optional[Quantity] = Field(
        description="Quantity of the specified nutrient",
        default=None,
    )

    @field_validator(
        *(
            "amount",
            "modifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class NutritionOrderOralDietTexture(BackboneElement):
    """
    Class that describes any texture modifications required for the patient to safely consume various types of solid foods.
    """

    modifier: Optional[CodeableConcept] = Field(
        description="Code to indicate how to alter the texture of the foods, e.g. pureed",
        default=None,
    )
    foodType: Optional[CodeableConcept] = Field(
        description="Concepts that are used to identify an entity that is ingested for nutritional purposes",
        default=None,
    )

    @field_validator(
        *(
            "foodType",
            "modifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class NutritionOrderOralDiet(BackboneElement):
    """
    Diet given orally in contrast to enteral (tube) feeding.
    """

    type: Optional[List[CodeableConcept]] = Field(
        description="Type of oral diet or diet restrictions that describe what can be consumed orally",
        default=None,
    )
    schedule: Optional[List[Timing]] = Field(
        description="Scheduled frequency of diet",
        default=None,
    )
    nutrient: Optional[List[NutritionOrderOralDietNutrient]] = Field(
        description="Required  nutrient modifications",
        default=None,
    )
    texture: Optional[List[NutritionOrderOralDietTexture]] = Field(
        description="Required  texture modifications",
        default=None,
    )
    fluidConsistencyType: Optional[List[CodeableConcept]] = Field(
        description="The required consistency of fluids and liquids provided to the patient",
        default=None,
    )
    instruction: Optional[String] = Field(
        description="Instructions or additional information about the oral diet",
        default=None,
    )
    instruction_ext: Optional[Element] = Field(
        description="Placeholder element for instruction extensions",
        default=None,
        alias="_instruction",
    )

    @field_validator(
        *(
            "instruction",
            "fluidConsistencyType",
            "texture",
            "nutrient",
            "schedule",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class NutritionOrderSupplement(BackboneElement):
    """
    Oral nutritional products given in order to add further nutritional value to the patient's diet.
    """

    type: Optional[CodeableConcept] = Field(
        description="Type of supplement product requested",
        default=None,
    )
    productName: Optional[String] = Field(
        description="Product or brand name of the nutritional supplement",
        default=None,
    )
    productName_ext: Optional[Element] = Field(
        description="Placeholder element for productName extensions",
        default=None,
        alias="_productName",
    )
    schedule: Optional[List[Timing]] = Field(
        description="Scheduled frequency of supplement",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="Amount of the nutritional supplement",
        default=None,
    )
    instruction: Optional[String] = Field(
        description="Instructions or additional information about the oral supplement",
        default=None,
    )
    instruction_ext: Optional[Element] = Field(
        description="Placeholder element for instruction extensions",
        default=None,
        alias="_instruction",
    )

    @field_validator(
        *(
            "instruction",
            "quantity",
            "schedule",
            "productName",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class NutritionOrderEnteralFormulaAdministration(BackboneElement):
    """
    Formula administration instructions as structured data.  This repeating structure allows for changing the administration rate or volume over time for both bolus and continuous feeding.  An example of this would be an instruction to increase the rate of continuous feeding every 2 hours.
    """

    schedule: Optional[Timing] = Field(
        description="Scheduled frequency of enteral feeding",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="The volume of formula to provide",
        default=None,
    )
    rateQuantity: Optional[Quantity] = Field(
        description="Speed with which the formula is provided per period of time",
        default=None,
    )
    rateRatio: Optional[Ratio] = Field(
        description="Speed with which the formula is provided per period of time",
        default=None,
    )

    @property
    def rate(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="rate",
        )

    @field_validator(
        *(
            "quantity",
            "schedule",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def rate_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Quantity, Ratio],
            field_name_base="rate",
            required=False,
        )


class NutritionOrderEnteralFormula(BackboneElement):
    """
    Feeding provided through the gastrointestinal tract via a tube, catheter, or stoma that delivers nutrition distal to the oral cavity.
    """

    baseFormulaType: Optional[CodeableConcept] = Field(
        description="Type of enteral or infant formula",
        default=None,
    )
    baseFormulaProductName: Optional[String] = Field(
        description="Product or brand name of the enteral or infant formula",
        default=None,
    )
    baseFormulaProductName_ext: Optional[Element] = Field(
        description="Placeholder element for baseFormulaProductName extensions",
        default=None,
        alias="_baseFormulaProductName",
    )
    additiveType: Optional[CodeableConcept] = Field(
        description="Type of modular component to add to the feeding",
        default=None,
    )
    additiveProductName: Optional[String] = Field(
        description="Product or brand name of the modular additive",
        default=None,
    )
    additiveProductName_ext: Optional[Element] = Field(
        description="Placeholder element for additiveProductName extensions",
        default=None,
        alias="_additiveProductName",
    )
    caloricDensity: Optional[Quantity] = Field(
        description="Amount of energy per specified volume that is required",
        default=None,
    )
    routeofAdministration: Optional[CodeableConcept] = Field(
        description="How the formula should enter the patient\u0027s gastrointestinal tract",
        default=None,
    )
    administration: Optional[List[NutritionOrderEnteralFormulaAdministration]] = Field(
        description="Formula feeding instruction as structured data",
        default=None,
    )
    maxVolumeToDeliver: Optional[Quantity] = Field(
        description="Upper limit on formula volume per unit of time",
        default=None,
    )
    administrationInstruction: Optional[String] = Field(
        description="Formula feeding instructions expressed as text",
        default=None,
    )
    administrationInstruction_ext: Optional[Element] = Field(
        description="Placeholder element for administrationInstruction extensions",
        default=None,
        alias="_administrationInstruction",
    )

    @field_validator(
        *(
            "administrationInstruction",
            "maxVolumeToDeliver",
            "administration",
            "routeofAdministration",
            "caloricDensity",
            "additiveProductName",
            "additiveType",
            "baseFormulaProductName",
            "baseFormulaType",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class NutritionOrder(DomainResource):
    """
    A request to supply a diet, formula feeding (enteral) or oral nutritional supplement to a patient/resident.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/NutritionOrder"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Identifiers assigned to this order",
        default=None,
    )
    instantiatesCanonical: Optional[List[Canonical]] = Field(
        description="Instantiates FHIR protocol or definition",
        default=None,
    )
    instantiatesCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesCanonical extensions",
        default=None,
        alias="_instantiatesCanonical",
    )
    instantiatesUri: Optional[List[Uri]] = Field(
        description="Instantiates external protocol or definition",
        default=None,
    )
    instantiatesUri_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesUri extensions",
        default=None,
        alias="_instantiatesUri",
    )
    instantiates: Optional[List[Uri]] = Field(
        description="Instantiates protocol or definition",
        default=None,
    )
    instantiates_ext: Optional[Element] = Field(
        description="Placeholder element for instantiates extensions",
        default=None,
        alias="_instantiates",
    )
    status: Optional[Code] = Field(
        description="draft | active | on-hold | revoked | completed | entered-in-error | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    intent: Optional[Code] = Field(
        description="proposal | plan | directive | order | original-order | reflex-order | filler-order | instance-order | option",
        default=None,
    )
    intent_ext: Optional[Element] = Field(
        description="Placeholder element for intent extensions",
        default=None,
        alias="_intent",
    )
    patient: Optional[Reference] = Field(
        description="The person who requires the diet, formula or nutritional supplement",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="The encounter associated with this nutrition order",
        default=None,
    )
    dateTime: Optional[DateTime] = Field(
        description="Date and time the nutrition order was requested",
        default=None,
    )
    dateTime_ext: Optional[Element] = Field(
        description="Placeholder element for dateTime extensions",
        default=None,
        alias="_dateTime",
    )
    orderer: Optional[Reference] = Field(
        description="Who ordered the diet, formula or nutritional supplement",
        default=None,
    )
    allergyIntolerance: Optional[List[Reference]] = Field(
        description="List of the patient\u0027s food and nutrition-related allergies and intolerances",
        default=None,
    )
    foodPreferenceModifier: Optional[List[CodeableConcept]] = Field(
        description="Order-specific modifier about the type of food that should be given",
        default=None,
    )
    excludeFoodModifier: Optional[List[CodeableConcept]] = Field(
        description="Order-specific modifier about the type of food that should not be given",
        default=None,
    )
    oralDiet: Optional[NutritionOrderOralDiet] = Field(
        description="Oral diet components",
        default=None,
    )
    supplement: Optional[List[NutritionOrderSupplement]] = Field(
        description="Supplement components",
        default=None,
    )
    enteralFormula: Optional[NutritionOrderEnteralFormula] = Field(
        description="Enteral formula components",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Comments",
        default=None,
    )
    resourceType: Literal["NutritionOrder"] = Field(
        description=None,
        default="NutritionOrder",
    )

    @field_validator(
        *(
            "note",
            "enteralFormula",
            "supplement",
            "oralDiet",
            "excludeFoodModifier",
            "foodPreferenceModifier",
            "allergyIntolerance",
            "orderer",
            "dateTime",
            "encounter",
            "patient",
            "intent",
            "status",
            "instantiates",
            "instantiatesUri",
            "instantiatesCanonical",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_nor_1_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="oralDiet.exists() or supplement.exists() or enteralFormula.exists()",
            human="Nutrition Order SHALL contain either Oral Diet , Supplement, or Enteral Formula class",
            key="nor-1",
            severity="warning",
        )
