# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Integer,
    Decimal,
    Boolean,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    Quantity,
    BackboneElement,
    CodeableConcept,
    DomainResource,
)


class MolecularSequenceReferenceSeq(BackboneElement):
    """
    A sequence that is used as a reference to describe variants that are present in a sequence analyzed.
    """

    chromosome: Optional[CodeableConcept] = Field(
        description="Chromosome containing genetic finding",
        default=None,
    )
    genomeBuild: Optional[String] = Field(
        description="The Genome Build used for reference, following GRCh build versions e.g. \u0027GRCh 37\u0027",
        default=None,
    )
    genomeBuild_ext: Optional[Element] = Field(
        description="Placeholder element for genomeBuild extensions",
        default=None,
        alias="_genomeBuild",
    )
    orientation: Optional[Code] = Field(
        description="sense | antisense",
        default=None,
    )
    orientation_ext: Optional[Element] = Field(
        description="Placeholder element for orientation extensions",
        default=None,
        alias="_orientation",
    )
    referenceSeqId: Optional[CodeableConcept] = Field(
        description="Reference identifier",
        default=None,
    )
    referenceSeqPointer: Optional[Reference] = Field(
        description="A pointer to another MolecularSequence entity as reference sequence",
        default=None,
    )
    referenceSeqString: Optional[String] = Field(
        description="A string to represent reference sequence",
        default=None,
    )
    referenceSeqString_ext: Optional[Element] = Field(
        description="Placeholder element for referenceSeqString extensions",
        default=None,
        alias="_referenceSeqString",
    )
    strand: Optional[Code] = Field(
        description="watson | crick",
        default=None,
    )
    strand_ext: Optional[Element] = Field(
        description="Placeholder element for strand extensions",
        default=None,
        alias="_strand",
    )
    windowStart: Optional[Integer] = Field(
        description="Start position of the window on the  reference sequence",
        default=None,
    )
    windowStart_ext: Optional[Element] = Field(
        description="Placeholder element for windowStart extensions",
        default=None,
        alias="_windowStart",
    )
    windowEnd: Optional[Integer] = Field(
        description="End position of the window on the reference sequence",
        default=None,
    )
    windowEnd_ext: Optional[Element] = Field(
        description="Placeholder element for windowEnd extensions",
        default=None,
        alias="_windowEnd",
    )

    @field_validator(
        *(
            "windowEnd",
            "windowStart",
            "strand",
            "referenceSeqString",
            "referenceSeqPointer",
            "referenceSeqId",
            "orientation",
            "genomeBuild",
            "chromosome",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MolecularSequenceVariant(BackboneElement):
    """
    The definition of variant here originates from Sequence ontology ([variant_of](http://www.sequenceontology.org/browser/current_svn/term/variant_of)). This element can represent amino acid or nucleic sequence change(including insertion,deletion,SNP,etc.)  It can represent some complex mutation or segment variation with the assist of CIGAR string.
    """

    start: Optional[Integer] = Field(
        description="Start position of the variant on the  reference sequence",
        default=None,
    )
    start_ext: Optional[Element] = Field(
        description="Placeholder element for start extensions",
        default=None,
        alias="_start",
    )
    end: Optional[Integer] = Field(
        description="End position of the variant on the reference sequence",
        default=None,
    )
    end_ext: Optional[Element] = Field(
        description="Placeholder element for end extensions",
        default=None,
        alias="_end",
    )
    observedAllele: Optional[String] = Field(
        description="Allele that was observed",
        default=None,
    )
    observedAllele_ext: Optional[Element] = Field(
        description="Placeholder element for observedAllele extensions",
        default=None,
        alias="_observedAllele",
    )
    referenceAllele: Optional[String] = Field(
        description="Allele in the reference sequence",
        default=None,
    )
    referenceAllele_ext: Optional[Element] = Field(
        description="Placeholder element for referenceAllele extensions",
        default=None,
        alias="_referenceAllele",
    )
    cigar: Optional[String] = Field(
        description="Extended CIGAR string for aligning the sequence with reference bases",
        default=None,
    )
    cigar_ext: Optional[Element] = Field(
        description="Placeholder element for cigar extensions",
        default=None,
        alias="_cigar",
    )
    variantPointer: Optional[Reference] = Field(
        description="Pointer to observed variant information",
        default=None,
    )

    @field_validator(
        *(
            "variantPointer",
            "cigar",
            "referenceAllele",
            "observedAllele",
            "end",
            "start",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MolecularSequenceQualityRoc(BackboneElement):
    """
    Receiver Operator Characteristic (ROC) Curve  to give sensitivity/specificity tradeoff.
    """

    score: Optional[List[Integer]] = Field(
        description="Genotype quality score",
        default=None,
    )
    score_ext: Optional[Element] = Field(
        description="Placeholder element for score extensions",
        default=None,
        alias="_score",
    )
    numTP: Optional[List[Integer]] = Field(
        description="Roc score true positive numbers",
        default=None,
    )
    numTP_ext: Optional[Element] = Field(
        description="Placeholder element for numTP extensions",
        default=None,
        alias="_numTP",
    )
    numFP: Optional[List[Integer]] = Field(
        description="Roc score false positive numbers",
        default=None,
    )
    numFP_ext: Optional[Element] = Field(
        description="Placeholder element for numFP extensions",
        default=None,
        alias="_numFP",
    )
    numFN: Optional[List[Integer]] = Field(
        description="Roc score false negative numbers",
        default=None,
    )
    numFN_ext: Optional[Element] = Field(
        description="Placeholder element for numFN extensions",
        default=None,
        alias="_numFN",
    )
    precision: Optional[List[Decimal]] = Field(
        description="Precision of the GQ score",
        default=None,
    )
    precision_ext: Optional[Element] = Field(
        description="Placeholder element for precision extensions",
        default=None,
        alias="_precision",
    )
    sensitivity: Optional[List[Decimal]] = Field(
        description="Sensitivity of the GQ score",
        default=None,
    )
    sensitivity_ext: Optional[Element] = Field(
        description="Placeholder element for sensitivity extensions",
        default=None,
        alias="_sensitivity",
    )
    fMeasure: Optional[List[Decimal]] = Field(
        description="FScore of the GQ score",
        default=None,
    )
    fMeasure_ext: Optional[Element] = Field(
        description="Placeholder element for fMeasure extensions",
        default=None,
        alias="_fMeasure",
    )

    @field_validator(
        *(
            "fMeasure",
            "sensitivity",
            "precision",
            "numFN",
            "numFP",
            "numTP",
            "score",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MolecularSequenceQuality(BackboneElement):
    """
    An experimental feature attribute that defines the quality of the feature in a quantitative way, such as a phred quality score ([SO:0001686](http://www.sequenceontology.org/browser/current_svn/term/SO:0001686)).
    """

    type: Optional[Code] = Field(
        description="indel | snp | unknown",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    standardSequence: Optional[CodeableConcept] = Field(
        description="Standard sequence for comparison",
        default=None,
    )
    start: Optional[Integer] = Field(
        description="Start position of the sequence",
        default=None,
    )
    start_ext: Optional[Element] = Field(
        description="Placeholder element for start extensions",
        default=None,
        alias="_start",
    )
    end: Optional[Integer] = Field(
        description="End position of the sequence",
        default=None,
    )
    end_ext: Optional[Element] = Field(
        description="Placeholder element for end extensions",
        default=None,
        alias="_end",
    )
    score: Optional[Quantity] = Field(
        description="Quality score for the comparison",
        default=None,
    )
    method: Optional[CodeableConcept] = Field(
        description="Method to get quality",
        default=None,
    )
    truthTP: Optional[Decimal] = Field(
        description="True positives from the perspective of the truth data",
        default=None,
    )
    truthTP_ext: Optional[Element] = Field(
        description="Placeholder element for truthTP extensions",
        default=None,
        alias="_truthTP",
    )
    queryTP: Optional[Decimal] = Field(
        description="True positives from the perspective of the query data",
        default=None,
    )
    queryTP_ext: Optional[Element] = Field(
        description="Placeholder element for queryTP extensions",
        default=None,
        alias="_queryTP",
    )
    truthFN: Optional[Decimal] = Field(
        description="False negatives",
        default=None,
    )
    truthFN_ext: Optional[Element] = Field(
        description="Placeholder element for truthFN extensions",
        default=None,
        alias="_truthFN",
    )
    queryFP: Optional[Decimal] = Field(
        description="False positives",
        default=None,
    )
    queryFP_ext: Optional[Element] = Field(
        description="Placeholder element for queryFP extensions",
        default=None,
        alias="_queryFP",
    )
    gtFP: Optional[Decimal] = Field(
        description="False positives where the non-REF alleles in the Truth and Query Call Sets match",
        default=None,
    )
    gtFP_ext: Optional[Element] = Field(
        description="Placeholder element for gtFP extensions",
        default=None,
        alias="_gtFP",
    )
    precision: Optional[Decimal] = Field(
        description="Precision of comparison",
        default=None,
    )
    precision_ext: Optional[Element] = Field(
        description="Placeholder element for precision extensions",
        default=None,
        alias="_precision",
    )
    recall: Optional[Decimal] = Field(
        description="Recall of comparison",
        default=None,
    )
    recall_ext: Optional[Element] = Field(
        description="Placeholder element for recall extensions",
        default=None,
        alias="_recall",
    )
    fScore: Optional[Decimal] = Field(
        description="F-score",
        default=None,
    )
    fScore_ext: Optional[Element] = Field(
        description="Placeholder element for fScore extensions",
        default=None,
        alias="_fScore",
    )
    roc: Optional[MolecularSequenceQualityRoc] = Field(
        description="Receiver Operator Characteristic (ROC) Curve",
        default=None,
    )

    @field_validator(
        *(
            "roc",
            "fScore",
            "recall",
            "precision",
            "gtFP",
            "queryFP",
            "truthFN",
            "queryTP",
            "truthTP",
            "method",
            "score",
            "end",
            "start",
            "standardSequence",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MolecularSequenceRepository(BackboneElement):
    """
    Configurations of the external repository. The repository shall store target's observedSeq or records related with target's observedSeq.
    """

    type: Optional[Code] = Field(
        description="directlink | openapi | login | oauth | other",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    url: Optional[Uri] = Field(
        description="URI of the repository",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    name: Optional[String] = Field(
        description="Repository\u0027s name",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    datasetId: Optional[String] = Field(
        description="Id of the dataset that used to call for dataset in repository",
        default=None,
    )
    datasetId_ext: Optional[Element] = Field(
        description="Placeholder element for datasetId extensions",
        default=None,
        alias="_datasetId",
    )
    variantsetId: Optional[String] = Field(
        description="Id of the variantset that used to call for variantset in repository",
        default=None,
    )
    variantsetId_ext: Optional[Element] = Field(
        description="Placeholder element for variantsetId extensions",
        default=None,
        alias="_variantsetId",
    )
    readsetId: Optional[String] = Field(
        description="Id of the read",
        default=None,
    )
    readsetId_ext: Optional[Element] = Field(
        description="Placeholder element for readsetId extensions",
        default=None,
        alias="_readsetId",
    )

    @field_validator(
        *(
            "readsetId",
            "variantsetId",
            "datasetId",
            "name",
            "url",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MolecularSequenceStructureVariantOuter(BackboneElement):
    """
    Structural variant outer.
    """

    start: Optional[Integer] = Field(
        description="Structural variant outer start",
        default=None,
    )
    start_ext: Optional[Element] = Field(
        description="Placeholder element for start extensions",
        default=None,
        alias="_start",
    )
    end: Optional[Integer] = Field(
        description="Structural variant outer end",
        default=None,
    )
    end_ext: Optional[Element] = Field(
        description="Placeholder element for end extensions",
        default=None,
        alias="_end",
    )

    @field_validator(
        *(
            "end",
            "start",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MolecularSequenceStructureVariantInner(BackboneElement):
    """
    Structural variant inner.
    """

    start: Optional[Integer] = Field(
        description="Structural variant inner start",
        default=None,
    )
    start_ext: Optional[Element] = Field(
        description="Placeholder element for start extensions",
        default=None,
        alias="_start",
    )
    end: Optional[Integer] = Field(
        description="Structural variant inner end",
        default=None,
    )
    end_ext: Optional[Element] = Field(
        description="Placeholder element for end extensions",
        default=None,
        alias="_end",
    )

    @field_validator(
        *(
            "end",
            "start",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MolecularSequenceStructureVariant(BackboneElement):
    """
    Information about chromosome structure variation.
    """

    variantType: Optional[CodeableConcept] = Field(
        description="Structural variant change type",
        default=None,
    )
    exact: Optional[Boolean] = Field(
        description="Does the structural variant have base pair resolution breakpoints?",
        default=None,
    )
    exact_ext: Optional[Element] = Field(
        description="Placeholder element for exact extensions",
        default=None,
        alias="_exact",
    )
    length: Optional[Integer] = Field(
        description="Structural variant length",
        default=None,
    )
    length_ext: Optional[Element] = Field(
        description="Placeholder element for length extensions",
        default=None,
        alias="_length",
    )
    outer: Optional[MolecularSequenceStructureVariantOuter] = Field(
        description="Structural variant outer",
        default=None,
    )
    inner: Optional[MolecularSequenceStructureVariantInner] = Field(
        description="Structural variant inner",
        default=None,
    )

    @field_validator(
        *(
            "inner",
            "outer",
            "length",
            "exact",
            "variantType",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MolecularSequence(DomainResource):
    """
    Raw data describing a biological sequence.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/MolecularSequence"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Unique ID for this particular sequence. This is a FHIR-defined id",
        default=None,
    )
    type: Optional[Code] = Field(
        description="aa | dna | rna",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    coordinateSystem: Optional[Integer] = Field(
        description="Base number of coordinate system (0 for 0-based numbering or coordinates, inclusive start, exclusive end, 1 for 1-based numbering, inclusive start, inclusive end)",
        default=None,
    )
    coordinateSystem_ext: Optional[Element] = Field(
        description="Placeholder element for coordinateSystem extensions",
        default=None,
        alias="_coordinateSystem",
    )
    patient: Optional[Reference] = Field(
        description="Who and/or what this is about",
        default=None,
    )
    specimen: Optional[Reference] = Field(
        description="Specimen used for sequencing",
        default=None,
    )
    device: Optional[Reference] = Field(
        description="The method for sequencing",
        default=None,
    )
    performer: Optional[Reference] = Field(
        description="Who should be responsible for test result",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="The number of copies of the sequence of interest.  (RNASeq)",
        default=None,
    )
    referenceSeq: Optional[MolecularSequenceReferenceSeq] = Field(
        description="A sequence used as reference",
        default=None,
    )
    variant: Optional[List[MolecularSequenceVariant]] = Field(
        description="Variant in sequence",
        default=None,
    )
    observedSeq: Optional[String] = Field(
        description="Sequence that was observed",
        default=None,
    )
    observedSeq_ext: Optional[Element] = Field(
        description="Placeholder element for observedSeq extensions",
        default=None,
        alias="_observedSeq",
    )
    quality: Optional[List[MolecularSequenceQuality]] = Field(
        description="An set of value as quality of sequence",
        default=None,
    )
    readCoverage: Optional[Integer] = Field(
        description="Average number of reads representing a given nucleotide in the reconstructed sequence",
        default=None,
    )
    readCoverage_ext: Optional[Element] = Field(
        description="Placeholder element for readCoverage extensions",
        default=None,
        alias="_readCoverage",
    )
    repository: Optional[List[MolecularSequenceRepository]] = Field(
        description="External repository which contains detailed report related with observedSeq in this resource",
        default=None,
    )
    pointer: Optional[List[Reference]] = Field(
        description="Pointer to next atomic sequence",
        default=None,
    )
    structureVariant: Optional[List[MolecularSequenceStructureVariant]] = Field(
        description="Structural variant",
        default=None,
    )
    resourceType: Literal["MolecularSequence"] = Field(
        description=None,
        default="MolecularSequence",
    )

    @field_validator(
        *(
            "structureVariant",
            "pointer",
            "repository",
            "readCoverage",
            "quality",
            "observedSeq",
            "variant",
            "referenceSeq",
            "quantity",
            "performer",
            "device",
            "specimen",
            "patient",
            "coordinateSystem",
            "type",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("referenceSeq",), mode="after", check_fields=None)
    @classmethod
    def FHIR_msq_5_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(chromosome.empty() and genomeBuild.empty()) or (chromosome.exists() and genomeBuild.exists())",
            human="GenomeBuild and chromosome must be both contained if either one of them is contained",
            key="msq-5",
            severity="error",
        )

    @field_validator(*("referenceSeq",), mode="after", check_fields=None)
    @classmethod
    def FHIR_msq_6_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(genomeBuild.count()+referenceSeqId.count()+ referenceSeqPointer.count()+ referenceSeqString.count()) = 1",
            human="Have and only have one of the following elements in referenceSeq : 1. genomeBuild ; 2 referenceSeqId; 3. referenceSeqPointer;  4. referenceSeqString;",
            key="msq-6",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_msq_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="coordinateSystem = 1 or coordinateSystem = 0",
            human="Only 0 and 1 are valid for coordinateSystem",
            key="msq-3",
            severity="error",
        )
