# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Url,
    Id,
    Canonical,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Coding,
    BackboneElement,
    Reference,
    ContactPoint,
    CodeableConcept,
    DomainResource,
)


class MessageHeaderDestination(BackboneElement):
    """
    The destination application which the message is intended for.
    """

    name: Optional[String] = Field(
        description="Name of system",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    target: Optional[Reference] = Field(
        description="Particular delivery destination within the destination",
        default=None,
    )
    endpoint: Optional[Url] = Field(
        description="Actual destination address or id",
        default=None,
    )
    endpoint_ext: Optional[Element] = Field(
        description="Placeholder element for endpoint extensions",
        default=None,
        alias="_endpoint",
    )
    receiver: Optional[Reference] = Field(
        description='Intended "real-world" recipient for the data',
        default=None,
    )

    @field_validator(
        *(
            "receiver",
            "endpoint",
            "target",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MessageHeaderSource(BackboneElement):
    """
    The source application from which this message originated.
    """

    name: Optional[String] = Field(
        description="Name of system",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    software: Optional[String] = Field(
        description="Name of software running the system",
        default=None,
    )
    software_ext: Optional[Element] = Field(
        description="Placeholder element for software extensions",
        default=None,
        alias="_software",
    )
    version: Optional[String] = Field(
        description="Version of software running",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    contact: Optional[ContactPoint] = Field(
        description="Human contact for problems",
        default=None,
    )
    endpoint: Optional[Url] = Field(
        description="Actual message source address or id",
        default=None,
    )
    endpoint_ext: Optional[Element] = Field(
        description="Placeholder element for endpoint extensions",
        default=None,
        alias="_endpoint",
    )

    @field_validator(
        *(
            "endpoint",
            "contact",
            "version",
            "software",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MessageHeaderResponse(BackboneElement):
    """
    Information about the message that this message is a response to.  Only present if this message is a response.
    """

    identifier: Optional[Id] = Field(
        description="Id of original message",
        default=None,
    )
    identifier_ext: Optional[Element] = Field(
        description="Placeholder element for identifier extensions",
        default=None,
        alias="_identifier",
    )
    code: Optional[Code] = Field(
        description="ok | transient-error | fatal-error",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    details: Optional[Reference] = Field(
        description="Specific list of hints/warnings/errors",
        default=None,
    )

    @field_validator(
        *(
            "details",
            "code",
            "identifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MessageHeader(DomainResource):
    """
    The header for a message exchange that is either requesting or responding to an action.  The reference(s) that are the subject of the action as well as other information related to the action are typically transmitted in a bundle in which the MessageHeader resource instance is the first resource in the bundle.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/MessageHeader"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    eventCoding: Optional[Coding] = Field(
        description="Code for the event this message represents or link to event definition",
        default=None,
    )
    eventUri: Optional[Uri] = Field(
        description="Code for the event this message represents or link to event definition",
        default=None,
    )
    eventUri_ext: Optional[Element] = Field(
        description="Placeholder element for eventUri extensions",
        default=None,
        alias="_eventUri",
    )
    destination: Optional[List[MessageHeaderDestination]] = Field(
        description="Message destination application(s)",
        default=None,
    )
    sender: Optional[Reference] = Field(
        description="Real world sender of the message",
        default=None,
    )
    enterer: Optional[Reference] = Field(
        description="The source of the data entry",
        default=None,
    )
    author: Optional[Reference] = Field(
        description="The source of the decision",
        default=None,
    )
    source: Optional[MessageHeaderSource] = Field(
        description="Message source application",
        default=None,
    )
    responsible: Optional[Reference] = Field(
        description="Final responsibility for event",
        default=None,
    )
    reason: Optional[CodeableConcept] = Field(
        description="Cause of event",
        default=None,
    )
    response: Optional[MessageHeaderResponse] = Field(
        description="If this is a reply to prior message",
        default=None,
    )
    focus: Optional[List[Reference]] = Field(
        description="The actual content of the message",
        default=None,
    )
    definition: Optional[Canonical] = Field(
        description="Link to the definition for this message",
        default=None,
    )
    definition_ext: Optional[Element] = Field(
        description="Placeholder element for definition extensions",
        default=None,
        alias="_definition",
    )
    resourceType: Literal["MessageHeader"] = Field(
        description=None,
        default="MessageHeader",
    )

    @property
    def event(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="event",
        )

    @field_validator(
        *(
            "definition",
            "focus",
            "response",
            "reason",
            "responsible",
            "source",
            "author",
            "enterer",
            "sender",
            "destination",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def event_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Coding, Uri],
            field_name_base="event",
            required=True,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
