# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    CodeableConcept,
    MarketingStatus,
    BackboneElement,
    Quantity,
    ProdCharacteristic,
    ProductShelfLife,
    DomainResource,
)


class MedicinalProductPackagedBatchIdentifier(BackboneElement):
    """
    Batch numbering.
    """

    outerPackaging: Optional[Identifier] = Field(
        description="A number appearing on the outer packaging of a specific batch",
        default=None,
    )
    immediatePackaging: Optional[Identifier] = Field(
        description="A number appearing on the immediate packaging (and not the outer packaging)",
        default=None,
    )

    @field_validator(
        *(
            "immediatePackaging",
            "outerPackaging",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicinalProductPackagedPackageItem(BackboneElement):
    """
    A packaging item, as a contained for medicine, possibly with other packaging items within.
    """

    identifier: Optional[List[Identifier]] = Field(
        description="Including possibly Data Carrier Identifier",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="The physical type of the container of the medicine",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="The quantity of this package in the medicinal product, at the current level of packaging. The outermost is always 1",
        default=None,
    )
    material: Optional[List[CodeableConcept]] = Field(
        description="Material type of the package item",
        default=None,
    )
    alternateMaterial: Optional[List[CodeableConcept]] = Field(
        description="A possible alternate material for the packaging",
        default=None,
    )
    device: Optional[List[Reference]] = Field(
        description="A device accompanying a medicinal product",
        default=None,
    )
    manufacturedItem: Optional[List[Reference]] = Field(
        description="The manufactured item as contained in the packaged medicinal product",
        default=None,
    )
    packageItem: Optional[List["MedicinalProductPackagedPackageItem"]] = Field(
        description="Allows containers within containers",
        default=None,
    )
    physicalCharacteristics: Optional[ProdCharacteristic] = Field(
        description="Dimensions, color etc.",
        default=None,
    )
    otherCharacteristics: Optional[List[CodeableConcept]] = Field(
        description="Other codeable characteristics",
        default=None,
    )
    shelfLifeStorage: Optional[List[ProductShelfLife]] = Field(
        description="Shelf Life and storage information",
        default=None,
    )
    manufacturer: Optional[List[Reference]] = Field(
        description="Manufacturer of this Package Item",
        default=None,
    )

    @field_validator(
        *(
            "manufacturer",
            "shelfLifeStorage",
            "otherCharacteristics",
            "physicalCharacteristics",
            "packageItem",
            "manufacturedItem",
            "device",
            "alternateMaterial",
            "material",
            "quantity",
            "type",
            "identifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicinalProductPackaged(DomainResource):
    """
    A medicinal product in a container or package.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/MedicinalProductPackaged"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Unique identifier",
        default=None,
    )
    subject: Optional[List[Reference]] = Field(
        description="The product with this is a pack for",
        default=None,
    )
    description: Optional[String] = Field(
        description="Textual description",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    legalStatusOfSupply: Optional[CodeableConcept] = Field(
        description="The legal status of supply of the medicinal product as classified by the regulator",
        default=None,
    )
    marketingStatus: Optional[List[MarketingStatus]] = Field(
        description="Marketing information",
        default=None,
    )
    marketingAuthorization: Optional[Reference] = Field(
        description="Manufacturer of this Package Item",
        default=None,
    )
    manufacturer: Optional[List[Reference]] = Field(
        description="Manufacturer of this Package Item",
        default=None,
    )
    batchIdentifier: Optional[List[MedicinalProductPackagedBatchIdentifier]] = Field(
        description="Batch numbering",
        default=None,
    )
    packageItem: Optional[List[MedicinalProductPackagedPackageItem]] = Field(
        description="A packaging item, as a contained for medicine, possibly with other packaging items within",
        default=None,
    )
    resourceType: Literal["MedicinalProductPackaged"] = Field(
        description=None,
        default="MedicinalProductPackaged",
    )

    @field_validator(
        *(
            "packageItem",
            "batchIdentifier",
            "manufacturer",
            "marketingAuthorization",
            "marketingStatus",
            "legalStatusOfSupply",
            "description",
            "subject",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
