# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code, Boolean

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    BackboneElement,
    Ratio,
    DomainResource,
)


class MedicinalProductIngredientSpecifiedSubstanceStrengthReferenceStrength(
    BackboneElement
):
    """
    Strength expressed in terms of a reference substance.
    """

    substance: Optional[CodeableConcept] = Field(
        description="Relevant reference substance",
        default=None,
    )
    strength: Optional[Ratio] = Field(
        description="Strength expressed in terms of a reference substance",
        default=None,
    )
    strengthLowLimit: Optional[Ratio] = Field(
        description="Strength expressed in terms of a reference substance",
        default=None,
    )
    measurementPoint: Optional[String] = Field(
        description="For when strength is measured at a particular point or distance",
        default=None,
    )
    measurementPoint_ext: Optional[Element] = Field(
        description="Placeholder element for measurementPoint extensions",
        default=None,
        alias="_measurementPoint",
    )
    country: Optional[List[CodeableConcept]] = Field(
        description="The country or countries for which the strength range applies",
        default=None,
    )

    @field_validator(
        *(
            "country",
            "measurementPoint",
            "strengthLowLimit",
            "strength",
            "substance",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicinalProductIngredientSpecifiedSubstanceStrength(BackboneElement):
    """
    Quantity of the substance or specified substance present in the manufactured item or pharmaceutical product.
    """

    presentation: Optional[Ratio] = Field(
        description="The quantity of substance in the unit of presentation, or in the volume (or mass) of the single pharmaceutical product or manufactured item",
        default=None,
    )
    presentationLowLimit: Optional[Ratio] = Field(
        description="A lower limit for the quantity of substance in the unit of presentation. For use when there is a range of strengths, this is the lower limit, with the presentation attribute becoming the upper limit",
        default=None,
    )
    concentration: Optional[Ratio] = Field(
        description="The strength per unitary volume (or mass)",
        default=None,
    )
    concentrationLowLimit: Optional[Ratio] = Field(
        description="A lower limit for the strength per unitary volume (or mass), for when there is a range. The concentration attribute then becomes the upper limit",
        default=None,
    )
    measurementPoint: Optional[String] = Field(
        description="For when strength is measured at a particular point or distance",
        default=None,
    )
    measurementPoint_ext: Optional[Element] = Field(
        description="Placeholder element for measurementPoint extensions",
        default=None,
        alias="_measurementPoint",
    )
    country: Optional[List[CodeableConcept]] = Field(
        description="The country or countries for which the strength range applies",
        default=None,
    )
    referenceStrength: Optional[
        List[MedicinalProductIngredientSpecifiedSubstanceStrengthReferenceStrength]
    ] = Field(
        description="Strength expressed in terms of a reference substance",
        default=None,
    )

    @field_validator(
        *(
            "referenceStrength",
            "country",
            "measurementPoint",
            "concentrationLowLimit",
            "concentration",
            "presentationLowLimit",
            "presentation",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicinalProductIngredientSpecifiedSubstance(BackboneElement):
    """
    A specified substance that comprises this ingredient.
    """

    code: Optional[CodeableConcept] = Field(
        description="The specified substance",
        default=None,
    )
    group: Optional[CodeableConcept] = Field(
        description="The group of specified substance, e.g. group 1 to 4",
        default=None,
    )
    confidentiality: Optional[CodeableConcept] = Field(
        description="Confidentiality level of the specified substance as the ingredient",
        default=None,
    )
    strength: Optional[List[MedicinalProductIngredientSpecifiedSubstanceStrength]] = (
        Field(
            description="Quantity of the substance or specified substance present in the manufactured item or pharmaceutical product",
            default=None,
        )
    )

    @field_validator(
        *(
            "strength",
            "confidentiality",
            "group",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicinalProductIngredientSpecifiedSubstanceStrengthReferenceStrength(
    BackboneElement
):
    """
    Strength expressed in terms of a reference substance.
    """

    substance: Optional[CodeableConcept] = Field(
        description="Relevant reference substance",
        default=None,
    )
    strength: Optional[Ratio] = Field(
        description="Strength expressed in terms of a reference substance",
        default=None,
    )
    strengthLowLimit: Optional[Ratio] = Field(
        description="Strength expressed in terms of a reference substance",
        default=None,
    )
    measurementPoint: Optional[String] = Field(
        description="For when strength is measured at a particular point or distance",
        default=None,
    )
    measurementPoint_ext: Optional[Element] = Field(
        description="Placeholder element for measurementPoint extensions",
        default=None,
        alias="_measurementPoint",
    )
    country: Optional[List[CodeableConcept]] = Field(
        description="The country or countries for which the strength range applies",
        default=None,
    )

    @field_validator(
        *(
            "country",
            "measurementPoint",
            "strengthLowLimit",
            "strength",
            "substance",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicinalProductIngredientSubstanceStrength(BackboneElement):
    """
    Quantity of the substance or specified substance present in the manufactured item or pharmaceutical product.
    """

    presentation: Optional[Ratio] = Field(
        description="The quantity of substance in the unit of presentation, or in the volume (or mass) of the single pharmaceutical product or manufactured item",
        default=None,
    )
    presentationLowLimit: Optional[Ratio] = Field(
        description="A lower limit for the quantity of substance in the unit of presentation. For use when there is a range of strengths, this is the lower limit, with the presentation attribute becoming the upper limit",
        default=None,
    )
    concentration: Optional[Ratio] = Field(
        description="The strength per unitary volume (or mass)",
        default=None,
    )
    concentrationLowLimit: Optional[Ratio] = Field(
        description="A lower limit for the strength per unitary volume (or mass), for when there is a range. The concentration attribute then becomes the upper limit",
        default=None,
    )
    measurementPoint: Optional[String] = Field(
        description="For when strength is measured at a particular point or distance",
        default=None,
    )
    measurementPoint_ext: Optional[Element] = Field(
        description="Placeholder element for measurementPoint extensions",
        default=None,
        alias="_measurementPoint",
    )
    country: Optional[List[CodeableConcept]] = Field(
        description="The country or countries for which the strength range applies",
        default=None,
    )
    referenceStrength: Optional[
        List[MedicinalProductIngredientSpecifiedSubstanceStrengthReferenceStrength]
    ] = Field(
        description="Strength expressed in terms of a reference substance",
        default=None,
    )

    @field_validator(
        *(
            "referenceStrength",
            "country",
            "measurementPoint",
            "concentrationLowLimit",
            "concentration",
            "presentationLowLimit",
            "presentation",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicinalProductIngredientSubstance(BackboneElement):
    """
    The ingredient substance.
    """

    code: Optional[CodeableConcept] = Field(
        description="The ingredient substance",
        default=None,
    )
    strength: Optional[List[MedicinalProductIngredientSubstanceStrength]] = Field(
        description="Quantity of the substance or specified substance present in the manufactured item or pharmaceutical product",
        default=None,
    )

    @field_validator(
        *(
            "strength",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicinalProductIngredient(DomainResource):
    """
    An ingredient of a manufactured item or pharmaceutical product.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=[
                "http://hl7.org/fhir/StructureDefinition/MedicinalProductIngredient"
            ]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[Identifier] = Field(
        description="Identifier for the ingredient",
        default=None,
    )
    role: Optional[CodeableConcept] = Field(
        description="Ingredient role e.g. Active ingredient, excipient",
        default=None,
    )
    allergenicIndicator: Optional[Boolean] = Field(
        description="If the ingredient is a known or suspected allergen",
        default=None,
    )
    allergenicIndicator_ext: Optional[Element] = Field(
        description="Placeholder element for allergenicIndicator extensions",
        default=None,
        alias="_allergenicIndicator",
    )
    manufacturer: Optional[List[Reference]] = Field(
        description="Manufacturer of this Ingredient",
        default=None,
    )
    specifiedSubstance: Optional[List[MedicinalProductIngredientSpecifiedSubstance]] = (
        Field(
            description="A specified substance that comprises this ingredient",
            default=None,
        )
    )
    substance: Optional[MedicinalProductIngredientSubstance] = Field(
        description="The ingredient substance",
        default=None,
    )
    resourceType: Literal["MedicinalProductIngredient"] = Field(
        description=None,
        default="MedicinalProductIngredient",
    )

    @field_validator(
        *(
            "substance",
            "specifiedSubstance",
            "manufacturer",
            "allergenicIndicator",
            "role",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
