# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code, DateTime

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    CodeableConcept,
    Period,
    BackboneElement,
    DomainResource,
)


class MedicinalProductAuthorizationJurisdictionalAuthorization(BackboneElement):
    """
    Authorization in areas within a country.
    """

    identifier: Optional[List[Identifier]] = Field(
        description="The assigned number for the marketing authorization",
        default=None,
    )
    country: Optional[CodeableConcept] = Field(
        description="Country of authorization",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Jurisdiction within a country",
        default=None,
    )
    legalStatusOfSupply: Optional[CodeableConcept] = Field(
        description="The legal status of supply in a jurisdiction or region",
        default=None,
    )
    validityPeriod: Optional[Period] = Field(
        description="The start and expected end date of the authorization",
        default=None,
    )

    @field_validator(
        *(
            "validityPeriod",
            "legalStatusOfSupply",
            "jurisdiction",
            "country",
            "identifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicinalProductAuthorizationProcedure(BackboneElement):
    """
    The regulatory procedure for granting or amending a marketing authorization.
    """

    identifier: Optional[Identifier] = Field(
        description="Identifier for this procedure",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="Type of procedure",
        default=None,
    )
    datePeriod: Optional[Period] = Field(
        description="Date of procedure",
        default=None,
    )
    dateDateTime: Optional[DateTime] = Field(
        description="Date of procedure",
        default=None,
    )
    dateDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for dateDateTime extensions",
        default=None,
        alias="_dateDateTime",
    )
    application: Optional[List["MedicinalProductAuthorizationProcedure"]] = Field(
        description="Applcations submitted to obtain a marketing authorization",
        default=None,
    )

    @property
    def date(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="date",
        )

    @field_validator(
        *(
            "application",
            "type",
            "identifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def date_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Period, DateTime],
            field_name_base="date",
            required=False,
        )


class MedicinalProductAuthorization(DomainResource):
    """
    The regulatory authorization of a medicinal product.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=[
                "http://hl7.org/fhir/StructureDefinition/MedicinalProductAuthorization"
            ]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business identifier for the marketing authorization, as assigned by a regulator",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="The medicinal product that is being authorized",
        default=None,
    )
    country: Optional[List[CodeableConcept]] = Field(
        description="The country in which the marketing authorization has been granted",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Jurisdiction within a country",
        default=None,
    )
    status: Optional[CodeableConcept] = Field(
        description="The status of the marketing authorization",
        default=None,
    )
    statusDate: Optional[DateTime] = Field(
        description="The date at which the given status has become applicable",
        default=None,
    )
    statusDate_ext: Optional[Element] = Field(
        description="Placeholder element for statusDate extensions",
        default=None,
        alias="_statusDate",
    )
    restoreDate: Optional[DateTime] = Field(
        description="The date when a suspended the marketing or the marketing authorization of the product is anticipated to be restored",
        default=None,
    )
    restoreDate_ext: Optional[Element] = Field(
        description="Placeholder element for restoreDate extensions",
        default=None,
        alias="_restoreDate",
    )
    validityPeriod: Optional[Period] = Field(
        description="The beginning of the time period in which the marketing authorization is in the specific status shall be specified A complete date consisting of day, month and year shall be specified using the ISO 8601 date format",
        default=None,
    )
    dataExclusivityPeriod: Optional[Period] = Field(
        description="A period of time after authorization before generic product applicatiosn can be submitted",
        default=None,
    )
    dateOfFirstAuthorization: Optional[DateTime] = Field(
        description="The date when the first authorization was granted by a Medicines Regulatory Agency",
        default=None,
    )
    dateOfFirstAuthorization_ext: Optional[Element] = Field(
        description="Placeholder element for dateOfFirstAuthorization extensions",
        default=None,
        alias="_dateOfFirstAuthorization",
    )
    internationalBirthDate: Optional[DateTime] = Field(
        description="Date of first marketing authorization for a company\u0027s new medicinal product in any country in the World",
        default=None,
    )
    internationalBirthDate_ext: Optional[Element] = Field(
        description="Placeholder element for internationalBirthDate extensions",
        default=None,
        alias="_internationalBirthDate",
    )
    legalBasis: Optional[CodeableConcept] = Field(
        description="The legal framework against which this authorization is granted",
        default=None,
    )
    jurisdictionalAuthorization: Optional[
        List[MedicinalProductAuthorizationJurisdictionalAuthorization]
    ] = Field(
        description="Authorization in areas within a country",
        default=None,
    )
    holder: Optional[Reference] = Field(
        description="Marketing Authorization Holder",
        default=None,
    )
    regulator: Optional[Reference] = Field(
        description="Medicines Regulatory Agency",
        default=None,
    )
    procedure: Optional[MedicinalProductAuthorizationProcedure] = Field(
        description="The regulatory procedure for granting or amending a marketing authorization",
        default=None,
    )
    resourceType: Literal["MedicinalProductAuthorization"] = Field(
        description=None,
        default="MedicinalProductAuthorization",
    )

    @field_validator(
        *(
            "procedure",
            "regulator",
            "holder",
            "jurisdictionalAuthorization",
            "legalBasis",
            "internationalBirthDate",
            "dateOfFirstAuthorization",
            "dataExclusivityPeriod",
            "validityPeriod",
            "restoreDate",
            "statusDate",
            "status",
            "jurisdiction",
            "country",
            "subject",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
