# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code, DateTime

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Coding,
    MarketingStatus,
    Reference,
    BackboneElement,
    DomainResource,
)


class MedicinalProductNameNamePart(BackboneElement):
    """
    Coding words or phrases of the name.
    """

    part: Optional[String] = Field(
        description="A fragment of a product name",
        default=None,
    )
    part_ext: Optional[Element] = Field(
        description="Placeholder element for part extensions",
        default=None,
        alias="_part",
    )
    type: Optional[Coding] = Field(
        description="Idenifying type for this part of the name (e.g. strength part)",
        default=None,
    )

    @field_validator(
        *(
            "type",
            "part",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicinalProductNameCountryLanguage(BackboneElement):
    """
    Country where the name applies.
    """

    country: Optional[CodeableConcept] = Field(
        description="Country code for where this name applies",
        default=None,
    )
    jurisdiction: Optional[CodeableConcept] = Field(
        description="Jurisdiction code for where this name applies",
        default=None,
    )
    language: Optional[CodeableConcept] = Field(
        description="Language code for this name",
        default=None,
    )

    @field_validator(
        *(
            "language",
            "jurisdiction",
            "country",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicinalProductName(BackboneElement):
    """
    The product's name, including full name and possibly coded parts.
    """

    productName: Optional[String] = Field(
        description="The full product name",
        default=None,
    )
    productName_ext: Optional[Element] = Field(
        description="Placeholder element for productName extensions",
        default=None,
        alias="_productName",
    )
    namePart: Optional[List[MedicinalProductNameNamePart]] = Field(
        description="Coding words or phrases of the name",
        default=None,
    )
    countryLanguage: Optional[List[MedicinalProductNameCountryLanguage]] = Field(
        description="Country where the name applies",
        default=None,
    )

    @field_validator(
        *(
            "countryLanguage",
            "namePart",
            "productName",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicinalProductManufacturingBusinessOperation(BackboneElement):
    """
    An operation applied to the product, for manufacturing or adminsitrative purpose.
    """

    operationType: Optional[CodeableConcept] = Field(
        description="The type of manufacturing operation",
        default=None,
    )
    authorisationReferenceNumber: Optional[Identifier] = Field(
        description="Regulatory authorization reference number",
        default=None,
    )
    effectiveDate: Optional[DateTime] = Field(
        description="Regulatory authorization date",
        default=None,
    )
    effectiveDate_ext: Optional[Element] = Field(
        description="Placeholder element for effectiveDate extensions",
        default=None,
        alias="_effectiveDate",
    )
    confidentialityIndicator: Optional[CodeableConcept] = Field(
        description="To indicate if this proces is commercially confidential",
        default=None,
    )
    manufacturer: Optional[List[Reference]] = Field(
        description="The manufacturer or establishment associated with the process",
        default=None,
    )
    regulator: Optional[Reference] = Field(
        description="A regulator which oversees the operation",
        default=None,
    )

    @field_validator(
        *(
            "regulator",
            "manufacturer",
            "confidentialityIndicator",
            "effectiveDate",
            "authorisationReferenceNumber",
            "operationType",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicinalProductSpecialDesignation(BackboneElement):
    """
    Indicates if the medicinal product has an orphan designation for the treatment of a rare disease.
    """

    identifier: Optional[List[Identifier]] = Field(
        description="Identifier for the designation, or procedure number",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="The type of special designation, e.g. orphan drug, minor use",
        default=None,
    )
    intendedUse: Optional[CodeableConcept] = Field(
        description="The intended use of the product, e.g. prevention, treatment",
        default=None,
    )
    indicationCodeableConcept: Optional[CodeableConcept] = Field(
        description="Condition for which the medicinal use applies",
        default=None,
    )
    indicationReference: Optional[Reference] = Field(
        description="Condition for which the medicinal use applies",
        default=None,
    )
    status: Optional[CodeableConcept] = Field(
        description="For example granted, pending, expired or withdrawn",
        default=None,
    )
    date: Optional[DateTime] = Field(
        description="Date when the designation was granted",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    species: Optional[CodeableConcept] = Field(
        description="Animal species for which this applies",
        default=None,
    )

    @property
    def indication(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="indication",
        )

    @field_validator(
        *(
            "species",
            "date",
            "status",
            "intendedUse",
            "type",
            "identifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def indication_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Reference],
            field_name_base="indication",
            required=False,
        )


class MedicinalProduct(DomainResource):
    """
    Detailed definition of a medicinal product, typically for uses other than direct patient care (e.g. regulatory use).
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/MedicinalProduct"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business identifier for this product. Could be an MPID",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="Regulatory type, e.g. Investigational or Authorized",
        default=None,
    )
    domain: Optional[Coding] = Field(
        description="If this medicine applies to human or veterinary uses",
        default=None,
    )
    combinedPharmaceuticalDoseForm: Optional[CodeableConcept] = Field(
        description="The dose form for a single part product, or combined form of a multiple part product",
        default=None,
    )
    legalStatusOfSupply: Optional[CodeableConcept] = Field(
        description="The legal status of supply of the medicinal product as classified by the regulator",
        default=None,
    )
    additionalMonitoringIndicator: Optional[CodeableConcept] = Field(
        description="Whether the Medicinal Product is subject to additional monitoring for regulatory reasons",
        default=None,
    )
    specialMeasures: Optional[List[String]] = Field(
        description="Whether the Medicinal Product is subject to special measures for regulatory reasons",
        default=None,
    )
    specialMeasures_ext: Optional[Element] = Field(
        description="Placeholder element for specialMeasures extensions",
        default=None,
        alias="_specialMeasures",
    )
    paediatricUseIndicator: Optional[CodeableConcept] = Field(
        description="If authorised for use in children",
        default=None,
    )
    productClassification: Optional[List[CodeableConcept]] = Field(
        description="Allows the product to be classified by various systems",
        default=None,
    )
    marketingStatus: Optional[List[MarketingStatus]] = Field(
        description="Marketing status of the medicinal product, in contrast to marketing authorizaton",
        default=None,
    )
    pharmaceuticalProduct: Optional[List[Reference]] = Field(
        description="Pharmaceutical aspects of product",
        default=None,
    )
    packagedMedicinalProduct: Optional[List[Reference]] = Field(
        description="Package representation for the product",
        default=None,
    )
    attachedDocument: Optional[List[Reference]] = Field(
        description="Supporting documentation, typically for regulatory submission",
        default=None,
    )
    masterFile: Optional[List[Reference]] = Field(
        description="A master file for to the medicinal product (e.g. Pharmacovigilance System Master File)",
        default=None,
    )
    contact: Optional[List[Reference]] = Field(
        description="A product specific contact, person (in a role), or an organization",
        default=None,
    )
    clinicalTrial: Optional[List[Reference]] = Field(
        description="Clinical trials or studies that this product is involved in",
        default=None,
    )
    name: Optional[List[MedicinalProductName]] = Field(
        description="The product\u0027s name, including full name and possibly coded parts",
        default=None,
    )
    crossReference: Optional[List[Identifier]] = Field(
        description="Reference to another product, e.g. for linking authorised to investigational product",
        default=None,
    )
    manufacturingBusinessOperation: Optional[
        List[MedicinalProductManufacturingBusinessOperation]
    ] = Field(
        description="An operation applied to the product, for manufacturing or adminsitrative purpose",
        default=None,
    )
    specialDesignation: Optional[List[MedicinalProductSpecialDesignation]] = Field(
        description="Indicates if the medicinal product has an orphan designation for the treatment of a rare disease",
        default=None,
    )
    resourceType: Literal["MedicinalProduct"] = Field(
        description=None,
        default="MedicinalProduct",
    )

    @field_validator(
        *(
            "specialDesignation",
            "manufacturingBusinessOperation",
            "crossReference",
            "name",
            "clinicalTrial",
            "contact",
            "masterFile",
            "attachedDocument",
            "packagedMedicinalProduct",
            "pharmaceuticalProduct",
            "marketingStatus",
            "productClassification",
            "paediatricUseIndicator",
            "specialMeasures",
            "additionalMonitoringIndicator",
            "legalStatusOfSupply",
            "combinedPharmaceuticalDoseForm",
            "domain",
            "type",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
