# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    Markdown,
    Base64Binary,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    CodeableConcept,
    Reference,
    Quantity,
    BackboneElement,
    Ratio,
    Money,
    Dosage,
    Duration,
    DomainResource,
)


class MedicationKnowledgeRelatedMedicationKnowledge(BackboneElement):
    """
    Associated or related knowledge about a medication.
    """

    type: Optional[CodeableConcept] = Field(
        description="Category of medicationKnowledge",
        default=None,
    )
    reference: Optional[List[Reference]] = Field(
        description="Associated documentation about the associated medication knowledge",
        default=None,
    )

    @field_validator(
        *(
            "reference",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeMonograph(BackboneElement):
    """
    Associated documentation about the medication.
    """

    type: Optional[CodeableConcept] = Field(
        description="The category of medication document",
        default=None,
    )
    source: Optional[Reference] = Field(
        description="Associated documentation about the medication",
        default=None,
    )

    @field_validator(
        *(
            "source",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeIngredient(BackboneElement):
    """
    Identifies a particular constituent of interest in the product.
    """

    itemCodeableConcept: Optional[CodeableConcept] = Field(
        description="Medication(s) or substance(s) contained in the medication",
        default=None,
    )
    itemReference: Optional[Reference] = Field(
        description="Medication(s) or substance(s) contained in the medication",
        default=None,
    )
    isActive: Optional[Boolean] = Field(
        description="Active ingredient indicator",
        default=None,
    )
    isActive_ext: Optional[Element] = Field(
        description="Placeholder element for isActive extensions",
        default=None,
        alias="_isActive",
    )
    strength: Optional[Ratio] = Field(
        description="Quantity of ingredient present",
        default=None,
    )

    @property
    def item(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="item",
        )

    @field_validator(
        *(
            "strength",
            "isActive",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def item_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Reference],
            field_name_base="item",
            required=True,
        )


class MedicationKnowledgeCost(BackboneElement):
    """
    The price of the medication.
    """

    type: Optional[CodeableConcept] = Field(
        description="The category of the cost information",
        default=None,
    )
    source: Optional[String] = Field(
        description="The source or owner for the price information",
        default=None,
    )
    source_ext: Optional[Element] = Field(
        description="Placeholder element for source extensions",
        default=None,
        alias="_source",
    )
    cost: Optional[Money] = Field(
        description="The price of the medication",
        default=None,
    )

    @field_validator(
        *(
            "cost",
            "source",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeMonitoringProgram(BackboneElement):
    """
    The program under which the medication is reviewed.
    """

    type: Optional[CodeableConcept] = Field(
        description="Type of program under which the medication is monitored",
        default=None,
    )
    name: Optional[String] = Field(
        description="Name of the reviewing program",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )

    @field_validator(
        *(
            "name",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeAdministrationGuidelinesDosage(BackboneElement):
    """
    Dosage for the medication for the specific guidelines.
    """

    type: Optional[CodeableConcept] = Field(
        description="Type of dosage",
        default=None,
    )
    dosage: Optional[List[Dosage]] = Field(
        description="Dosage for the medication for the specific guidelines",
        default=None,
    )

    @field_validator(
        *(
            "dosage",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeAdministrationGuidelinesPatientCharacteristics(
    BackboneElement
):
    """
    Characteristics of the patient that are relevant to the administration guidelines (for example, height, weight, gender, etc.).
    """

    characteristicCodeableConcept: Optional[CodeableConcept] = Field(
        description="Specific characteristic that is relevant to the administration guideline",
        default=None,
    )
    characteristicQuantity: Optional[Quantity] = Field(
        description="Specific characteristic that is relevant to the administration guideline",
        default=None,
    )
    value: Optional[List[String]] = Field(
        description="The specific characteristic",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @property
    def characteristic(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="characteristic",
        )

    @field_validator(
        *("value", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def characteristic_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Quantity],
            field_name_base="characteristic",
            required=True,
        )


class MedicationKnowledgeAdministrationGuidelines(BackboneElement):
    """
    Guidelines for the administration of the medication.
    """

    dosage: Optional[List[MedicationKnowledgeAdministrationGuidelinesDosage]] = Field(
        description="Dosage for the medication for the specific guidelines",
        default=None,
    )
    indicationCodeableConcept: Optional[CodeableConcept] = Field(
        description="Indication for use that apply to the specific administration guidelines",
        default=None,
    )
    indicationReference: Optional[Reference] = Field(
        description="Indication for use that apply to the specific administration guidelines",
        default=None,
    )
    patientCharacteristics: Optional[
        List[MedicationKnowledgeAdministrationGuidelinesPatientCharacteristics]
    ] = Field(
        description="Characteristics of the patient that are relevant to the administration guidelines",
        default=None,
    )

    @property
    def indication(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="indication",
        )

    @field_validator(
        *(
            "patientCharacteristics",
            "dosage",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def indication_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Reference],
            field_name_base="indication",
            required=False,
        )


class MedicationKnowledgeMedicineClassification(BackboneElement):
    """
    Categorization of the medication within a formulary or classification system.
    """

    type: Optional[CodeableConcept] = Field(
        description="The type of category for the medication (for example, therapeutic classification, therapeutic sub-classification)",
        default=None,
    )
    classification: Optional[List[CodeableConcept]] = Field(
        description="Specific category assigned to the medication",
        default=None,
    )

    @field_validator(
        *(
            "classification",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgePackaging(BackboneElement):
    """
    Information that only applies to packages (not products).
    """

    type: Optional[CodeableConcept] = Field(
        description="A code that defines the specific type of packaging that the medication can be found in",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="The number of product units the package would contain if fully loaded",
        default=None,
    )

    @field_validator(
        *(
            "quantity",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeDrugCharacteristic(BackboneElement):
    """
    Specifies descriptive properties of the medicine, such as color, shape, imprints, etc.
    """

    type: Optional[CodeableConcept] = Field(
        description="Code specifying the type of characteristic of medication",
        default=None,
    )
    valueCodeableConcept: Optional[CodeableConcept] = Field(
        description="Description of the characteristic",
        default=None,
    )
    valueString: Optional[String] = Field(
        description="Description of the characteristic",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueQuantity: Optional[Quantity] = Field(
        description="Description of the characteristic",
        default=None,
    )
    valueBase64Binary: Optional[Base64Binary] = Field(
        description="Description of the characteristic",
        default=None,
    )
    valueBase64Binary_ext: Optional[Element] = Field(
        description="Placeholder element for valueBase64Binary extensions",
        default=None,
        alias="_valueBase64Binary",
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("type", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, String, Quantity, Base64Binary],
            field_name_base="value",
            required=False,
        )


class MedicationKnowledgeRegulatorySubstitution(BackboneElement):
    """
    Specifies if changes are allowed when dispensing a medication from a regulatory perspective.
    """

    type: Optional[CodeableConcept] = Field(
        description="Specifies the type of substitution allowed",
        default=None,
    )
    allowed: Optional[Boolean] = Field(
        description="Specifies if regulation allows for changes in the medication when dispensing",
        default=None,
    )
    allowed_ext: Optional[Element] = Field(
        description="Placeholder element for allowed extensions",
        default=None,
        alias="_allowed",
    )

    @field_validator(
        *(
            "allowed",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeRegulatorySchedule(BackboneElement):
    """
    Specifies the schedule of a medication in jurisdiction.
    """

    schedule: Optional[CodeableConcept] = Field(
        description="Specifies the specific drug schedule",
        default=None,
    )

    @field_validator(
        *("schedule", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeRegulatoryMaxDispense(BackboneElement):
    """
    The maximum number of units of the medication that can be dispensed in a period.
    """

    quantity: Optional[Quantity] = Field(
        description="The maximum number of units of the medication that can be dispensed",
        default=None,
    )
    period: Optional[Duration] = Field(
        description="The period that applies to the maximum number of units",
        default=None,
    )

    @field_validator(
        *(
            "period",
            "quantity",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeRegulatory(BackboneElement):
    """
    Regulatory information about a medication.
    """

    regulatoryAuthority: Optional[Reference] = Field(
        description="Specifies the authority of the regulation",
        default=None,
    )
    substitution: Optional[List[MedicationKnowledgeRegulatorySubstitution]] = Field(
        description="Specifies if changes are allowed when dispensing a medication from a regulatory perspective",
        default=None,
    )
    schedule: Optional[List[MedicationKnowledgeRegulatorySchedule]] = Field(
        description="Specifies the schedule of a medication in jurisdiction",
        default=None,
    )
    maxDispense: Optional[MedicationKnowledgeRegulatoryMaxDispense] = Field(
        description="The maximum number of units of the medication that can be dispensed in a period",
        default=None,
    )

    @field_validator(
        *(
            "maxDispense",
            "schedule",
            "substitution",
            "regulatoryAuthority",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledgeKinetics(BackboneElement):
    """
    The time course of drug absorption, distribution, metabolism and excretion of a medication from the body.
    """

    areaUnderCurve: Optional[List[Quantity]] = Field(
        description="The drug concentration measured at certain discrete points in time",
        default=None,
    )
    lethalDose50: Optional[List[Quantity]] = Field(
        description="The median lethal dose of a drug",
        default=None,
    )
    halfLifePeriod: Optional[Duration] = Field(
        description="Time required for concentration in the body to decrease by half",
        default=None,
    )

    @field_validator(
        *(
            "halfLifePeriod",
            "lethalDose50",
            "areaUnderCurve",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class MedicationKnowledge(DomainResource):
    """
    Information about a medication that is used to support knowledge.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/MedicationKnowledge"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    code: Optional[CodeableConcept] = Field(
        description="Code that identifies this medication",
        default=None,
    )
    status: Optional[Code] = Field(
        description="active | inactive | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    manufacturer: Optional[Reference] = Field(
        description="Manufacturer of the item",
        default=None,
    )
    doseForm: Optional[CodeableConcept] = Field(
        description="powder | tablets | capsule +",
        default=None,
    )
    amount: Optional[Quantity] = Field(
        description="Amount of drug in package",
        default=None,
    )
    synonym: Optional[List[String]] = Field(
        description="Additional names for a medication",
        default=None,
    )
    synonym_ext: Optional[Element] = Field(
        description="Placeholder element for synonym extensions",
        default=None,
        alias="_synonym",
    )
    relatedMedicationKnowledge: Optional[
        List[MedicationKnowledgeRelatedMedicationKnowledge]
    ] = Field(
        description="Associated or related medication information",
        default=None,
    )
    associatedMedication: Optional[List[Reference]] = Field(
        description="A medication resource that is associated with this medication",
        default=None,
    )
    productType: Optional[List[CodeableConcept]] = Field(
        description="Category of the medication or product",
        default=None,
    )
    monograph: Optional[List[MedicationKnowledgeMonograph]] = Field(
        description="Associated documentation about the medication",
        default=None,
    )
    ingredient: Optional[List[MedicationKnowledgeIngredient]] = Field(
        description="Active or inactive ingredient",
        default=None,
    )
    preparationInstruction: Optional[Markdown] = Field(
        description="The instructions for preparing the medication",
        default=None,
    )
    preparationInstruction_ext: Optional[Element] = Field(
        description="Placeholder element for preparationInstruction extensions",
        default=None,
        alias="_preparationInstruction",
    )
    intendedRoute: Optional[List[CodeableConcept]] = Field(
        description="The intended or approved route of administration",
        default=None,
    )
    cost: Optional[List[MedicationKnowledgeCost]] = Field(
        description="The pricing of the medication",
        default=None,
    )
    monitoringProgram: Optional[List[MedicationKnowledgeMonitoringProgram]] = Field(
        description="Program under which a medication is reviewed",
        default=None,
    )
    administrationGuidelines: Optional[
        List[MedicationKnowledgeAdministrationGuidelines]
    ] = Field(
        description="Guidelines for administration of the medication",
        default=None,
    )
    medicineClassification: Optional[
        List[MedicationKnowledgeMedicineClassification]
    ] = Field(
        description="Categorization of the medication within a formulary or classification system",
        default=None,
    )
    packaging: Optional[MedicationKnowledgePackaging] = Field(
        description="Details about packaged medications",
        default=None,
    )
    drugCharacteristic: Optional[List[MedicationKnowledgeDrugCharacteristic]] = Field(
        description="Specifies descriptive properties of the medicine",
        default=None,
    )
    contraindication: Optional[List[Reference]] = Field(
        description="Potential clinical issue with or between medication(s)",
        default=None,
    )
    regulatory: Optional[List[MedicationKnowledgeRegulatory]] = Field(
        description="Regulatory information about a medication",
        default=None,
    )
    kinetics: Optional[List[MedicationKnowledgeKinetics]] = Field(
        description="The time course of drug absorption, distribution, metabolism and excretion of a medication from the body",
        default=None,
    )
    resourceType: Literal["MedicationKnowledge"] = Field(
        description=None,
        default="MedicationKnowledge",
    )

    @field_validator(
        *(
            "kinetics",
            "regulatory",
            "contraindication",
            "drugCharacteristic",
            "packaging",
            "medicineClassification",
            "administrationGuidelines",
            "monitoringProgram",
            "cost",
            "intendedRoute",
            "preparationInstruction",
            "ingredient",
            "monograph",
            "productType",
            "associatedMedication",
            "relatedMedicationKnowledge",
            "synonym",
            "amount",
            "doseForm",
            "manufacturer",
            "status",
            "code",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
