# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    Instant,
    PositiveInt,
    Decimal,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    CodeableConcept,
    Period,
    Attachment,
    Annotation,
    DomainResource,
)


class Media(DomainResource):
    """
    A photo, video, or audio recording acquired or used in healthcare. The actual content may be inline or provided by direct reference.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Media"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Identifier(s) for the image",
        default=None,
    )
    basedOn: Optional[List[Reference]] = Field(
        description="Procedure that caused this media to be created",
        default=None,
    )
    partOf: Optional[List[Reference]] = Field(
        description="Part of referenced event",
        default=None,
    )
    status: Optional[Code] = Field(
        description="preparation | in-progress | not-done | on-hold | stopped | completed | entered-in-error | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    type: Optional[CodeableConcept] = Field(
        description="Classification of media as image, video, or audio",
        default=None,
    )
    modality: Optional[CodeableConcept] = Field(
        description="The type of acquisition equipment/process",
        default=None,
    )
    view: Optional[CodeableConcept] = Field(
        description="Imaging view, e.g. Lateral or Antero-posterior",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="Who/What this Media is a record of",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Encounter associated with media",
        default=None,
    )
    createdDateTime: Optional[DateTime] = Field(
        description="When Media was collected",
        default=None,
    )
    createdDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for createdDateTime extensions",
        default=None,
        alias="_createdDateTime",
    )
    createdPeriod: Optional[Period] = Field(
        description="When Media was collected",
        default=None,
    )
    issued: Optional[Instant] = Field(
        description="Date/Time this version was made available",
        default=None,
    )
    issued_ext: Optional[Element] = Field(
        description="Placeholder element for issued extensions",
        default=None,
        alias="_issued",
    )
    operator: Optional[Reference] = Field(
        description="The person who generated the image",
        default=None,
    )
    reasonCode: Optional[List[CodeableConcept]] = Field(
        description="Why was event performed?",
        default=None,
    )
    bodySite: Optional[CodeableConcept] = Field(
        description="Observed body part",
        default=None,
    )
    deviceName: Optional[String] = Field(
        description="Name of the device/manufacturer",
        default=None,
    )
    deviceName_ext: Optional[Element] = Field(
        description="Placeholder element for deviceName extensions",
        default=None,
        alias="_deviceName",
    )
    device: Optional[Reference] = Field(
        description="Observing Device",
        default=None,
    )
    height: Optional[PositiveInt] = Field(
        description="Height of the image in pixels (photo/video)",
        default=None,
    )
    height_ext: Optional[Element] = Field(
        description="Placeholder element for height extensions",
        default=None,
        alias="_height",
    )
    width: Optional[PositiveInt] = Field(
        description="Width of the image in pixels (photo/video)",
        default=None,
    )
    width_ext: Optional[Element] = Field(
        description="Placeholder element for width extensions",
        default=None,
        alias="_width",
    )
    frames: Optional[PositiveInt] = Field(
        description="Number of frames if \u003e 1 (photo)",
        default=None,
    )
    frames_ext: Optional[Element] = Field(
        description="Placeholder element for frames extensions",
        default=None,
        alias="_frames",
    )
    duration: Optional[Decimal] = Field(
        description="Length in seconds (audio / video)",
        default=None,
    )
    duration_ext: Optional[Element] = Field(
        description="Placeholder element for duration extensions",
        default=None,
        alias="_duration",
    )
    content: Optional[Attachment] = Field(
        description="Actual Media - reference or data",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Comments made about the media",
        default=None,
    )
    resourceType: Literal["Media"] = Field(
        description=None,
        default="Media",
    )

    @property
    def created(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="created",
        )

    @field_validator(
        *(
            "note",
            "content",
            "duration",
            "frames",
            "width",
            "height",
            "device",
            "deviceName",
            "bodySite",
            "reasonCode",
            "operator",
            "issued",
            "encounter",
            "subject",
            "view",
            "modality",
            "type",
            "status",
            "partOf",
            "basedOn",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def created_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, Period],
            field_name_base="created",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
