# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    Boolean,
    Date,
    PositiveInt,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    Quantity,
    BackboneElement,
    Annotation,
    DomainResource,
)


class ImmunizationPerformer(BackboneElement):
    """
    Indicates who performed the immunization event.
    """

    function: Optional[CodeableConcept] = Field(
        description="What type of performance was done",
        default=None,
    )
    actor: Optional[Reference] = Field(
        description="Individual or organization who was performing",
        default=None,
    )

    @field_validator(
        *(
            "actor",
            "function",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImmunizationEducation(BackboneElement):
    """
    Educational material presented to the patient (or guardian) at the time of vaccine administration.
    """

    documentType: Optional[String] = Field(
        description="Educational material document identifier",
        default=None,
    )
    documentType_ext: Optional[Element] = Field(
        description="Placeholder element for documentType extensions",
        default=None,
        alias="_documentType",
    )
    reference: Optional[Uri] = Field(
        description="Educational material reference pointer",
        default=None,
    )
    reference_ext: Optional[Element] = Field(
        description="Placeholder element for reference extensions",
        default=None,
        alias="_reference",
    )
    publicationDate: Optional[DateTime] = Field(
        description="Educational material publication date",
        default=None,
    )
    publicationDate_ext: Optional[Element] = Field(
        description="Placeholder element for publicationDate extensions",
        default=None,
        alias="_publicationDate",
    )
    presentationDate: Optional[DateTime] = Field(
        description="Educational material presentation date",
        default=None,
    )
    presentationDate_ext: Optional[Element] = Field(
        description="Placeholder element for presentationDate extensions",
        default=None,
        alias="_presentationDate",
    )

    @field_validator(
        *(
            "presentationDate",
            "publicationDate",
            "reference",
            "documentType",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImmunizationReaction(BackboneElement):
    """
    Categorical data indicating that an adverse event is associated in time to an immunization.
    """

    date: Optional[DateTime] = Field(
        description="When reaction started",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    detail: Optional[Reference] = Field(
        description="Additional information on reaction",
        default=None,
    )
    reported: Optional[Boolean] = Field(
        description="Indicates self-reported reaction",
        default=None,
    )
    reported_ext: Optional[Element] = Field(
        description="Placeholder element for reported extensions",
        default=None,
        alias="_reported",
    )

    @field_validator(
        *(
            "reported",
            "detail",
            "date",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ImmunizationProtocolApplied(BackboneElement):
    """
    The protocol (set of recommendations) being followed by the provider who administered the dose.
    """

    series: Optional[String] = Field(
        description="Name of vaccine series",
        default=None,
    )
    series_ext: Optional[Element] = Field(
        description="Placeholder element for series extensions",
        default=None,
        alias="_series",
    )
    authority: Optional[Reference] = Field(
        description="Who is responsible for publishing the recommendations",
        default=None,
    )
    targetDisease: Optional[List[CodeableConcept]] = Field(
        description="Vaccine preventatable disease being targetted",
        default=None,
    )
    doseNumberPositiveInt: Optional[PositiveInt] = Field(
        description="Dose number within series",
        default=None,
    )
    doseNumberPositiveInt_ext: Optional[Element] = Field(
        description="Placeholder element for doseNumberPositiveInt extensions",
        default=None,
        alias="_doseNumberPositiveInt",
    )
    doseNumberString: Optional[String] = Field(
        description="Dose number within series",
        default=None,
    )
    doseNumberString_ext: Optional[Element] = Field(
        description="Placeholder element for doseNumberString extensions",
        default=None,
        alias="_doseNumberString",
    )
    seriesDosesPositiveInt: Optional[PositiveInt] = Field(
        description="Recommended number of doses for immunity",
        default=None,
    )
    seriesDosesPositiveInt_ext: Optional[Element] = Field(
        description="Placeholder element for seriesDosesPositiveInt extensions",
        default=None,
        alias="_seriesDosesPositiveInt",
    )
    seriesDosesString: Optional[String] = Field(
        description="Recommended number of doses for immunity",
        default=None,
    )
    seriesDosesString_ext: Optional[Element] = Field(
        description="Placeholder element for seriesDosesString extensions",
        default=None,
        alias="_seriesDosesString",
    )

    @property
    def doseNumber(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="doseNumber",
        )

    @property
    def seriesDoses(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="seriesDoses",
        )

    @field_validator(
        *(
            "targetDisease",
            "authority",
            "series",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def doseNumber_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[PositiveInt, String],
            field_name_base="doseNumber",
            required=True,
        )

    @model_validator(mode="after")
    def seriesDoses_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[PositiveInt, String],
            field_name_base="seriesDoses",
            required=False,
        )


class Immunization(DomainResource):
    """
    Describes the event of a patient being administered a vaccine or a record of an immunization as reported by a patient, a clinician or another party.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Immunization"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business identifier",
        default=None,
    )
    status: Optional[Code] = Field(
        description="completed | entered-in-error | not-done",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    statusReason: Optional[CodeableConcept] = Field(
        description="Reason not done",
        default=None,
    )
    vaccineCode: Optional[CodeableConcept] = Field(
        description="Vaccine product administered",
        default=None,
    )
    patient: Optional[Reference] = Field(
        description="Who was immunized",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Encounter immunization was part of",
        default=None,
    )
    occurrenceDateTime: Optional[DateTime] = Field(
        description="Vaccine administration date",
        default=None,
    )
    occurrenceDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for occurrenceDateTime extensions",
        default=None,
        alias="_occurrenceDateTime",
    )
    occurrenceString: Optional[String] = Field(
        description="Vaccine administration date",
        default=None,
    )
    occurrenceString_ext: Optional[Element] = Field(
        description="Placeholder element for occurrenceString extensions",
        default=None,
        alias="_occurrenceString",
    )
    recorded: Optional[DateTime] = Field(
        description="When the immunization was first captured in the subject\u0027s record",
        default=None,
    )
    recorded_ext: Optional[Element] = Field(
        description="Placeholder element for recorded extensions",
        default=None,
        alias="_recorded",
    )
    primarySource: Optional[Boolean] = Field(
        description="Indicates context the data was recorded in",
        default=None,
    )
    primarySource_ext: Optional[Element] = Field(
        description="Placeholder element for primarySource extensions",
        default=None,
        alias="_primarySource",
    )
    reportOrigin: Optional[CodeableConcept] = Field(
        description="Indicates the source of a secondarily reported record",
        default=None,
    )
    location: Optional[Reference] = Field(
        description="Where immunization occurred",
        default=None,
    )
    manufacturer: Optional[Reference] = Field(
        description="Vaccine manufacturer",
        default=None,
    )
    lotNumber: Optional[String] = Field(
        description="Vaccine lot number",
        default=None,
    )
    lotNumber_ext: Optional[Element] = Field(
        description="Placeholder element for lotNumber extensions",
        default=None,
        alias="_lotNumber",
    )
    expirationDate: Optional[Date] = Field(
        description="Vaccine expiration date",
        default=None,
    )
    expirationDate_ext: Optional[Element] = Field(
        description="Placeholder element for expirationDate extensions",
        default=None,
        alias="_expirationDate",
    )
    site: Optional[CodeableConcept] = Field(
        description="Body site vaccine  was administered",
        default=None,
    )
    route: Optional[CodeableConcept] = Field(
        description="How vaccine entered body",
        default=None,
    )
    doseQuantity: Optional[Quantity] = Field(
        description="Amount of vaccine administered",
        default=None,
    )
    performer: Optional[List[ImmunizationPerformer]] = Field(
        description="Who performed event",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Additional immunization notes",
        default=None,
    )
    reasonCode: Optional[List[CodeableConcept]] = Field(
        description="Why immunization occurred",
        default=None,
    )
    reasonReference: Optional[List[Reference]] = Field(
        description="Why immunization occurred",
        default=None,
    )
    isSubpotent: Optional[Boolean] = Field(
        description="Dose potency",
        default=None,
    )
    isSubpotent_ext: Optional[Element] = Field(
        description="Placeholder element for isSubpotent extensions",
        default=None,
        alias="_isSubpotent",
    )
    subpotentReason: Optional[List[CodeableConcept]] = Field(
        description="Reason for being subpotent",
        default=None,
    )
    education: Optional[List[ImmunizationEducation]] = Field(
        description="Educational material presented to patient",
        default=None,
    )
    programEligibility: Optional[List[CodeableConcept]] = Field(
        description="Patient eligibility for a vaccination program",
        default=None,
    )
    fundingSource: Optional[CodeableConcept] = Field(
        description="Funding source for the vaccine",
        default=None,
    )
    reaction: Optional[List[ImmunizationReaction]] = Field(
        description="Details of a reaction that follows immunization",
        default=None,
    )
    protocolApplied: Optional[List[ImmunizationProtocolApplied]] = Field(
        description="Protocol followed by the provider",
        default=None,
    )
    resourceType: Literal["Immunization"] = Field(
        description=None,
        default="Immunization",
    )

    @property
    def occurrence(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="occurrence",
        )

    @field_validator(
        *(
            "protocolApplied",
            "reaction",
            "fundingSource",
            "programEligibility",
            "education",
            "subpotentReason",
            "isSubpotent",
            "reasonReference",
            "reasonCode",
            "note",
            "performer",
            "doseQuantity",
            "route",
            "site",
            "expirationDate",
            "lotNumber",
            "manufacturer",
            "location",
            "reportOrigin",
            "primarySource",
            "recorded",
            "encounter",
            "patient",
            "vaccineCode",
            "statusReason",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("education",), mode="after", check_fields=None)
    @classmethod
    def FHIR_imm_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="documentType.exists() or reference.exists()",
            human="One of documentType or reference SHALL be present",
            key="imm-1",
            severity="error",
        )

    @model_validator(mode="after")
    def occurrence_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, String],
            field_name_base="occurrence",
            required=True,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
