# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Date,
    Boolean,
    Integer,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    BackboneElement,
    Quantity,
    Range,
    Ratio,
    Duration,
    Annotation,
    DomainResource,
)


class GoalTarget(BackboneElement):
    """
    Indicates what should be done by when.
    """

    measure: Optional[CodeableConcept] = Field(
        description="The parameter whose value is being tracked",
        default=None,
    )
    detailQuantity: Optional[Quantity] = Field(
        description="The target value to be achieved",
        default=None,
    )
    detailRange: Optional[Range] = Field(
        description="The target value to be achieved",
        default=None,
    )
    detailCodeableConcept: Optional[CodeableConcept] = Field(
        description="The target value to be achieved",
        default=None,
    )
    detailString: Optional[String] = Field(
        description="The target value to be achieved",
        default=None,
    )
    detailString_ext: Optional[Element] = Field(
        description="Placeholder element for detailString extensions",
        default=None,
        alias="_detailString",
    )
    detailBoolean: Optional[Boolean] = Field(
        description="The target value to be achieved",
        default=None,
    )
    detailBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for detailBoolean extensions",
        default=None,
        alias="_detailBoolean",
    )
    detailInteger: Optional[Integer] = Field(
        description="The target value to be achieved",
        default=None,
    )
    detailInteger_ext: Optional[Element] = Field(
        description="Placeholder element for detailInteger extensions",
        default=None,
        alias="_detailInteger",
    )
    detailRatio: Optional[Ratio] = Field(
        description="The target value to be achieved",
        default=None,
    )
    dueDate: Optional[Date] = Field(
        description="Reach goal on or before",
        default=None,
    )
    dueDate_ext: Optional[Element] = Field(
        description="Placeholder element for dueDate extensions",
        default=None,
        alias="_dueDate",
    )
    dueDuration: Optional[Duration] = Field(
        description="Reach goal on or before",
        default=None,
    )

    @property
    def detail(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="detail",
        )

    @property
    def due(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="due",
        )

    @field_validator(
        *("measure", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def detail_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[
                Quantity,
                Range,
                CodeableConcept,
                String,
                Boolean,
                Integer,
                Ratio,
            ],
            field_name_base="detail",
            required=False,
        )

    @model_validator(mode="after")
    def due_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Date, Duration],
            field_name_base="due",
            required=False,
        )


class Goal(DomainResource):
    """
    Describes the intended objective(s) for a patient, group or organization care, for example, weight loss, restoring an activity of daily living, obtaining herd immunity via immunization, meeting a process improvement objective, etc.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Goal"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="External Ids for this goal",
        default=None,
    )
    lifecycleStatus: Optional[Code] = Field(
        description="proposed | planned | accepted | active | on-hold | completed | cancelled | entered-in-error | rejected",
        default=None,
    )
    lifecycleStatus_ext: Optional[Element] = Field(
        description="Placeholder element for lifecycleStatus extensions",
        default=None,
        alias="_lifecycleStatus",
    )
    achievementStatus: Optional[CodeableConcept] = Field(
        description="in-progress | improving | worsening | no-change | achieved | sustaining | not-achieved | no-progress | not-attainable",
        default=None,
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="E.g. Treatment, dietary, behavioral, etc.",
        default=None,
    )
    priority: Optional[CodeableConcept] = Field(
        description="high-priority | medium-priority | low-priority",
        default=None,
    )
    description: Optional[CodeableConcept] = Field(
        description="Code or text describing goal",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="Who this goal is intended for",
        default=None,
    )
    startDate: Optional[Date] = Field(
        description="When goal pursuit begins",
        default=None,
    )
    startDate_ext: Optional[Element] = Field(
        description="Placeholder element for startDate extensions",
        default=None,
        alias="_startDate",
    )
    startCodeableConcept: Optional[CodeableConcept] = Field(
        description="When goal pursuit begins",
        default=None,
    )
    target: Optional[List[GoalTarget]] = Field(
        description="Target outcome for the goal",
        default=None,
    )
    statusDate: Optional[Date] = Field(
        description="When goal status took effect",
        default=None,
    )
    statusDate_ext: Optional[Element] = Field(
        description="Placeholder element for statusDate extensions",
        default=None,
        alias="_statusDate",
    )
    statusReason: Optional[String] = Field(
        description="Reason for current status",
        default=None,
    )
    statusReason_ext: Optional[Element] = Field(
        description="Placeholder element for statusReason extensions",
        default=None,
        alias="_statusReason",
    )
    expressedBy: Optional[Reference] = Field(
        description="Who\u0027s responsible for creating Goal?",
        default=None,
    )
    addresses: Optional[List[Reference]] = Field(
        description="Issues addressed by this goal",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Comments about the goal",
        default=None,
    )
    outcomeCode: Optional[List[CodeableConcept]] = Field(
        description="What result was achieved regarding the goal?",
        default=None,
    )
    outcomeReference: Optional[List[Reference]] = Field(
        description="Observation that resulted from goal",
        default=None,
    )
    resourceType: Literal["Goal"] = Field(
        description=None,
        default="Goal",
    )

    @property
    def start(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="start",
        )

    @field_validator(
        *(
            "outcomeReference",
            "outcomeCode",
            "note",
            "addresses",
            "expressedBy",
            "statusReason",
            "statusDate",
            "target",
            "subject",
            "description",
            "priority",
            "category",
            "achievementStatus",
            "lifecycleStatus",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("target",), mode="after", check_fields=None)
    @classmethod
    def FHIR_gol_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(detail.exists() and measure.exists()) or detail.exists().not()",
            human="Goal.target.measure is required if Goal.target.detail is populated",
            key="gol-1",
            severity="error",
        )

    @model_validator(mode="after")
    def start_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Date, CodeableConcept],
            field_name_base="start",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
