# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Canonical,
    DateTime,
    Date,
    Boolean,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    Period,
    Age,
    Range,
    Annotation,
    BackboneElement,
    DomainResource,
)


class FamilyMemberHistoryCondition(BackboneElement):
    """
    The significant Conditions (or condition) that the family member had. This is a repeating section to allow a system to represent more than one condition per resource, though there is nothing stopping multiple resources - one per condition.
    """

    code: Optional[CodeableConcept] = Field(
        description="Condition suffered by relation",
        default=None,
    )
    outcome: Optional[CodeableConcept] = Field(
        description="deceased | permanent disability | etc.",
        default=None,
    )
    contributedToDeath: Optional[Boolean] = Field(
        description="Whether the condition contributed to the cause of death",
        default=None,
    )
    contributedToDeath_ext: Optional[Element] = Field(
        description="Placeholder element for contributedToDeath extensions",
        default=None,
        alias="_contributedToDeath",
    )
    onsetAge: Optional[Age] = Field(
        description="When condition first manifested",
        default=None,
    )
    onsetRange: Optional[Range] = Field(
        description="When condition first manifested",
        default=None,
    )
    onsetPeriod: Optional[Period] = Field(
        description="When condition first manifested",
        default=None,
    )
    onsetString: Optional[String] = Field(
        description="When condition first manifested",
        default=None,
    )
    onsetString_ext: Optional[Element] = Field(
        description="Placeholder element for onsetString extensions",
        default=None,
        alias="_onsetString",
    )
    note: Optional[List[Annotation]] = Field(
        description="Extra information about condition",
        default=None,
    )

    @property
    def onset(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="onset",
        )

    @field_validator(
        *(
            "note",
            "contributedToDeath",
            "outcome",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def onset_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Age, Range, Period, String],
            field_name_base="onset",
            required=False,
        )


class FamilyMemberHistory(DomainResource):
    """
    Significant health conditions for a person related to the patient relevant in the context of care for the patient.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/FamilyMemberHistory"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="External Id(s) for this record",
        default=None,
    )
    instantiatesCanonical: Optional[List[Canonical]] = Field(
        description="Instantiates FHIR protocol or definition",
        default=None,
    )
    instantiatesCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesCanonical extensions",
        default=None,
        alias="_instantiatesCanonical",
    )
    instantiatesUri: Optional[List[Uri]] = Field(
        description="Instantiates external protocol or definition",
        default=None,
    )
    instantiatesUri_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesUri extensions",
        default=None,
        alias="_instantiatesUri",
    )
    status: Optional[Code] = Field(
        description="partial | completed | entered-in-error | health-unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    dataAbsentReason: Optional[CodeableConcept] = Field(
        description="subject-unknown | withheld | unable-to-obtain | deferred",
        default=None,
    )
    patient: Optional[Reference] = Field(
        description="Patient history is about",
        default=None,
    )
    date: Optional[DateTime] = Field(
        description="When history was recorded or last updated",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    name: Optional[String] = Field(
        description="The family member described",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    relationship: Optional[CodeableConcept] = Field(
        description="Relationship to the subject",
        default=None,
    )
    sex: Optional[CodeableConcept] = Field(
        description="male | female | other | unknown",
        default=None,
    )
    bornPeriod: Optional[Period] = Field(
        description="(approximate) date of birth",
        default=None,
    )
    bornDate: Optional[Date] = Field(
        description="(approximate) date of birth",
        default=None,
    )
    bornDate_ext: Optional[Element] = Field(
        description="Placeholder element for bornDate extensions",
        default=None,
        alias="_bornDate",
    )
    bornString: Optional[String] = Field(
        description="(approximate) date of birth",
        default=None,
    )
    bornString_ext: Optional[Element] = Field(
        description="Placeholder element for bornString extensions",
        default=None,
        alias="_bornString",
    )
    ageAge: Optional[Age] = Field(
        description="(approximate) age",
        default=None,
    )
    ageRange: Optional[Range] = Field(
        description="(approximate) age",
        default=None,
    )
    ageString: Optional[String] = Field(
        description="(approximate) age",
        default=None,
    )
    ageString_ext: Optional[Element] = Field(
        description="Placeholder element for ageString extensions",
        default=None,
        alias="_ageString",
    )
    estimatedAge: Optional[Boolean] = Field(
        description="Age is estimated?",
        default=None,
    )
    estimatedAge_ext: Optional[Element] = Field(
        description="Placeholder element for estimatedAge extensions",
        default=None,
        alias="_estimatedAge",
    )
    deceasedBoolean: Optional[Boolean] = Field(
        description="Dead? How old/when?",
        default=None,
    )
    deceasedBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for deceasedBoolean extensions",
        default=None,
        alias="_deceasedBoolean",
    )
    deceasedAge: Optional[Age] = Field(
        description="Dead? How old/when?",
        default=None,
    )
    deceasedRange: Optional[Range] = Field(
        description="Dead? How old/when?",
        default=None,
    )
    deceasedDate: Optional[Date] = Field(
        description="Dead? How old/when?",
        default=None,
    )
    deceasedDate_ext: Optional[Element] = Field(
        description="Placeholder element for deceasedDate extensions",
        default=None,
        alias="_deceasedDate",
    )
    deceasedString: Optional[String] = Field(
        description="Dead? How old/when?",
        default=None,
    )
    deceasedString_ext: Optional[Element] = Field(
        description="Placeholder element for deceasedString extensions",
        default=None,
        alias="_deceasedString",
    )
    reasonCode: Optional[List[CodeableConcept]] = Field(
        description="Why was family member history performed?",
        default=None,
    )
    reasonReference: Optional[List[Reference]] = Field(
        description="Why was family member history performed?",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="General note about related person",
        default=None,
    )
    condition: Optional[List[FamilyMemberHistoryCondition]] = Field(
        description="Condition that the related person had",
        default=None,
    )
    resourceType: Literal["FamilyMemberHistory"] = Field(
        description=None,
        default="FamilyMemberHistory",
    )

    @property
    def born(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="born",
        )

    @property
    def age(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="age",
        )

    @property
    def deceased(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="deceased",
        )

    @field_validator(
        *(
            "condition",
            "note",
            "reasonReference",
            "reasonCode",
            "estimatedAge",
            "sex",
            "relationship",
            "name",
            "date",
            "patient",
            "dataAbsentReason",
            "status",
            "instantiatesUri",
            "instantiatesCanonical",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def born_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Period, Date, String],
            field_name_base="born",
            required=False,
        )

    @model_validator(mode="after")
    def age_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Age, Range, String],
            field_name_base="age",
            required=False,
        )

    @model_validator(mode="after")
    def deceased_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Boolean, Age, Range, Date, String],
            field_name_base="deceased",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_fhs_1_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="age.empty() or born.empty()",
            human="Can have age[x] or born[x], but not both",
            key="fhs-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_fhs_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="age.exists() or estimatedAge.empty()",
            human="Can only have estimatedAge if age[x] is present",
            key="fhs-2",
            severity="error",
        )
