# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code, PositiveInt

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    BackboneElement,
    Period,
    Coding,
    CodeableConcept,
    Reference,
    Duration,
    DomainResource,
)


class EncounterStatusHistory(BackboneElement):
    """
    The status history permits the encounter resource to contain the status history without needing to read through the historical versions of the resource, or even have the server store them.
    """

    status: Optional[Code] = Field(
        description="planned | arrived | triaged | in-progress | onleave | finished | cancelled +",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    period: Optional[Period] = Field(
        description="The time that the episode was in the specified status",
        default=None,
    )

    @field_validator(
        *(
            "period",
            "status",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EncounterClassHistory(BackboneElement):
    """
    The class history permits the tracking of the encounters transitions without needing to go  through the resource history.  This would be used for a case where an admission starts of as an emergency encounter, then transitions into an inpatient scenario. Doing this and not restarting a new encounter ensures that any lab/diagnostic results can more easily follow the patient and not require re-processing and not get lost or cancelled during a kind of discharge from emergency to inpatient.
    """

    class_: Optional[Coding] = Field(
        description="inpatient | outpatient | ambulatory | emergency +",
        default=None,
    )
    period: Optional[Period] = Field(
        description="The time that the episode was in the specified class",
        default=None,
    )

    @field_validator(
        *(
            "period",
            "class_",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EncounterParticipant(BackboneElement):
    """
    The list of people responsible for providing the service.
    """

    type: Optional[List[CodeableConcept]] = Field(
        description="Role of participant in encounter",
        default=None,
    )
    period: Optional[Period] = Field(
        description="Period of time during the encounter that the participant participated",
        default=None,
    )
    individual: Optional[Reference] = Field(
        description="Persons involved in the encounter other than the patient",
        default=None,
    )

    @field_validator(
        *(
            "individual",
            "period",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EncounterDiagnosis(BackboneElement):
    """
    The list of diagnosis relevant to this encounter.
    """

    condition: Optional[Reference] = Field(
        description="The diagnosis or procedure relevant to the encounter",
        default=None,
    )
    use: Optional[CodeableConcept] = Field(
        description="Role that this diagnosis has within the encounter (e.g. admission, billing, discharge \u2026)",
        default=None,
    )
    rank: Optional[PositiveInt] = Field(
        description="Ranking of the diagnosis (for each role type)",
        default=None,
    )
    rank_ext: Optional[Element] = Field(
        description="Placeholder element for rank extensions",
        default=None,
        alias="_rank",
    )

    @field_validator(
        *(
            "rank",
            "use",
            "condition",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EncounterHospitalization(BackboneElement):
    """
    Details about the admission to a healthcare service.
    """

    preAdmissionIdentifier: Optional[Identifier] = Field(
        description="Pre-admission identifier",
        default=None,
    )
    origin: Optional[Reference] = Field(
        description="The location/organization from which the patient came before admission",
        default=None,
    )
    admitSource: Optional[CodeableConcept] = Field(
        description="From where patient was admitted (physician referral, transfer)",
        default=None,
    )
    reAdmission: Optional[CodeableConcept] = Field(
        description="The type of hospital re-admission that has occurred (if any). If the value is absent, then this is not identified as a readmission",
        default=None,
    )
    dietPreference: Optional[List[CodeableConcept]] = Field(
        description="Diet preferences reported by the patient",
        default=None,
    )
    specialCourtesy: Optional[List[CodeableConcept]] = Field(
        description="Special courtesies (VIP, board member)",
        default=None,
    )
    specialArrangement: Optional[List[CodeableConcept]] = Field(
        description="Wheelchair, translator, stretcher, etc.",
        default=None,
    )
    destination: Optional[Reference] = Field(
        description="Location/organization to which the patient is discharged",
        default=None,
    )
    dischargeDisposition: Optional[CodeableConcept] = Field(
        description="Category or kind of location after discharge",
        default=None,
    )

    @field_validator(
        *(
            "dischargeDisposition",
            "destination",
            "specialArrangement",
            "specialCourtesy",
            "dietPreference",
            "reAdmission",
            "admitSource",
            "origin",
            "preAdmissionIdentifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EncounterLocation(BackboneElement):
    """
    List of locations where  the patient has been during this encounter.
    """

    location: Optional[Reference] = Field(
        description="Location the encounter takes place",
        default=None,
    )
    status: Optional[Code] = Field(
        description="planned | active | reserved | completed",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    physicalType: Optional[CodeableConcept] = Field(
        description="The physical type of the location (usually the level in the location hierachy - bed room ward etc.)",
        default=None,
    )
    period: Optional[Period] = Field(
        description="Time period during which the patient was present at the location",
        default=None,
    )

    @field_validator(
        *(
            "period",
            "physicalType",
            "status",
            "location",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class Encounter(DomainResource):
    """
    An interaction between a patient and healthcare provider(s) for the purpose of providing healthcare service(s) or assessing the health status of a patient.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Encounter"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Identifier(s) by which this encounter is known",
        default=None,
    )
    status: Optional[Code] = Field(
        description="planned | arrived | triaged | in-progress | onleave | finished | cancelled +",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    statusHistory: Optional[List[EncounterStatusHistory]] = Field(
        description="List of past encounter statuses",
        default=None,
    )
    class_: Optional[Coding] = Field(
        description="Classification of patient encounter",
        default=None,
    )
    classHistory: Optional[List[EncounterClassHistory]] = Field(
        description="List of past encounter classes",
        default=None,
    )
    type: Optional[List[CodeableConcept]] = Field(
        description="Specific type of encounter",
        default=None,
    )
    serviceType: Optional[CodeableConcept] = Field(
        description="Specific type of service",
        default=None,
    )
    priority: Optional[CodeableConcept] = Field(
        description="Indicates the urgency of the encounter",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="The patient or group present at the encounter",
        default=None,
    )
    episodeOfCare: Optional[List[Reference]] = Field(
        description="Episode(s) of care that this encounter should be recorded against",
        default=None,
    )
    basedOn: Optional[List[Reference]] = Field(
        description="The ServiceRequest that initiated this encounter",
        default=None,
    )
    participant: Optional[List[EncounterParticipant]] = Field(
        description="List of participants involved in the encounter",
        default=None,
    )
    appointment: Optional[List[Reference]] = Field(
        description="The appointment that scheduled this encounter",
        default=None,
    )
    period: Optional[Period] = Field(
        description="The start and end time of the encounter",
        default=None,
    )
    length: Optional[Duration] = Field(
        description="Quantity of time the encounter lasted (less time absent)",
        default=None,
    )
    reasonCode: Optional[List[CodeableConcept]] = Field(
        description="Coded reason the encounter takes place",
        default=None,
    )
    reasonReference: Optional[List[Reference]] = Field(
        description="Reason the encounter takes place (reference)",
        default=None,
    )
    diagnosis: Optional[List[EncounterDiagnosis]] = Field(
        description="The list of diagnosis relevant to this encounter",
        default=None,
    )
    account: Optional[List[Reference]] = Field(
        description="The set of accounts that may be used for billing for this Encounter",
        default=None,
    )
    hospitalization: Optional[EncounterHospitalization] = Field(
        description="Details about the admission to a healthcare service",
        default=None,
    )
    location: Optional[List[EncounterLocation]] = Field(
        description="List of locations where the patient has been",
        default=None,
    )
    serviceProvider: Optional[Reference] = Field(
        description="The organization (facility) responsible for this encounter",
        default=None,
    )
    partOf: Optional[Reference] = Field(
        description="Another Encounter this encounter is part of",
        default=None,
    )
    resourceType: Literal["Encounter"] = Field(
        description=None,
        default="Encounter",
    )

    @field_validator(
        *(
            "partOf",
            "serviceProvider",
            "location",
            "hospitalization",
            "account",
            "diagnosis",
            "reasonReference",
            "reasonCode",
            "length",
            "period",
            "appointment",
            "participant",
            "basedOn",
            "episodeOfCare",
            "subject",
            "priority",
            "serviceType",
            "type",
            "classHistory",
            "class_",
            "statusHistory",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
