# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    Markdown,
    Date,
    Integer,
    Decimal,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    ContactDetail,
    Annotation,
    UsageContext,
    CodeableConcept,
    Period,
    RelatedArtifact,
    Reference,
    BackboneElement,
    DomainResource,
)


class EffectEvidenceSynthesisSampleSize(BackboneElement):
    """
    A description of the size of the sample involved in the synthesis.
    """

    description: Optional[String] = Field(
        description="Description of sample size",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    numberOfStudies: Optional[Integer] = Field(
        description="How many studies?",
        default=None,
    )
    numberOfStudies_ext: Optional[Element] = Field(
        description="Placeholder element for numberOfStudies extensions",
        default=None,
        alias="_numberOfStudies",
    )
    numberOfParticipants: Optional[Integer] = Field(
        description="How many participants?",
        default=None,
    )
    numberOfParticipants_ext: Optional[Element] = Field(
        description="Placeholder element for numberOfParticipants extensions",
        default=None,
        alias="_numberOfParticipants",
    )

    @field_validator(
        *(
            "numberOfParticipants",
            "numberOfStudies",
            "description",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EffectEvidenceSynthesisResultsByExposure(BackboneElement):
    """
    A description of the results for each exposure considered in the effect estimate.
    """

    description: Optional[String] = Field(
        description="Description of results by exposure",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    exposureState: Optional[Code] = Field(
        description="exposure | exposure-alternative",
        default=None,
    )
    exposureState_ext: Optional[Element] = Field(
        description="Placeholder element for exposureState extensions",
        default=None,
        alias="_exposureState",
    )
    variantState: Optional[CodeableConcept] = Field(
        description="Variant exposure states",
        default=None,
    )
    riskEvidenceSynthesis: Optional[Reference] = Field(
        description="Risk evidence synthesis",
        default=None,
    )

    @field_validator(
        *(
            "riskEvidenceSynthesis",
            "variantState",
            "exposureState",
            "description",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EffectEvidenceSynthesisEffectEstimatePrecisionEstimate(BackboneElement):
    """
    A description of the precision of the estimate for the effect.
    """

    type: Optional[CodeableConcept] = Field(
        description="Type of precision estimate",
        default=None,
    )
    level: Optional[Decimal] = Field(
        description="Level of confidence interval",
        default=None,
    )
    level_ext: Optional[Element] = Field(
        description="Placeholder element for level extensions",
        default=None,
        alias="_level",
    )
    from_: Optional[Decimal] = Field(
        description="Lower bound",
        default=None,
    )
    from_ext: Optional[Element] = Field(
        description="Placeholder element for from extensions",
        default=None,
        alias="_from",
    )
    to: Optional[Decimal] = Field(
        description="Upper bound",
        default=None,
    )
    to_ext: Optional[Element] = Field(
        description="Placeholder element for to extensions",
        default=None,
        alias="_to",
    )

    @field_validator(
        *(
            "to",
            "from_",
            "level",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EffectEvidenceSynthesisEffectEstimate(BackboneElement):
    """
    The estimated effect of the exposure variant.
    """

    description: Optional[String] = Field(
        description="Description of effect estimate",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    type: Optional[CodeableConcept] = Field(
        description="Type of efffect estimate",
        default=None,
    )
    variantState: Optional[CodeableConcept] = Field(
        description="Variant exposure states",
        default=None,
    )
    value: Optional[Decimal] = Field(
        description="Point estimate",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )
    unitOfMeasure: Optional[CodeableConcept] = Field(
        description="What unit is the outcome described in?",
        default=None,
    )
    precisionEstimate: Optional[
        List[EffectEvidenceSynthesisEffectEstimatePrecisionEstimate]
    ] = Field(
        description="How precise the estimate is",
        default=None,
    )

    @field_validator(
        *(
            "precisionEstimate",
            "unitOfMeasure",
            "value",
            "variantState",
            "type",
            "description",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EffectEvidenceSynthesisCertaintyCertaintySubcomponent(BackboneElement):
    """
    A description of a component of the overall certainty.
    """

    type: Optional[CodeableConcept] = Field(
        description="Type of subcomponent of certainty rating",
        default=None,
    )
    rating: Optional[List[CodeableConcept]] = Field(
        description="Subcomponent certainty rating",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Used for footnotes or explanatory notes",
        default=None,
    )

    @field_validator(
        *(
            "note",
            "rating",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EffectEvidenceSynthesisCertainty(BackboneElement):
    """
    A description of the certainty of the effect estimate.
    """

    rating: Optional[List[CodeableConcept]] = Field(
        description="Certainty rating",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Used for footnotes or explanatory notes",
        default=None,
    )
    certaintySubcomponent: Optional[
        List[EffectEvidenceSynthesisCertaintyCertaintySubcomponent]
    ] = Field(
        description="A component that contributes to the overall certainty",
        default=None,
    )

    @field_validator(
        *(
            "certaintySubcomponent",
            "note",
            "rating",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class EffectEvidenceSynthesis(DomainResource):
    """
    The EffectEvidenceSynthesis resource describes the difference in an outcome between exposures states in a population where the effect estimate is derived from a combination of research studies.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/EffectEvidenceSynthesis"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this effect evidence synthesis, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifier for the effect evidence synthesis",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the effect evidence synthesis",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    name: Optional[String] = Field(
        description="Name for this effect evidence synthesis (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this effect evidence synthesis (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the effect evidence synthesis",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    note: Optional[List[Annotation]] = Field(
        description="Used for footnotes or explanatory notes",
        default=None,
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for effect evidence synthesis (if applicable)",
        default=None,
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    approvalDate: Optional[Date] = Field(
        description="When the effect evidence synthesis was approved by publisher",
        default=None,
    )
    approvalDate_ext: Optional[Element] = Field(
        description="Placeholder element for approvalDate extensions",
        default=None,
        alias="_approvalDate",
    )
    lastReviewDate: Optional[Date] = Field(
        description="When the effect evidence synthesis was last reviewed",
        default=None,
    )
    lastReviewDate_ext: Optional[Element] = Field(
        description="Placeholder element for lastReviewDate extensions",
        default=None,
        alias="_lastReviewDate",
    )
    effectivePeriod: Optional[Period] = Field(
        description="When the effect evidence synthesis is expected to be used",
        default=None,
    )
    topic: Optional[List[CodeableConcept]] = Field(
        description="The category of the EffectEvidenceSynthesis, such as Education, Treatment, Assessment, etc.",
        default=None,
    )
    author: Optional[List[ContactDetail]] = Field(
        description="Who authored the content",
        default=None,
    )
    editor: Optional[List[ContactDetail]] = Field(
        description="Who edited the content",
        default=None,
    )
    reviewer: Optional[List[ContactDetail]] = Field(
        description="Who reviewed the content",
        default=None,
    )
    endorser: Optional[List[ContactDetail]] = Field(
        description="Who endorsed the content",
        default=None,
    )
    relatedArtifact: Optional[List[RelatedArtifact]] = Field(
        description="Additional documentation, citations, etc.",
        default=None,
    )
    synthesisType: Optional[CodeableConcept] = Field(
        description="Type of synthesis",
        default=None,
    )
    studyType: Optional[CodeableConcept] = Field(
        description="Type of study",
        default=None,
    )
    population: Optional[Reference] = Field(
        description="What population?",
        default=None,
    )
    exposure: Optional[Reference] = Field(
        description="What exposure?",
        default=None,
    )
    exposureAlternative: Optional[Reference] = Field(
        description="What comparison exposure?",
        default=None,
    )
    outcome: Optional[Reference] = Field(
        description="What outcome?",
        default=None,
    )
    sampleSize: Optional[EffectEvidenceSynthesisSampleSize] = Field(
        description="What sample size was involved?",
        default=None,
    )
    resultsByExposure: Optional[List[EffectEvidenceSynthesisResultsByExposure]] = Field(
        description="What was the result per exposure?",
        default=None,
    )
    effectEstimate: Optional[List[EffectEvidenceSynthesisEffectEstimate]] = Field(
        description="What was the estimated effect",
        default=None,
    )
    certainty: Optional[List[EffectEvidenceSynthesisCertainty]] = Field(
        description="How certain is the effect",
        default=None,
    )
    resourceType: Literal["EffectEvidenceSynthesis"] = Field(
        description=None,
        default="EffectEvidenceSynthesis",
    )

    @field_validator(
        *(
            "certainty",
            "effectEstimate",
            "resultsByExposure",
            "sampleSize",
            "outcome",
            "exposureAlternative",
            "exposure",
            "population",
            "studyType",
            "synthesisType",
            "relatedArtifact",
            "endorser",
            "reviewer",
            "editor",
            "author",
            "topic",
            "effectivePeriod",
            "lastReviewDate",
            "approvalDate",
            "copyright",
            "jurisdiction",
            "useContext",
            "note",
            "description",
            "contact",
            "publisher",
            "date",
            "status",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_ees_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.matches('[A-Z]([A-Za-z0-9_]){0,254}')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="ees-0",
            severity="warning",
        )
