# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code, Instant

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    BackboneElement,
    Attachment,
    Coding,
    Period,
    DomainResource,
)


class DocumentReferenceRelatesTo(BackboneElement):
    """
    Relationships that this document has with other document references that already exist.
    """

    code: Optional[Code] = Field(
        description="replaces | transforms | signs | appends",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    target: Optional[Reference] = Field(
        description="Target of the relationship",
        default=None,
    )

    @field_validator(
        *(
            "target",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DocumentReferenceContent(BackboneElement):
    """
    The document and format referenced. There may be multiple content element repetitions, each with a different format.
    """

    attachment: Optional[Attachment] = Field(
        description="Where to access the document",
        default=None,
    )
    format: Optional[Coding] = Field(
        description="Format/content rules for the document",
        default=None,
    )

    @field_validator(
        *(
            "format",
            "attachment",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DocumentReferenceContext(BackboneElement):
    """
    The clinical context in which the document was prepared.
    """

    encounter: Optional[List[Reference]] = Field(
        description="Context of the document  content",
        default=None,
    )
    event: Optional[List[CodeableConcept]] = Field(
        description="Main clinical acts documented",
        default=None,
    )
    period: Optional[Period] = Field(
        description="Time of service that is being documented",
        default=None,
    )
    facilityType: Optional[CodeableConcept] = Field(
        description="Kind of facility where patient was seen",
        default=None,
    )
    practiceSetting: Optional[CodeableConcept] = Field(
        description="Additional details about where the content was created (e.g. clinical specialty)",
        default=None,
    )
    sourcePatientInfo: Optional[Reference] = Field(
        description="Patient demographics from source",
        default=None,
    )
    related: Optional[List[Reference]] = Field(
        description="Related identifiers or resources",
        default=None,
    )

    @field_validator(
        *(
            "related",
            "sourcePatientInfo",
            "practiceSetting",
            "facilityType",
            "period",
            "event",
            "encounter",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DocumentReference(DomainResource):
    """
    A reference to a document of any kind for any purpose. Provides metadata about the document so that the document can be discovered and managed. The scope of a document is any seralized object with a mime-type, so includes formal patient centric documents (CDA), cliical notes, scanned paper, and non-patient specific documents like policy text.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/DocumentReference"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    masterIdentifier: Optional[Identifier] = Field(
        description="Master Version Specific Identifier",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Other identifiers for the document",
        default=None,
    )
    status: Optional[Code] = Field(
        description="current | superseded | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    docStatus: Optional[Code] = Field(
        description="preliminary | final | amended | entered-in-error",
        default=None,
    )
    docStatus_ext: Optional[Element] = Field(
        description="Placeholder element for docStatus extensions",
        default=None,
        alias="_docStatus",
    )
    type: Optional[CodeableConcept] = Field(
        description="Kind of document (LOINC if possible)",
        default=None,
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Categorization of document",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="Who/what is the subject of the document",
        default=None,
    )
    date: Optional[Instant] = Field(
        description="When this document reference was created",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    author: Optional[List[Reference]] = Field(
        description="Who and/or what authored the document",
        default=None,
    )
    authenticator: Optional[Reference] = Field(
        description="Who/what authenticated the document",
        default=None,
    )
    custodian: Optional[Reference] = Field(
        description="Organization which maintains the document",
        default=None,
    )
    relatesTo: Optional[List[DocumentReferenceRelatesTo]] = Field(
        description="Relationships to other documents",
        default=None,
    )
    description: Optional[String] = Field(
        description="Human-readable description",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    securityLabel: Optional[List[CodeableConcept]] = Field(
        description="Document security-tags",
        default=None,
    )
    content: Optional[List[DocumentReferenceContent]] = Field(
        description="Document referenced",
        default=None,
    )
    context: Optional[DocumentReferenceContext] = Field(
        description="Clinical context of document",
        default=None,
    )
    resourceType: Literal["DocumentReference"] = Field(
        description=None,
        default="DocumentReference",
    )

    @field_validator(
        *(
            "context",
            "content",
            "securityLabel",
            "description",
            "relatesTo",
            "custodian",
            "authenticator",
            "author",
            "date",
            "subject",
            "category",
            "type",
            "docStatus",
            "status",
            "identifier",
            "masterIdentifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
