# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code, Boolean

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    BackboneElement,
    Reference,
    CodeableConcept,
    ProductShelfLife,
    ProdCharacteristic,
    Quantity,
    ContactPoint,
    Annotation,
    DomainResource,
)


class DeviceDefinitionUdiDeviceIdentifier(BackboneElement):
    """
    Unique device identifier (UDI) assigned to device label or package.  Note that the Device may include multiple udiCarriers as it either may include just the udiCarrier for the jurisdiction it is sold, or for multiple jurisdictions it could have been sold.
    """

    deviceIdentifier: Optional[String] = Field(
        description="The identifier that is to be associated with every Device that references this DeviceDefintiion for the issuer and jurisdication porvided in the DeviceDefinition.udiDeviceIdentifier",
        default=None,
    )
    deviceIdentifier_ext: Optional[Element] = Field(
        description="Placeholder element for deviceIdentifier extensions",
        default=None,
        alias="_deviceIdentifier",
    )
    issuer: Optional[Uri] = Field(
        description="The organization that assigns the identifier algorithm",
        default=None,
    )
    issuer_ext: Optional[Element] = Field(
        description="Placeholder element for issuer extensions",
        default=None,
        alias="_issuer",
    )
    jurisdiction: Optional[Uri] = Field(
        description="The jurisdiction to which the deviceIdentifier applies",
        default=None,
    )
    jurisdiction_ext: Optional[Element] = Field(
        description="Placeholder element for jurisdiction extensions",
        default=None,
        alias="_jurisdiction",
    )

    @field_validator(
        *(
            "jurisdiction",
            "issuer",
            "deviceIdentifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionDeviceName(BackboneElement):
    """
    A name given to the device to identify it.
    """

    name: Optional[String] = Field(
        description="The name of the device",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    type: Optional[Code] = Field(
        description="udi-label-name | user-friendly-name | patient-reported-name | manufacturer-name | model-name | other",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )

    @field_validator(
        *(
            "type",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionSpecialization(BackboneElement):
    """
    The capabilities supported on a  device, the standards to which the device conforms for a particular purpose, and used for the communication.
    """

    systemType: Optional[String] = Field(
        description="The standard that is used to operate and communicate",
        default=None,
    )
    systemType_ext: Optional[Element] = Field(
        description="Placeholder element for systemType extensions",
        default=None,
        alias="_systemType",
    )
    version: Optional[String] = Field(
        description="The version of the standard that is used to operate and communicate",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )

    @field_validator(
        *(
            "version",
            "systemType",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionCapability(BackboneElement):
    """
    Device capabilities.
    """

    type: Optional[CodeableConcept] = Field(
        description="Type of capability",
        default=None,
    )
    description: Optional[List[CodeableConcept]] = Field(
        description="Description of capability",
        default=None,
    )

    @field_validator(
        *(
            "description",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionProperty(BackboneElement):
    """
    The actual configuration settings of a device as it actually operates, e.g., regulation status, time properties.
    """

    type: Optional[CodeableConcept] = Field(
        description="Code that specifies the property DeviceDefinitionPropetyCode (Extensible)",
        default=None,
    )
    valueQuantity: Optional[List[Quantity]] = Field(
        description="Property value as a quantity",
        default=None,
    )
    valueCode: Optional[List[CodeableConcept]] = Field(
        description="Property value as a code, e.g., NTP4 (synced to NTP)",
        default=None,
    )

    @field_validator(
        *(
            "valueCode",
            "valueQuantity",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinitionMaterial(BackboneElement):
    """
    A substance used to create the material(s) of which the device is made.
    """

    substance: Optional[CodeableConcept] = Field(
        description="The substance",
        default=None,
    )
    alternate: Optional[Boolean] = Field(
        description="Indicates an alternative material of the device",
        default=None,
    )
    alternate_ext: Optional[Element] = Field(
        description="Placeholder element for alternate extensions",
        default=None,
        alias="_alternate",
    )
    allergenicIndicator: Optional[Boolean] = Field(
        description="Whether the substance is a known or suspected allergen",
        default=None,
    )
    allergenicIndicator_ext: Optional[Element] = Field(
        description="Placeholder element for allergenicIndicator extensions",
        default=None,
        alias="_allergenicIndicator",
    )

    @field_validator(
        *(
            "allergenicIndicator",
            "alternate",
            "substance",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDefinition(DomainResource):
    """
    The characteristics, operational status and capabilities of a medical-related component of a medical device.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/DeviceDefinition"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Instance identifier",
        default=None,
    )
    udiDeviceIdentifier: Optional[List[DeviceDefinitionUdiDeviceIdentifier]] = Field(
        description="Unique Device Identifier (UDI) Barcode string",
        default=None,
    )
    manufacturerString: Optional[String] = Field(
        description="Name of device manufacturer",
        default=None,
    )
    manufacturerString_ext: Optional[Element] = Field(
        description="Placeholder element for manufacturerString extensions",
        default=None,
        alias="_manufacturerString",
    )
    manufacturerReference: Optional[Reference] = Field(
        description="Name of device manufacturer",
        default=None,
    )
    deviceName: Optional[List[DeviceDefinitionDeviceName]] = Field(
        description="A name given to the device to identify it",
        default=None,
    )
    modelNumber: Optional[String] = Field(
        description="The model number for the device",
        default=None,
    )
    modelNumber_ext: Optional[Element] = Field(
        description="Placeholder element for modelNumber extensions",
        default=None,
        alias="_modelNumber",
    )
    type: Optional[CodeableConcept] = Field(
        description="What kind of device or device system this is",
        default=None,
    )
    specialization: Optional[List[DeviceDefinitionSpecialization]] = Field(
        description="The capabilities supported on a  device, the standards to which the device conforms for a particular purpose, and used for the communication",
        default=None,
    )
    version: Optional[List[String]] = Field(
        description="Available versions",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    safety: Optional[List[CodeableConcept]] = Field(
        description="Safety characteristics of the device",
        default=None,
    )
    shelfLifeStorage: Optional[List[ProductShelfLife]] = Field(
        description="Shelf Life and storage information",
        default=None,
    )
    physicalCharacteristics: Optional[ProdCharacteristic] = Field(
        description="Dimensions, color etc.",
        default=None,
    )
    languageCode: Optional[List[CodeableConcept]] = Field(
        description="Language code for the human-readable text strings produced by the device (all supported)",
        default=None,
    )
    capability: Optional[List[DeviceDefinitionCapability]] = Field(
        description="Device capabilities",
        default=None,
    )
    property_: Optional[List[DeviceDefinitionProperty]] = Field(
        description="The actual configuration settings of a device as it actually operates, e.g., regulation status, time properties",
        default=None,
    )
    owner: Optional[Reference] = Field(
        description="Organization responsible for device",
        default=None,
    )
    contact: Optional[List[ContactPoint]] = Field(
        description="Details for human/organization for support",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Network address to contact device",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    onlineInformation: Optional[Uri] = Field(
        description="Access to on-line information",
        default=None,
    )
    onlineInformation_ext: Optional[Element] = Field(
        description="Placeholder element for onlineInformation extensions",
        default=None,
        alias="_onlineInformation",
    )
    note: Optional[List[Annotation]] = Field(
        description="Device notes and comments",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="The quantity of the device present in the packaging (e.g. the number of devices present in a pack, or the number of devices in the same package of the medicinal product)",
        default=None,
    )
    parentDevice: Optional[Reference] = Field(
        description="The parent device it can be part of",
        default=None,
    )
    material: Optional[List[DeviceDefinitionMaterial]] = Field(
        description="A substance used to create the material(s) of which the device is made",
        default=None,
    )
    resourceType: Literal["DeviceDefinition"] = Field(
        description=None,
        default="DeviceDefinition",
    )

    @property
    def manufacturer(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="manufacturer",
        )

    @field_validator(
        *(
            "material",
            "parentDevice",
            "quantity",
            "note",
            "onlineInformation",
            "url",
            "contact",
            "owner",
            "property_",
            "capability",
            "languageCode",
            "physicalCharacteristics",
            "shelfLifeStorage",
            "safety",
            "version",
            "specialization",
            "type",
            "modelNumber",
            "deviceName",
            "udiDeviceIdentifier",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def manufacturer_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[String, Reference],
            field_name_base="manufacturer",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
