# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Base64Binary,
    DateTime,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    BackboneElement,
    CodeableConcept,
    Quantity,
    ContactPoint,
    Annotation,
    DomainResource,
)


class DeviceUdiCarrier(BackboneElement):
    """
    Unique device identifier (UDI) assigned to device label or package.  Note that the Device may include multiple udiCarriers as it either may include just the udiCarrier for the jurisdiction it is sold, or for multiple jurisdictions it could have been sold.
    """

    deviceIdentifier: Optional[String] = Field(
        description="Mandatory fixed portion of UDI",
        default=None,
    )
    deviceIdentifier_ext: Optional[Element] = Field(
        description="Placeholder element for deviceIdentifier extensions",
        default=None,
        alias="_deviceIdentifier",
    )
    issuer: Optional[Uri] = Field(
        description="UDI Issuing Organization",
        default=None,
    )
    issuer_ext: Optional[Element] = Field(
        description="Placeholder element for issuer extensions",
        default=None,
        alias="_issuer",
    )
    jurisdiction: Optional[Uri] = Field(
        description="Regional UDI authority",
        default=None,
    )
    jurisdiction_ext: Optional[Element] = Field(
        description="Placeholder element for jurisdiction extensions",
        default=None,
        alias="_jurisdiction",
    )
    carrierAIDC: Optional[Base64Binary] = Field(
        description="UDI Machine Readable Barcode String",
        default=None,
    )
    carrierAIDC_ext: Optional[Element] = Field(
        description="Placeholder element for carrierAIDC extensions",
        default=None,
        alias="_carrierAIDC",
    )
    carrierHRF: Optional[String] = Field(
        description="UDI Human Readable Barcode String",
        default=None,
    )
    carrierHRF_ext: Optional[Element] = Field(
        description="Placeholder element for carrierHRF extensions",
        default=None,
        alias="_carrierHRF",
    )
    entryType: Optional[Code] = Field(
        description="barcode | rfid | manual +",
        default=None,
    )
    entryType_ext: Optional[Element] = Field(
        description="Placeholder element for entryType extensions",
        default=None,
        alias="_entryType",
    )

    @field_validator(
        *(
            "entryType",
            "carrierHRF",
            "carrierAIDC",
            "jurisdiction",
            "issuer",
            "deviceIdentifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceDeviceName(BackboneElement):
    """
    This represents the manufacturer's name of the device as provided by the device, from a UDI label, or by a person describing the Device.  This typically would be used when a person provides the name(s) or when the device represents one of the names available from DeviceDefinition.
    """

    name: Optional[String] = Field(
        description="The name of the device",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    type: Optional[Code] = Field(
        description="udi-label-name | user-friendly-name | patient-reported-name | manufacturer-name | model-name | other",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )

    @field_validator(
        *(
            "type",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceSpecialization(BackboneElement):
    """
    The capabilities supported on a  device, the standards to which the device conforms for a particular purpose, and used for the communication.
    """

    systemType: Optional[CodeableConcept] = Field(
        description="The standard that is used to operate and communicate",
        default=None,
    )
    version: Optional[String] = Field(
        description="The version of the standard that is used to operate and communicate",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )

    @field_validator(
        *(
            "version",
            "systemType",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceVersion(BackboneElement):
    """
    The actual design of the device or software version running on the device.
    """

    type: Optional[CodeableConcept] = Field(
        description="The type of the device version",
        default=None,
    )
    component: Optional[Identifier] = Field(
        description="A single component of the device version",
        default=None,
    )
    value: Optional[String] = Field(
        description="The version text",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "component",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class DeviceProperty(BackboneElement):
    """
    The actual configuration settings of a device as it actually operates, e.g., regulation status, time properties.
    """

    type: Optional[CodeableConcept] = Field(
        description="Code that specifies the property DeviceDefinitionPropetyCode (Extensible)",
        default=None,
    )
    valueQuantity: Optional[List[Quantity]] = Field(
        description="Property value as a quantity",
        default=None,
    )
    valueCode: Optional[List[CodeableConcept]] = Field(
        description="Property value as a code, e.g., NTP4 (synced to NTP)",
        default=None,
    )

    @field_validator(
        *(
            "valueCode",
            "valueQuantity",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class Device(DomainResource):
    """
    A type of a manufactured item that is used in the provision of healthcare without being substantially changed through that activity. The device may be a medical or non-medical device.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Device"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Instance identifier",
        default=None,
    )
    definition: Optional[Reference] = Field(
        description="The reference to the definition for the device",
        default=None,
    )
    udiCarrier: Optional[List[DeviceUdiCarrier]] = Field(
        description="Unique Device Identifier (UDI) Barcode string",
        default=None,
    )
    status: Optional[Code] = Field(
        description="active | inactive | entered-in-error | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    statusReason: Optional[List[CodeableConcept]] = Field(
        description="online | paused | standby | offline | not-ready | transduc-discon | hw-discon | off",
        default=None,
    )
    distinctIdentifier: Optional[String] = Field(
        description="The distinct identification string",
        default=None,
    )
    distinctIdentifier_ext: Optional[Element] = Field(
        description="Placeholder element for distinctIdentifier extensions",
        default=None,
        alias="_distinctIdentifier",
    )
    manufacturer: Optional[String] = Field(
        description="Name of device manufacturer",
        default=None,
    )
    manufacturer_ext: Optional[Element] = Field(
        description="Placeholder element for manufacturer extensions",
        default=None,
        alias="_manufacturer",
    )
    manufactureDate: Optional[DateTime] = Field(
        description="Date when the device was made",
        default=None,
    )
    manufactureDate_ext: Optional[Element] = Field(
        description="Placeholder element for manufactureDate extensions",
        default=None,
        alias="_manufactureDate",
    )
    expirationDate: Optional[DateTime] = Field(
        description="Date and time of expiry of this device (if applicable)",
        default=None,
    )
    expirationDate_ext: Optional[Element] = Field(
        description="Placeholder element for expirationDate extensions",
        default=None,
        alias="_expirationDate",
    )
    lotNumber: Optional[String] = Field(
        description="Lot number of manufacture",
        default=None,
    )
    lotNumber_ext: Optional[Element] = Field(
        description="Placeholder element for lotNumber extensions",
        default=None,
        alias="_lotNumber",
    )
    serialNumber: Optional[String] = Field(
        description="Serial number assigned by the manufacturer",
        default=None,
    )
    serialNumber_ext: Optional[Element] = Field(
        description="Placeholder element for serialNumber extensions",
        default=None,
        alias="_serialNumber",
    )
    deviceName: Optional[List[DeviceDeviceName]] = Field(
        description="The name of the device as given by the manufacturer",
        default=None,
    )
    modelNumber: Optional[String] = Field(
        description="The model number for the device",
        default=None,
    )
    modelNumber_ext: Optional[Element] = Field(
        description="Placeholder element for modelNumber extensions",
        default=None,
        alias="_modelNumber",
    )
    partNumber: Optional[String] = Field(
        description="The part number of the device",
        default=None,
    )
    partNumber_ext: Optional[Element] = Field(
        description="Placeholder element for partNumber extensions",
        default=None,
        alias="_partNumber",
    )
    type: Optional[CodeableConcept] = Field(
        description="The kind or type of device",
        default=None,
    )
    specialization: Optional[List[DeviceSpecialization]] = Field(
        description="The capabilities supported on a  device, the standards to which the device conforms for a particular purpose, and used for the communication",
        default=None,
    )
    version: Optional[List[DeviceVersion]] = Field(
        description="The actual design of the device or software version running on the device",
        default=None,
    )
    property_: Optional[List[DeviceProperty]] = Field(
        description="The actual configuration settings of a device as it actually operates, e.g., regulation status, time properties",
        default=None,
    )
    patient: Optional[Reference] = Field(
        description="Patient to whom Device is affixed",
        default=None,
    )
    owner: Optional[Reference] = Field(
        description="Organization responsible for device",
        default=None,
    )
    contact: Optional[List[ContactPoint]] = Field(
        description="Details for human/organization for support",
        default=None,
    )
    location: Optional[Reference] = Field(
        description="Where the device is found",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Network address to contact device",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    note: Optional[List[Annotation]] = Field(
        description="Device notes and comments",
        default=None,
    )
    safety: Optional[List[CodeableConcept]] = Field(
        description="Safety Characteristics of Device",
        default=None,
    )
    parent: Optional[Reference] = Field(
        description="The parent device",
        default=None,
    )
    resourceType: Literal["Device"] = Field(
        description=None,
        default="Device",
    )

    @field_validator(
        *(
            "parent",
            "safety",
            "note",
            "url",
            "location",
            "contact",
            "owner",
            "patient",
            "property_",
            "version",
            "specialization",
            "type",
            "partNumber",
            "modelNumber",
            "deviceName",
            "serialNumber",
            "lotNumber",
            "expirationDate",
            "manufactureDate",
            "manufacturer",
            "distinctIdentifier",
            "statusReason",
            "status",
            "udiCarrier",
            "definition",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
