# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    Markdown,
    UnsignedInt,
    Boolean,
    Decimal,
    Integer,
    Date,
    Time,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    Period,
    BackboneElement,
    Coding,
    Attachment,
    Quantity,
    Money,
    Timing,
    Annotation,
    Signature,
    DomainResource,
)


class ContractContentDefinition(BackboneElement):
    """
    Precusory content developed with a focus and intent of supporting the formation a Contract instance, which may be associated with and transformable into a Contract.
    """

    type: Optional[CodeableConcept] = Field(
        description="Content structure and use",
        default=None,
    )
    subType: Optional[CodeableConcept] = Field(
        description="Detailed Content Type Definition",
        default=None,
    )
    publisher: Optional[Reference] = Field(
        description="Publisher Entity",
        default=None,
    )
    publicationDate: Optional[DateTime] = Field(
        description="When published",
        default=None,
    )
    publicationDate_ext: Optional[Element] = Field(
        description="Placeholder element for publicationDate extensions",
        default=None,
        alias="_publicationDate",
    )
    publicationStatus: Optional[Code] = Field(
        description="amended | appended | cancelled | disputed | entered-in-error | executable | executed | negotiable | offered | policy | rejected | renewed | revoked | resolved | terminated",
        default=None,
    )
    publicationStatus_ext: Optional[Element] = Field(
        description="Placeholder element for publicationStatus extensions",
        default=None,
        alias="_publicationStatus",
    )
    copyright: Optional[Markdown] = Field(
        description="Publication Ownership",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )

    @field_validator(
        *(
            "copyright",
            "publicationStatus",
            "publicationDate",
            "publisher",
            "subType",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ContractTermSecurityLabel(BackboneElement):
    """
    Security labels that protect the handling of information about the term and its elements, which may be specifically identified..
    """

    number: Optional[List[UnsignedInt]] = Field(
        description="Link to Security Labels",
        default=None,
    )
    number_ext: Optional[Element] = Field(
        description="Placeholder element for number extensions",
        default=None,
        alias="_number",
    )
    classification: Optional[Coding] = Field(
        description="Confidentiality Protection",
        default=None,
    )
    category: Optional[List[Coding]] = Field(
        description="Applicable Policy",
        default=None,
    )
    control: Optional[List[Coding]] = Field(
        description="Handling Instructions",
        default=None,
    )

    @field_validator(
        *(
            "control",
            "category",
            "classification",
            "number",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ContractTermOfferParty(BackboneElement):
    """
    Offer Recipient.
    """

    reference: Optional[List[Reference]] = Field(
        description="Referenced entity",
        default=None,
    )
    role: Optional[CodeableConcept] = Field(
        description="Participant engagement type",
        default=None,
    )

    @field_validator(
        *(
            "role",
            "reference",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ContractTermOfferAnswer(BackboneElement):
    """
    Response to offer text.
    """

    valueBoolean: Optional[Boolean] = Field(
        description="The actual answer response",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueDecimal: Optional[Decimal] = Field(
        description="The actual answer response",
        default=None,
    )
    valueDecimal_ext: Optional[Element] = Field(
        description="Placeholder element for valueDecimal extensions",
        default=None,
        alias="_valueDecimal",
    )
    valueInteger: Optional[Integer] = Field(
        description="The actual answer response",
        default=None,
    )
    valueInteger_ext: Optional[Element] = Field(
        description="Placeholder element for valueInteger extensions",
        default=None,
        alias="_valueInteger",
    )
    valueDate: Optional[Date] = Field(
        description="The actual answer response",
        default=None,
    )
    valueDate_ext: Optional[Element] = Field(
        description="Placeholder element for valueDate extensions",
        default=None,
        alias="_valueDate",
    )
    valueDateTime: Optional[DateTime] = Field(
        description="The actual answer response",
        default=None,
    )
    valueDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for valueDateTime extensions",
        default=None,
        alias="_valueDateTime",
    )
    valueTime: Optional[Time] = Field(
        description="The actual answer response",
        default=None,
    )
    valueTime_ext: Optional[Element] = Field(
        description="Placeholder element for valueTime extensions",
        default=None,
        alias="_valueTime",
    )
    valueString: Optional[String] = Field(
        description="The actual answer response",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueUri: Optional[Uri] = Field(
        description="The actual answer response",
        default=None,
    )
    valueUri_ext: Optional[Element] = Field(
        description="Placeholder element for valueUri extensions",
        default=None,
        alias="_valueUri",
    )
    valueAttachment: Optional[Attachment] = Field(
        description="The actual answer response",
        default=None,
    )
    valueCoding: Optional[Coding] = Field(
        description="The actual answer response",
        default=None,
    )
    valueQuantity: Optional[Quantity] = Field(
        description="The actual answer response",
        default=None,
    )
    valueReference: Optional[Reference] = Field(
        description="The actual answer response",
        default=None,
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[
                Boolean,
                Decimal,
                Integer,
                Date,
                DateTime,
                Time,
                String,
                Uri,
                Attachment,
                Coding,
                Quantity,
                Reference,
            ],
            field_name_base="value",
            required=True,
        )


class ContractTermOffer(BackboneElement):
    """
    The matter of concern in the context of this provision of the agrement.
    """

    identifier: Optional[List[Identifier]] = Field(
        description="Offer business ID",
        default=None,
    )
    party: Optional[List[ContractTermOfferParty]] = Field(
        description="Offer Recipient",
        default=None,
    )
    topic: Optional[Reference] = Field(
        description="Negotiable offer asset",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="Contract Offer Type or Form",
        default=None,
    )
    decision: Optional[CodeableConcept] = Field(
        description="Accepting party choice",
        default=None,
    )
    decisionMode: Optional[List[CodeableConcept]] = Field(
        description="How decision is conveyed",
        default=None,
    )
    answer: Optional[List[ContractTermOfferAnswer]] = Field(
        description="Response to offer text",
        default=None,
    )
    text: Optional[String] = Field(
        description="Human readable offer text",
        default=None,
    )
    text_ext: Optional[Element] = Field(
        description="Placeholder element for text extensions",
        default=None,
        alias="_text",
    )
    linkId: Optional[List[String]] = Field(
        description="Pointer to text",
        default=None,
    )
    linkId_ext: Optional[Element] = Field(
        description="Placeholder element for linkId extensions",
        default=None,
        alias="_linkId",
    )
    securityLabelNumber: Optional[List[UnsignedInt]] = Field(
        description="Offer restriction numbers",
        default=None,
    )
    securityLabelNumber_ext: Optional[Element] = Field(
        description="Placeholder element for securityLabelNumber extensions",
        default=None,
        alias="_securityLabelNumber",
    )

    @field_validator(
        *(
            "securityLabelNumber",
            "linkId",
            "text",
            "answer",
            "decisionMode",
            "decision",
            "type",
            "topic",
            "party",
            "identifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ContractTermAssetContext(BackboneElement):
    """
    Circumstance of the asset.
    """

    reference: Optional[Reference] = Field(
        description="Creator,custodian or owner",
        default=None,
    )
    code: Optional[List[CodeableConcept]] = Field(
        description="Codeable asset context",
        default=None,
    )
    text: Optional[String] = Field(
        description="Context description",
        default=None,
    )
    text_ext: Optional[Element] = Field(
        description="Placeholder element for text extensions",
        default=None,
        alias="_text",
    )

    @field_validator(
        *(
            "text",
            "code",
            "reference",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ContractTermAssetAnswer(BackboneElement):
    """
    Response to assets.
    """

    valueBoolean: Optional[Boolean] = Field(
        description="The actual answer response",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueDecimal: Optional[Decimal] = Field(
        description="The actual answer response",
        default=None,
    )
    valueDecimal_ext: Optional[Element] = Field(
        description="Placeholder element for valueDecimal extensions",
        default=None,
        alias="_valueDecimal",
    )
    valueInteger: Optional[Integer] = Field(
        description="The actual answer response",
        default=None,
    )
    valueInteger_ext: Optional[Element] = Field(
        description="Placeholder element for valueInteger extensions",
        default=None,
        alias="_valueInteger",
    )
    valueDate: Optional[Date] = Field(
        description="The actual answer response",
        default=None,
    )
    valueDate_ext: Optional[Element] = Field(
        description="Placeholder element for valueDate extensions",
        default=None,
        alias="_valueDate",
    )
    valueDateTime: Optional[DateTime] = Field(
        description="The actual answer response",
        default=None,
    )
    valueDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for valueDateTime extensions",
        default=None,
        alias="_valueDateTime",
    )
    valueTime: Optional[Time] = Field(
        description="The actual answer response",
        default=None,
    )
    valueTime_ext: Optional[Element] = Field(
        description="Placeholder element for valueTime extensions",
        default=None,
        alias="_valueTime",
    )
    valueString: Optional[String] = Field(
        description="The actual answer response",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueUri: Optional[Uri] = Field(
        description="The actual answer response",
        default=None,
    )
    valueUri_ext: Optional[Element] = Field(
        description="Placeholder element for valueUri extensions",
        default=None,
        alias="_valueUri",
    )
    valueAttachment: Optional[Attachment] = Field(
        description="The actual answer response",
        default=None,
    )
    valueCoding: Optional[Coding] = Field(
        description="The actual answer response",
        default=None,
    )
    valueQuantity: Optional[Quantity] = Field(
        description="The actual answer response",
        default=None,
    )
    valueReference: Optional[Reference] = Field(
        description="The actual answer response",
        default=None,
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[
                Boolean,
                Decimal,
                Integer,
                Date,
                DateTime,
                Time,
                String,
                Uri,
                Attachment,
                Coding,
                Quantity,
                Reference,
            ],
            field_name_base="value",
            required=True,
        )


class ContractTermAssetValuedItem(BackboneElement):
    """
    Contract Valued Item List.
    """

    entityCodeableConcept: Optional[CodeableConcept] = Field(
        description="Contract Valued Item Type",
        default=None,
    )
    entityReference: Optional[Reference] = Field(
        description="Contract Valued Item Type",
        default=None,
    )
    identifier: Optional[Identifier] = Field(
        description="Contract Valued Item Number",
        default=None,
    )
    effectiveTime: Optional[DateTime] = Field(
        description="Contract Valued Item Effective Tiem",
        default=None,
    )
    effectiveTime_ext: Optional[Element] = Field(
        description="Placeholder element for effectiveTime extensions",
        default=None,
        alias="_effectiveTime",
    )
    quantity: Optional[Quantity] = Field(
        description="Count of Contract Valued Items",
        default=None,
    )
    unitPrice: Optional[Money] = Field(
        description="Contract Valued Item fee, charge, or cost",
        default=None,
    )
    factor: Optional[Decimal] = Field(
        description="Contract Valued Item Price Scaling Factor",
        default=None,
    )
    factor_ext: Optional[Element] = Field(
        description="Placeholder element for factor extensions",
        default=None,
        alias="_factor",
    )
    points: Optional[Decimal] = Field(
        description="Contract Valued Item Difficulty Scaling Factor",
        default=None,
    )
    points_ext: Optional[Element] = Field(
        description="Placeholder element for points extensions",
        default=None,
        alias="_points",
    )
    net: Optional[Money] = Field(
        description="Total Contract Valued Item Value",
        default=None,
    )
    payment: Optional[String] = Field(
        description="Terms of valuation",
        default=None,
    )
    payment_ext: Optional[Element] = Field(
        description="Placeholder element for payment extensions",
        default=None,
        alias="_payment",
    )
    paymentDate: Optional[DateTime] = Field(
        description="When payment is due",
        default=None,
    )
    paymentDate_ext: Optional[Element] = Field(
        description="Placeholder element for paymentDate extensions",
        default=None,
        alias="_paymentDate",
    )
    responsible: Optional[Reference] = Field(
        description="Who will make payment",
        default=None,
    )
    recipient: Optional[Reference] = Field(
        description="Who will receive payment",
        default=None,
    )
    linkId: Optional[List[String]] = Field(
        description="Pointer to specific item",
        default=None,
    )
    linkId_ext: Optional[Element] = Field(
        description="Placeholder element for linkId extensions",
        default=None,
        alias="_linkId",
    )
    securityLabelNumber: Optional[List[UnsignedInt]] = Field(
        description="Security Labels that define affected terms",
        default=None,
    )
    securityLabelNumber_ext: Optional[Element] = Field(
        description="Placeholder element for securityLabelNumber extensions",
        default=None,
        alias="_securityLabelNumber",
    )

    @property
    def entity(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="entity",
        )

    @field_validator(
        *(
            "securityLabelNumber",
            "linkId",
            "recipient",
            "responsible",
            "paymentDate",
            "payment",
            "net",
            "points",
            "factor",
            "unitPrice",
            "quantity",
            "effectiveTime",
            "identifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def entity_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Reference],
            field_name_base="entity",
            required=False,
        )


class ContractTermAsset(BackboneElement):
    """
    Contract Term Asset List.
    """

    scope: Optional[CodeableConcept] = Field(
        description="Range of asset",
        default=None,
    )
    type: Optional[List[CodeableConcept]] = Field(
        description="Asset category",
        default=None,
    )
    typeReference: Optional[List[Reference]] = Field(
        description="Associated entities",
        default=None,
    )
    subtype: Optional[List[CodeableConcept]] = Field(
        description="Asset sub-category",
        default=None,
    )
    relationship: Optional[Coding] = Field(
        description="Kinship of the asset",
        default=None,
    )
    context: Optional[List[ContractTermAssetContext]] = Field(
        description="Circumstance of the asset",
        default=None,
    )
    condition: Optional[String] = Field(
        description="Quality desctiption of asset",
        default=None,
    )
    condition_ext: Optional[Element] = Field(
        description="Placeholder element for condition extensions",
        default=None,
        alias="_condition",
    )
    periodType: Optional[List[CodeableConcept]] = Field(
        description="Asset availability types",
        default=None,
    )
    period: Optional[List[Period]] = Field(
        description="Time period of the asset",
        default=None,
    )
    usePeriod: Optional[List[Period]] = Field(
        description="Time period",
        default=None,
    )
    text: Optional[String] = Field(
        description="Asset clause or question text",
        default=None,
    )
    text_ext: Optional[Element] = Field(
        description="Placeholder element for text extensions",
        default=None,
        alias="_text",
    )
    linkId: Optional[List[String]] = Field(
        description="Pointer to asset text",
        default=None,
    )
    linkId_ext: Optional[Element] = Field(
        description="Placeholder element for linkId extensions",
        default=None,
        alias="_linkId",
    )
    answer: Optional[List[ContractTermAssetAnswer]] = Field(
        description="Response to assets",
        default=None,
    )
    securityLabelNumber: Optional[List[UnsignedInt]] = Field(
        description="Asset restriction numbers",
        default=None,
    )
    securityLabelNumber_ext: Optional[Element] = Field(
        description="Placeholder element for securityLabelNumber extensions",
        default=None,
        alias="_securityLabelNumber",
    )
    valuedItem: Optional[List[ContractTermAssetValuedItem]] = Field(
        description="Contract Valued Item List",
        default=None,
    )

    @field_validator(
        *(
            "valuedItem",
            "securityLabelNumber",
            "answer",
            "linkId",
            "text",
            "usePeriod",
            "period",
            "periodType",
            "condition",
            "context",
            "relationship",
            "subtype",
            "typeReference",
            "type",
            "scope",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ContractTermActionSubject(BackboneElement):
    """
    Entity of the action.
    """

    reference: Optional[List[Reference]] = Field(
        description="Entity of the action",
        default=None,
    )
    role: Optional[CodeableConcept] = Field(
        description="Role type of the agent",
        default=None,
    )

    @field_validator(
        *(
            "role",
            "reference",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ContractTermAction(BackboneElement):
    """
    An actor taking a role in an activity for which it can be assigned some degree of responsibility for the activity taking place.
    """

    doNotPerform: Optional[Boolean] = Field(
        description="True if the term prohibits the  action",
        default=None,
    )
    doNotPerform_ext: Optional[Element] = Field(
        description="Placeholder element for doNotPerform extensions",
        default=None,
        alias="_doNotPerform",
    )
    type: Optional[CodeableConcept] = Field(
        description="Type or form of the action",
        default=None,
    )
    subject: Optional[List[ContractTermActionSubject]] = Field(
        description="Entity of the action",
        default=None,
    )
    intent: Optional[CodeableConcept] = Field(
        description="Purpose for the Contract Term Action",
        default=None,
    )
    linkId: Optional[List[String]] = Field(
        description="Pointer to specific item",
        default=None,
    )
    linkId_ext: Optional[Element] = Field(
        description="Placeholder element for linkId extensions",
        default=None,
        alias="_linkId",
    )
    status: Optional[CodeableConcept] = Field(
        description="State of the action",
        default=None,
    )
    context: Optional[Reference] = Field(
        description="Episode associated with action",
        default=None,
    )
    contextLinkId: Optional[List[String]] = Field(
        description="Pointer to specific item",
        default=None,
    )
    contextLinkId_ext: Optional[Element] = Field(
        description="Placeholder element for contextLinkId extensions",
        default=None,
        alias="_contextLinkId",
    )
    occurrenceDateTime: Optional[DateTime] = Field(
        description="When action happens",
        default=None,
    )
    occurrenceDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for occurrenceDateTime extensions",
        default=None,
        alias="_occurrenceDateTime",
    )
    occurrencePeriod: Optional[Period] = Field(
        description="When action happens",
        default=None,
    )
    occurrenceTiming: Optional[Timing] = Field(
        description="When action happens",
        default=None,
    )
    requester: Optional[List[Reference]] = Field(
        description="Who asked for action",
        default=None,
    )
    requesterLinkId: Optional[List[String]] = Field(
        description="Pointer to specific item",
        default=None,
    )
    requesterLinkId_ext: Optional[Element] = Field(
        description="Placeholder element for requesterLinkId extensions",
        default=None,
        alias="_requesterLinkId",
    )
    performerType: Optional[List[CodeableConcept]] = Field(
        description="Kind of service performer",
        default=None,
    )
    performerRole: Optional[CodeableConcept] = Field(
        description="Competency of the performer",
        default=None,
    )
    performer: Optional[Reference] = Field(
        description="Actor that wil execute (or not) the action",
        default=None,
    )
    performerLinkId: Optional[List[String]] = Field(
        description="Pointer to specific item",
        default=None,
    )
    performerLinkId_ext: Optional[Element] = Field(
        description="Placeholder element for performerLinkId extensions",
        default=None,
        alias="_performerLinkId",
    )
    reasonCode: Optional[List[CodeableConcept]] = Field(
        description="Why is action (not) needed?",
        default=None,
    )
    reasonReference: Optional[List[Reference]] = Field(
        description="Why is action (not) needed?",
        default=None,
    )
    reason: Optional[List[String]] = Field(
        description="Why action is to be performed",
        default=None,
    )
    reason_ext: Optional[Element] = Field(
        description="Placeholder element for reason extensions",
        default=None,
        alias="_reason",
    )
    reasonLinkId: Optional[List[String]] = Field(
        description="Pointer to specific item",
        default=None,
    )
    reasonLinkId_ext: Optional[Element] = Field(
        description="Placeholder element for reasonLinkId extensions",
        default=None,
        alias="_reasonLinkId",
    )
    note: Optional[List[Annotation]] = Field(
        description="Comments about the action",
        default=None,
    )
    securityLabelNumber: Optional[List[UnsignedInt]] = Field(
        description="Action restriction numbers",
        default=None,
    )
    securityLabelNumber_ext: Optional[Element] = Field(
        description="Placeholder element for securityLabelNumber extensions",
        default=None,
        alias="_securityLabelNumber",
    )

    @property
    def occurrence(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="occurrence",
        )

    @field_validator(
        *(
            "securityLabelNumber",
            "note",
            "reasonLinkId",
            "reason",
            "reasonReference",
            "reasonCode",
            "performerLinkId",
            "performer",
            "performerRole",
            "performerType",
            "requesterLinkId",
            "requester",
            "contextLinkId",
            "context",
            "status",
            "linkId",
            "intent",
            "subject",
            "type",
            "doNotPerform",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def occurrence_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, Period, Timing],
            field_name_base="occurrence",
            required=False,
        )


class ContractTerm(BackboneElement):
    """
    One or more Contract Provisions, which may be related and conveyed as a group, and may contain nested groups.
    """

    identifier: Optional[Identifier] = Field(
        description="Contract Term Number",
        default=None,
    )
    issued: Optional[DateTime] = Field(
        description="Contract Term Issue Date Time",
        default=None,
    )
    issued_ext: Optional[Element] = Field(
        description="Placeholder element for issued extensions",
        default=None,
        alias="_issued",
    )
    applies: Optional[Period] = Field(
        description="Contract Term Effective Time",
        default=None,
    )
    topicCodeableConcept: Optional[CodeableConcept] = Field(
        description="Term Concern",
        default=None,
    )
    topicReference: Optional[Reference] = Field(
        description="Term Concern",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="Contract Term Type or Form",
        default=None,
    )
    subType: Optional[CodeableConcept] = Field(
        description="Contract Term Type specific classification",
        default=None,
    )
    text: Optional[String] = Field(
        description="Term Statement",
        default=None,
    )
    text_ext: Optional[Element] = Field(
        description="Placeholder element for text extensions",
        default=None,
        alias="_text",
    )
    securityLabel: Optional[List[ContractTermSecurityLabel]] = Field(
        description="Protection for the Term",
        default=None,
    )
    offer: Optional[ContractTermOffer] = Field(
        description="Context of the Contract term",
        default=None,
    )
    asset: Optional[List[ContractTermAsset]] = Field(
        description="Contract Term Asset List",
        default=None,
    )
    action: Optional[List[ContractTermAction]] = Field(
        description="Entity being ascribed responsibility",
        default=None,
    )
    group: Optional[List["ContractTerm"]] = Field(
        description="Nested Contract Term Group",
        default=None,
    )

    @property
    def topic(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="topic",
        )

    @field_validator(
        *(
            "group",
            "action",
            "asset",
            "offer",
            "securityLabel",
            "text",
            "subType",
            "type",
            "applies",
            "issued",
            "identifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def topic_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Reference],
            field_name_base="topic",
            required=False,
        )


class ContractSigner(BackboneElement):
    """
    Parties with legal standing in the Contract, including the principal parties, the grantor(s) and grantee(s), which are any person or organization bound by the contract, and any ancillary parties, which facilitate the execution of the contract such as a notary or witness.
    """

    type: Optional[Coding] = Field(
        description="Contract Signatory Role",
        default=None,
    )
    party: Optional[Reference] = Field(
        description="Contract Signatory Party",
        default=None,
    )
    signature: Optional[List[Signature]] = Field(
        description="Contract Documentation Signature",
        default=None,
    )

    @field_validator(
        *(
            "signature",
            "party",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ContractFriendly(BackboneElement):
    """
    The "patient friendly language" versionof the Contract in whole or in parts. "Patient friendly language" means the representation of the Contract and Contract Provisions in a manner that is readily accessible and understandable by a layperson in accordance with best practices for communication styles that ensure that those agreeing to or signing the Contract understand the roles, actions, obligations, responsibilities, and implication of the agreement.
    """

    contentAttachment: Optional[Attachment] = Field(
        description="Easily comprehended representation of this Contract",
        default=None,
    )
    contentReference: Optional[Reference] = Field(
        description="Easily comprehended representation of this Contract",
        default=None,
    )

    @property
    def content(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="content",
        )

    @model_validator(mode="after")
    def content_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Attachment, Reference],
            field_name_base="content",
            required=True,
        )


class ContractLegal(BackboneElement):
    """
    List of Legal expressions or representations of this Contract.
    """

    contentAttachment: Optional[Attachment] = Field(
        description="Contract Legal Text",
        default=None,
    )
    contentReference: Optional[Reference] = Field(
        description="Contract Legal Text",
        default=None,
    )

    @property
    def content(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="content",
        )

    @model_validator(mode="after")
    def content_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Attachment, Reference],
            field_name_base="content",
            required=True,
        )


class ContractRule(BackboneElement):
    """
    List of Computable Policy Rule Language Representations of this Contract.
    """

    contentAttachment: Optional[Attachment] = Field(
        description="Computable Contract Rules",
        default=None,
    )
    contentReference: Optional[Reference] = Field(
        description="Computable Contract Rules",
        default=None,
    )

    @property
    def content(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="content",
        )

    @model_validator(mode="after")
    def content_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Attachment, Reference],
            field_name_base="content",
            required=True,
        )


class Contract(DomainResource):
    """
    Legally enforceable, formally recorded unilateral or bilateral directive i.e., a policy or agreement.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Contract"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Contract number",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Basal definition",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    version: Optional[String] = Field(
        description="Business edition",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    status: Optional[Code] = Field(
        description="amended | appended | cancelled | disputed | entered-in-error | executable | executed | negotiable | offered | policy | rejected | renewed | revoked | resolved | terminated",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    legalState: Optional[CodeableConcept] = Field(
        description="Negotiation status",
        default=None,
    )
    instantiatesCanonical: Optional[Reference] = Field(
        description="Source Contract Definition",
        default=None,
    )
    instantiatesUri: Optional[Uri] = Field(
        description="External Contract Definition",
        default=None,
    )
    instantiatesUri_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesUri extensions",
        default=None,
        alias="_instantiatesUri",
    )
    contentDerivative: Optional[CodeableConcept] = Field(
        description="Content derived from the basal information",
        default=None,
    )
    issued: Optional[DateTime] = Field(
        description="When this Contract was issued",
        default=None,
    )
    issued_ext: Optional[Element] = Field(
        description="Placeholder element for issued extensions",
        default=None,
        alias="_issued",
    )
    applies: Optional[Period] = Field(
        description="Effective time",
        default=None,
    )
    expirationType: Optional[CodeableConcept] = Field(
        description="Contract cessation cause",
        default=None,
    )
    subject: Optional[List[Reference]] = Field(
        description="Contract Target Entity",
        default=None,
    )
    authority: Optional[List[Reference]] = Field(
        description="Authority under which this Contract has standing",
        default=None,
    )
    domain: Optional[List[Reference]] = Field(
        description="A sphere of control governed by an authoritative jurisdiction, organization, or person",
        default=None,
    )
    site: Optional[List[Reference]] = Field(
        description="Specific Location",
        default=None,
    )
    name: Optional[String] = Field(
        description="Computer friendly designation",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Human Friendly name",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    subtitle: Optional[String] = Field(
        description="Subordinate Friendly name",
        default=None,
    )
    subtitle_ext: Optional[Element] = Field(
        description="Placeholder element for subtitle extensions",
        default=None,
        alias="_subtitle",
    )
    alias: Optional[List[String]] = Field(
        description="Acronym or short name",
        default=None,
    )
    alias_ext: Optional[Element] = Field(
        description="Placeholder element for alias extensions",
        default=None,
        alias="_alias",
    )
    author: Optional[Reference] = Field(
        description="Source of Contract",
        default=None,
    )
    scope: Optional[CodeableConcept] = Field(
        description="Range of Legal Concerns",
        default=None,
    )
    topicCodeableConcept: Optional[CodeableConcept] = Field(
        description="Focus of contract interest",
        default=None,
    )
    topicReference: Optional[Reference] = Field(
        description="Focus of contract interest",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="Legal instrument category",
        default=None,
    )
    subType: Optional[List[CodeableConcept]] = Field(
        description="Subtype within the context of type",
        default=None,
    )
    contentDefinition: Optional[ContractContentDefinition] = Field(
        description="Contract precursor content",
        default=None,
    )
    term: Optional[List[ContractTerm]] = Field(
        description="Contract Term List",
        default=None,
    )
    supportingInfo: Optional[List[Reference]] = Field(
        description="Extra Information",
        default=None,
    )
    relevantHistory: Optional[List[Reference]] = Field(
        description="Key event in Contract History",
        default=None,
    )
    signer: Optional[List[ContractSigner]] = Field(
        description="Contract Signatory",
        default=None,
    )
    friendly: Optional[List[ContractFriendly]] = Field(
        description="Contract Friendly Language",
        default=None,
    )
    legal: Optional[List[ContractLegal]] = Field(
        description="Contract Legal Language",
        default=None,
    )
    rule: Optional[List[ContractRule]] = Field(
        description="Computable Contract Language",
        default=None,
    )
    legallyBindingAttachment: Optional[Attachment] = Field(
        description="Binding Contract",
        default=None,
    )
    legallyBindingReference: Optional[Reference] = Field(
        description="Binding Contract",
        default=None,
    )
    resourceType: Literal["Contract"] = Field(
        description=None,
        default="Contract",
    )

    @property
    def topic(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="topic",
        )

    @property
    def legallyBinding(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="legallyBinding",
        )

    @field_validator(
        *(
            "rule",
            "legal",
            "friendly",
            "signer",
            "relevantHistory",
            "supportingInfo",
            "term",
            "contentDefinition",
            "subType",
            "type",
            "scope",
            "author",
            "alias",
            "subtitle",
            "title",
            "name",
            "site",
            "domain",
            "authority",
            "subject",
            "expirationType",
            "applies",
            "issued",
            "contentDerivative",
            "instantiatesUri",
            "instantiatesCanonical",
            "legalState",
            "status",
            "version",
            "url",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def topic_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Reference],
            field_name_base="topic",
            required=False,
        )

    @model_validator(mode="after")
    def legallyBinding_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Attachment, Reference],
            field_name_base="legallyBinding",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
