# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code, DateTime

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    Age,
    Period,
    Range,
    BackboneElement,
    Annotation,
    DomainResource,
)


class ConditionStage(BackboneElement):
    """
    Clinical stage or grade of a condition. May include formal severity assessments.
    """

    summary: Optional[CodeableConcept] = Field(
        description="Simple summary (disease specific)",
        default=None,
    )
    assessment: Optional[List[Reference]] = Field(
        description="Formal record of assessment",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="Kind of staging",
        default=None,
    )

    @field_validator(
        *(
            "type",
            "assessment",
            "summary",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ConditionEvidence(BackboneElement):
    """
    Supporting evidence / manifestations that are the basis of the Condition's verification status, such as evidence that confirmed or refuted the condition.
    """

    code: Optional[List[CodeableConcept]] = Field(
        description="Manifestation/symptom",
        default=None,
    )
    detail: Optional[List[Reference]] = Field(
        description="Supporting information found elsewhere",
        default=None,
    )

    @field_validator(
        *(
            "detail",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class Condition(DomainResource):
    """
    A clinical condition, problem, diagnosis, or other event, situation, issue, or clinical concept that has risen to a level of concern.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Condition"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="External Ids for this condition",
        default=None,
    )
    clinicalStatus: Optional[CodeableConcept] = Field(
        description="active | recurrence | relapse | inactive | remission | resolved",
        default=None,
    )
    verificationStatus: Optional[CodeableConcept] = Field(
        description="unconfirmed | provisional | differential | confirmed | refuted | entered-in-error",
        default=None,
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="problem-list-item | encounter-diagnosis",
        default=None,
    )
    severity: Optional[CodeableConcept] = Field(
        description="Subjective severity of condition",
        default=None,
    )
    code: Optional[CodeableConcept] = Field(
        description="Identification of the condition, problem or diagnosis",
        default=None,
    )
    bodySite: Optional[List[CodeableConcept]] = Field(
        description="Anatomical location, if relevant",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="Who has the condition?",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Encounter created as part of",
        default=None,
    )
    onsetDateTime: Optional[DateTime] = Field(
        description="Estimated or actual date,  date-time, or age",
        default=None,
    )
    onsetDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for onsetDateTime extensions",
        default=None,
        alias="_onsetDateTime",
    )
    onsetAge: Optional[Age] = Field(
        description="Estimated or actual date,  date-time, or age",
        default=None,
    )
    onsetPeriod: Optional[Period] = Field(
        description="Estimated or actual date,  date-time, or age",
        default=None,
    )
    onsetRange: Optional[Range] = Field(
        description="Estimated or actual date,  date-time, or age",
        default=None,
    )
    onsetString: Optional[String] = Field(
        description="Estimated or actual date,  date-time, or age",
        default=None,
    )
    onsetString_ext: Optional[Element] = Field(
        description="Placeholder element for onsetString extensions",
        default=None,
        alias="_onsetString",
    )
    abatementDateTime: Optional[DateTime] = Field(
        description="When in resolution/remission",
        default=None,
    )
    abatementDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for abatementDateTime extensions",
        default=None,
        alias="_abatementDateTime",
    )
    abatementAge: Optional[Age] = Field(
        description="When in resolution/remission",
        default=None,
    )
    abatementPeriod: Optional[Period] = Field(
        description="When in resolution/remission",
        default=None,
    )
    abatementRange: Optional[Range] = Field(
        description="When in resolution/remission",
        default=None,
    )
    abatementString: Optional[String] = Field(
        description="When in resolution/remission",
        default=None,
    )
    abatementString_ext: Optional[Element] = Field(
        description="Placeholder element for abatementString extensions",
        default=None,
        alias="_abatementString",
    )
    recordedDate: Optional[DateTime] = Field(
        description="Date record was first recorded",
        default=None,
    )
    recordedDate_ext: Optional[Element] = Field(
        description="Placeholder element for recordedDate extensions",
        default=None,
        alias="_recordedDate",
    )
    recorder: Optional[Reference] = Field(
        description="Who recorded the condition",
        default=None,
    )
    asserter: Optional[Reference] = Field(
        description="Person who asserts this condition",
        default=None,
    )
    stage: Optional[List[ConditionStage]] = Field(
        description="Stage/grade, usually assessed formally",
        default=None,
    )
    evidence: Optional[List[ConditionEvidence]] = Field(
        description="Supporting evidence",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Additional information about the Condition",
        default=None,
    )
    resourceType: Literal["Condition"] = Field(
        description=None,
        default="Condition",
    )

    @property
    def onset(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="onset",
        )

    @property
    def abatement(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="abatement",
        )

    @field_validator(
        *(
            "note",
            "evidence",
            "stage",
            "asserter",
            "recorder",
            "recordedDate",
            "encounter",
            "subject",
            "bodySite",
            "code",
            "severity",
            "category",
            "verificationStatus",
            "clinicalStatus",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("stage",), mode="after", check_fields=None)
    @classmethod
    def FHIR_con_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="summary.exists() or assessment.exists()",
            human="Stage SHALL have summary or assessment",
            key="con-1",
            severity="error",
        )

    @field_validator(*("evidence",), mode="after", check_fields=None)
    @classmethod
    def FHIR_con_2_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="code.exists() or detail.exists()",
            human="evidence SHALL have code or details",
            key="con-2",
            severity="error",
        )

    @model_validator(mode="after")
    def onset_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, Age, Period, Range, String],
            field_name_base="onset",
            required=False,
        )

    @model_validator(mode="after")
    def abatement_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, Age, Period, Range, String],
            field_name_base="abatement",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_con_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="clinicalStatus.exists() or verificationStatus.coding.where(system='http://terminology.hl7.org/CodeSystem/condition-ver-status' and code = 'entered-in-error').exists() or category.select($this='problem-list-item').empty()",
            human="Condition.clinicalStatus SHALL be present if verificationStatus is not entered-in-error and category is problem-list-item",
            key="con-3",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_con_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="abatement.empty() or clinicalStatus.coding.where(system='http://terminology.hl7.org/CodeSystem/condition-clinical' and (code='resolved' or code='remission' or code='inactive')).exists()",
            human="If condition is abated, then clinicalStatus must be either inactive, resolved, or remission",
            key="con-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_con_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="verificationStatus.coding.where(system='http://terminology.hl7.org/CodeSystem/condition-ver-status' and code='entered-in-error').empty() or clinicalStatus.empty()",
            human="Condition.clinicalStatus SHALL NOT be present if verification Status is entered-in-error",
            key="con-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
