# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    DateTime,
    Markdown,
    Canonical,
    UnsignedInt,
    Integer,
    Decimal,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    ContactDetail,
    UsageContext,
    CodeableConcept,
    BackboneElement,
    Coding,
    DomainResource,
)


class CodeSystemFilter(BackboneElement):
    """
    A filter that can be used in a value set compose statement when selecting concepts using a filter.
    """

    code: Optional[Code] = Field(
        description="Code that identifies the filter",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    description: Optional[String] = Field(
        description="How or why the filter is used",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    operator: Optional[List[Code]] = Field(
        description="= | is-a | descendent-of | is-not-a | regex | in | not-in | generalizes | exists",
        default=None,
    )
    operator_ext: Optional[Element] = Field(
        description="Placeholder element for operator extensions",
        default=None,
        alias="_operator",
    )
    value: Optional[String] = Field(
        description="What to use for the value",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "operator",
            "description",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CodeSystemProperty(BackboneElement):
    """
    A property defines an additional slot through which additional information can be provided about a concept.
    """

    code: Optional[Code] = Field(
        description="Identifies the property on the concepts, and when referred to in operations",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    uri: Optional[Uri] = Field(
        description="Formal identifier for the property",
        default=None,
    )
    uri_ext: Optional[Element] = Field(
        description="Placeholder element for uri extensions",
        default=None,
        alias="_uri",
    )
    description: Optional[String] = Field(
        description="Why the property is defined, and/or what it conveys",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    type: Optional[Code] = Field(
        description="code | Coding | string | integer | boolean | dateTime | decimal",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )

    @field_validator(
        *(
            "type",
            "description",
            "uri",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CodeSystemConceptDesignation(BackboneElement):
    """
    Additional representations for the concept - other languages, aliases, specialized purposes, used for particular purposes, etc.
    """

    language: Optional[Code] = Field(
        description="Human language of the designation",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    use: Optional[Coding] = Field(
        description="Details how this designation would be used",
        default=None,
    )
    value: Optional[String] = Field(
        description="The text value for this designation",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "use",
            "language",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CodeSystemConceptProperty(BackboneElement):
    """
    A property value for this concept.
    """

    code: Optional[Code] = Field(
        description="Reference to CodeSystem.property.code",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    valueCode: Optional[Code] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueCode_ext: Optional[Element] = Field(
        description="Placeholder element for valueCode extensions",
        default=None,
        alias="_valueCode",
    )
    valueCoding: Optional[Coding] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueString: Optional[String] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueInteger: Optional[Integer] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueInteger_ext: Optional[Element] = Field(
        description="Placeholder element for valueInteger extensions",
        default=None,
        alias="_valueInteger",
    )
    valueBoolean: Optional[Boolean] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueDateTime: Optional[DateTime] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for valueDateTime extensions",
        default=None,
        alias="_valueDateTime",
    )
    valueDecimal: Optional[Decimal] = Field(
        description="Value of the property for this concept",
        default=None,
    )
    valueDecimal_ext: Optional[Element] = Field(
        description="Placeholder element for valueDecimal extensions",
        default=None,
        alias="_valueDecimal",
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("code", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Code, Coding, String, Integer, Boolean, DateTime, Decimal],
            field_name_base="value",
            required=True,
        )


class CodeSystemConcept(BackboneElement):
    """
    Concepts that are in the code system. The concept definitions are inherently hierarchical, but the definitions must be consulted to determine what the meanings of the hierarchical relationships are.
    """

    code: Optional[Code] = Field(
        description="Code that identifies concept",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    display: Optional[String] = Field(
        description="Text to display to the user",
        default=None,
    )
    display_ext: Optional[Element] = Field(
        description="Placeholder element for display extensions",
        default=None,
        alias="_display",
    )
    definition: Optional[String] = Field(
        description="Formal definition",
        default=None,
    )
    definition_ext: Optional[Element] = Field(
        description="Placeholder element for definition extensions",
        default=None,
        alias="_definition",
    )
    designation: Optional[List[CodeSystemConceptDesignation]] = Field(
        description="Additional representations for the concept",
        default=None,
    )
    property_: Optional[List[CodeSystemConceptProperty]] = Field(
        description="Property value for the concept",
        default=None,
    )
    concept: Optional[List["CodeSystemConcept"]] = Field(
        description="Child Concepts (is-a/contains/categorizes)",
        default=None,
    )

    @field_validator(
        *(
            "concept",
            "property_",
            "designation",
            "definition",
            "display",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CodeSystem(DomainResource):
    """
    The CodeSystem resource is used to declare the existence of and describe a code system or code system supplement and its key properties, and optionally define a part or all of its content.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/CodeSystem"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this code system, represented as a URI (globally unique) (Coding.system)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifier for the code system (business identifier)",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the code system (Coding.version)",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    name: Optional[String] = Field(
        description="Name for this code system (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this code system (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the code system",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for code system (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this code system is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    caseSensitive: Optional[Boolean] = Field(
        description="If code comparison is case sensitive",
        default=None,
    )
    caseSensitive_ext: Optional[Element] = Field(
        description="Placeholder element for caseSensitive extensions",
        default=None,
        alias="_caseSensitive",
    )
    valueSet: Optional[Canonical] = Field(
        description="Canonical reference to the value set with entire code system",
        default=None,
    )
    valueSet_ext: Optional[Element] = Field(
        description="Placeholder element for valueSet extensions",
        default=None,
        alias="_valueSet",
    )
    hierarchyMeaning: Optional[Code] = Field(
        description="grouped-by | is-a | part-of | classified-with",
        default=None,
    )
    hierarchyMeaning_ext: Optional[Element] = Field(
        description="Placeholder element for hierarchyMeaning extensions",
        default=None,
        alias="_hierarchyMeaning",
    )
    compositional: Optional[Boolean] = Field(
        description="If code system defines a compositional grammar",
        default=None,
    )
    compositional_ext: Optional[Element] = Field(
        description="Placeholder element for compositional extensions",
        default=None,
        alias="_compositional",
    )
    versionNeeded: Optional[Boolean] = Field(
        description="If definitions are not stable",
        default=None,
    )
    versionNeeded_ext: Optional[Element] = Field(
        description="Placeholder element for versionNeeded extensions",
        default=None,
        alias="_versionNeeded",
    )
    content: Optional[Code] = Field(
        description="not-present | example | fragment | complete | supplement",
        default=None,
    )
    content_ext: Optional[Element] = Field(
        description="Placeholder element for content extensions",
        default=None,
        alias="_content",
    )
    supplements: Optional[Canonical] = Field(
        description="Canonical URL of Code System this adds designations and properties to",
        default=None,
    )
    supplements_ext: Optional[Element] = Field(
        description="Placeholder element for supplements extensions",
        default=None,
        alias="_supplements",
    )
    count: Optional[UnsignedInt] = Field(
        description="Total concepts in the code system",
        default=None,
    )
    count_ext: Optional[Element] = Field(
        description="Placeholder element for count extensions",
        default=None,
        alias="_count",
    )
    filter: Optional[List[CodeSystemFilter]] = Field(
        description="Filter that can be used in a value set",
        default=None,
    )
    property_: Optional[List[CodeSystemProperty]] = Field(
        description="Additional information supplied about each concept",
        default=None,
    )
    concept: Optional[List[CodeSystemConcept]] = Field(
        description="Concepts in the code system",
        default=None,
    )
    resourceType: Literal["CodeSystem"] = Field(
        description=None,
        default="CodeSystem",
    )

    @field_validator(
        *(
            "concept",
            "property_",
            "filter",
            "count",
            "supplements",
            "content",
            "versionNeeded",
            "compositional",
            "hierarchyMeaning",
            "valueSet",
            "caseSensitive",
            "copyright",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "experimental",
            "status",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_csd_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.matches('[A-Z]([A-Za-z0-9_]){0,254}')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="csd-0",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_csd_1_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="concept.code.combine($this.descendants().concept.code).isDistinct()",
            human="Within a code system definition, all the codes SHALL be unique",
            key="csd-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
