# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code, DateTime

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    Period,
    BackboneElement,
    Annotation,
    DomainResource,
)


class ClinicalImpressionInvestigation(BackboneElement):
    """
    One or more sets of investigations (signs, symptoms, etc.). The actual grouping of investigations varies greatly depending on the type and context of the assessment. These investigations may include data generated during the assessment process, or data previously generated and recorded that is pertinent to the outcomes.
    """

    code: Optional[CodeableConcept] = Field(
        description="A name/code for the set",
        default=None,
    )
    item: Optional[List[Reference]] = Field(
        description="Record of a specific investigation",
        default=None,
    )

    @field_validator(
        *(
            "item",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClinicalImpressionFinding(BackboneElement):
    """
    Specific findings or diagnoses that were considered likely or relevant to ongoing treatment.
    """

    itemCodeableConcept: Optional[CodeableConcept] = Field(
        description="What was found",
        default=None,
    )
    itemReference: Optional[Reference] = Field(
        description="What was found",
        default=None,
    )
    basis: Optional[String] = Field(
        description="Which investigations support finding",
        default=None,
    )
    basis_ext: Optional[Element] = Field(
        description="Placeholder element for basis extensions",
        default=None,
        alias="_basis",
    )

    @field_validator(
        *(
            "basis",
            "itemReference",
            "itemCodeableConcept",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClinicalImpression(DomainResource):
    """
    A record of a clinical assessment performed to determine what problem(s) may affect the patient and before planning the treatments or management strategies that are best to manage a patient's condition. Assessments are often 1:1 with a clinical consultation / encounter,  but this varies greatly depending on the clinical workflow. This resource is called "ClinicalImpression" rather than "ClinicalAssessment" to avoid confusion with the recording of assessment tools such as Apgar score.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/ClinicalImpression"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business identifier",
        default=None,
    )
    status: Optional[Code] = Field(
        description="in-progress | completed | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    statusReason: Optional[CodeableConcept] = Field(
        description="Reason for current status",
        default=None,
    )
    code: Optional[CodeableConcept] = Field(
        description="Kind of assessment performed",
        default=None,
    )
    description: Optional[String] = Field(
        description="Why/how the assessment was performed",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    subject: Optional[Reference] = Field(
        description="Patient or group assessed",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Encounter created as part of",
        default=None,
    )
    effectiveDateTime: Optional[DateTime] = Field(
        description="Time of assessment",
        default=None,
    )
    effectiveDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for effectiveDateTime extensions",
        default=None,
        alias="_effectiveDateTime",
    )
    effectivePeriod: Optional[Period] = Field(
        description="Time of assessment",
        default=None,
    )
    date: Optional[DateTime] = Field(
        description="When the assessment was documented",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    assessor: Optional[Reference] = Field(
        description="The clinician performing the assessment",
        default=None,
    )
    previous: Optional[Reference] = Field(
        description="Reference to last assessment",
        default=None,
    )
    problem: Optional[List[Reference]] = Field(
        description="Relevant impressions of patient state",
        default=None,
    )
    investigation: Optional[List[ClinicalImpressionInvestigation]] = Field(
        description="One or more sets of investigations (signs, symptoms, etc.)",
        default=None,
    )
    protocol: Optional[List[Uri]] = Field(
        description="Clinical Protocol followed",
        default=None,
    )
    protocol_ext: Optional[Element] = Field(
        description="Placeholder element for protocol extensions",
        default=None,
        alias="_protocol",
    )
    summary: Optional[String] = Field(
        description="Summary of the assessment",
        default=None,
    )
    summary_ext: Optional[Element] = Field(
        description="Placeholder element for summary extensions",
        default=None,
        alias="_summary",
    )
    finding: Optional[List[ClinicalImpressionFinding]] = Field(
        description="Possible or likely findings and diagnoses",
        default=None,
    )
    prognosisCodeableConcept: Optional[List[CodeableConcept]] = Field(
        description="Estimate of likely outcome",
        default=None,
    )
    prognosisReference: Optional[List[Reference]] = Field(
        description="RiskAssessment expressing likely outcome",
        default=None,
    )
    supportingInfo: Optional[List[Reference]] = Field(
        description="Information supporting the clinical impression",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Comments made about the ClinicalImpression",
        default=None,
    )
    resourceType: Literal["ClinicalImpression"] = Field(
        description=None,
        default="ClinicalImpression",
    )

    @property
    def effective(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="effective",
        )

    @field_validator(
        *(
            "note",
            "supportingInfo",
            "prognosisReference",
            "prognosisCodeableConcept",
            "finding",
            "summary",
            "protocol",
            "investigation",
            "problem",
            "previous",
            "assessor",
            "date",
            "encounter",
            "subject",
            "description",
            "code",
            "statusReason",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def effective_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, Period],
            field_name_base="effective",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
