# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    PositiveInt,
    Decimal,
    Date,
    Boolean,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    Period,
    BackboneElement,
    Money,
    Address,
    Quantity,
    Attachment,
    DomainResource,
)


class ClaimResponseItemAdjudication(BackboneElement):
    """
    If this item is a group then the values here are a summary of the adjudication of the detail items. If this item is a simple product or service then this is the result of the adjudication of this item.
    """

    category: Optional[CodeableConcept] = Field(
        description="Type of adjudication information",
        default=None,
    )
    reason: Optional[CodeableConcept] = Field(
        description="Explanation of adjudication outcome",
        default=None,
    )
    amount: Optional[Money] = Field(
        description="Monetary amount",
        default=None,
    )
    value: Optional[Decimal] = Field(
        description="Non-monetary value",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "amount",
            "reason",
            "category",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClaimResponseItemDetailAdjudication(BackboneElement):
    """
    The adjudication results.
    """

    category: Optional[CodeableConcept] = Field(
        description="Type of adjudication information",
        default=None,
    )
    reason: Optional[CodeableConcept] = Field(
        description="Explanation of adjudication outcome",
        default=None,
    )
    amount: Optional[Money] = Field(
        description="Monetary amount",
        default=None,
    )
    value: Optional[Decimal] = Field(
        description="Non-monetary value",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "amount",
            "reason",
            "category",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClaimResponseItemDetailSubDetail(BackboneElement):
    """
    A sub-detail adjudication of a simple product or service.
    """

    subDetailSequence: Optional[PositiveInt] = Field(
        description="Claim sub-detail instance identifier",
        default=None,
    )
    subDetailSequence_ext: Optional[Element] = Field(
        description="Placeholder element for subDetailSequence extensions",
        default=None,
        alias="_subDetailSequence",
    )
    noteNumber: Optional[List[PositiveInt]] = Field(
        description="Applicable note numbers",
        default=None,
    )
    noteNumber_ext: Optional[Element] = Field(
        description="Placeholder element for noteNumber extensions",
        default=None,
        alias="_noteNumber",
    )
    adjudication: Optional[List[ClaimResponseItemAdjudication]] = Field(
        description="Subdetail level adjudication details",
        default=None,
    )

    @field_validator(
        *(
            "adjudication",
            "noteNumber",
            "subDetailSequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClaimResponseItemDetail(BackboneElement):
    """
    A claim detail. Either a simple (a product or service) or a 'group' of sub-details which are simple items.
    """

    detailSequence: Optional[PositiveInt] = Field(
        description="Claim detail instance identifier",
        default=None,
    )
    detailSequence_ext: Optional[Element] = Field(
        description="Placeholder element for detailSequence extensions",
        default=None,
        alias="_detailSequence",
    )
    noteNumber: Optional[List[PositiveInt]] = Field(
        description="Applicable note numbers",
        default=None,
    )
    noteNumber_ext: Optional[Element] = Field(
        description="Placeholder element for noteNumber extensions",
        default=None,
        alias="_noteNumber",
    )
    adjudication: Optional[List[ClaimResponseItemDetailAdjudication]] = Field(
        description="Detail level adjudication details",
        default=None,
    )
    subDetail: Optional[List[ClaimResponseItemDetailSubDetail]] = Field(
        description="Adjudication for claim sub-details",
        default=None,
    )

    @field_validator(
        *(
            "subDetail",
            "adjudication",
            "noteNumber",
            "detailSequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClaimResponseItem(BackboneElement):
    """
    A claim line. Either a simple (a product or service) or a 'group' of details which can also be a simple items or groups of sub-details.
    """

    itemSequence: Optional[PositiveInt] = Field(
        description="Claim item instance identifier",
        default=None,
    )
    itemSequence_ext: Optional[Element] = Field(
        description="Placeholder element for itemSequence extensions",
        default=None,
        alias="_itemSequence",
    )
    noteNumber: Optional[List[PositiveInt]] = Field(
        description="Applicable note numbers",
        default=None,
    )
    noteNumber_ext: Optional[Element] = Field(
        description="Placeholder element for noteNumber extensions",
        default=None,
        alias="_noteNumber",
    )
    adjudication: Optional[List[ClaimResponseItemAdjudication]] = Field(
        description="Adjudication details",
        default=None,
    )
    detail: Optional[List[ClaimResponseItemDetail]] = Field(
        description="Adjudication for claim details",
        default=None,
    )

    @field_validator(
        *(
            "detail",
            "adjudication",
            "noteNumber",
            "itemSequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClaimResponseAddItemDetailSubDetail(BackboneElement):
    """
    The third-tier service adjudications for payor added services.
    """

    productOrService: Optional[CodeableConcept] = Field(
        description="Billing, service, product, or drug code",
        default=None,
    )
    modifier: Optional[List[CodeableConcept]] = Field(
        description="Service/Product billing modifiers",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="Count of products or services",
        default=None,
    )
    unitPrice: Optional[Money] = Field(
        description="Fee, charge or cost per item",
        default=None,
    )
    factor: Optional[Decimal] = Field(
        description="Price scaling factor",
        default=None,
    )
    factor_ext: Optional[Element] = Field(
        description="Placeholder element for factor extensions",
        default=None,
        alias="_factor",
    )
    net: Optional[Money] = Field(
        description="Total item cost",
        default=None,
    )
    noteNumber: Optional[List[PositiveInt]] = Field(
        description="Applicable note numbers",
        default=None,
    )
    noteNumber_ext: Optional[Element] = Field(
        description="Placeholder element for noteNumber extensions",
        default=None,
        alias="_noteNumber",
    )
    adjudication: Optional[List[ClaimResponseItemAdjudication]] = Field(
        description="Added items detail adjudication",
        default=None,
    )

    @field_validator(
        *(
            "adjudication",
            "noteNumber",
            "net",
            "factor",
            "unitPrice",
            "quantity",
            "modifier",
            "productOrService",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClaimResponseAddItemDetail(BackboneElement):
    """
    The second-tier service adjudications for payor added services.
    """

    productOrService: Optional[CodeableConcept] = Field(
        description="Billing, service, product, or drug code",
        default=None,
    )
    modifier: Optional[List[CodeableConcept]] = Field(
        description="Service/Product billing modifiers",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="Count of products or services",
        default=None,
    )
    unitPrice: Optional[Money] = Field(
        description="Fee, charge or cost per item",
        default=None,
    )
    factor: Optional[Decimal] = Field(
        description="Price scaling factor",
        default=None,
    )
    factor_ext: Optional[Element] = Field(
        description="Placeholder element for factor extensions",
        default=None,
        alias="_factor",
    )
    net: Optional[Money] = Field(
        description="Total item cost",
        default=None,
    )
    noteNumber: Optional[List[PositiveInt]] = Field(
        description="Applicable note numbers",
        default=None,
    )
    noteNumber_ext: Optional[Element] = Field(
        description="Placeholder element for noteNumber extensions",
        default=None,
        alias="_noteNumber",
    )
    adjudication: Optional[List[ClaimResponseItemAdjudication]] = Field(
        description="Added items detail adjudication",
        default=None,
    )
    subDetail: Optional[List[ClaimResponseAddItemDetailSubDetail]] = Field(
        description="Insurer added line items",
        default=None,
    )

    @field_validator(
        *(
            "subDetail",
            "adjudication",
            "noteNumber",
            "net",
            "factor",
            "unitPrice",
            "quantity",
            "modifier",
            "productOrService",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClaimResponseAddItem(BackboneElement):
    """
    The first-tier service adjudications for payor added product or service lines.
    """

    itemSequence: Optional[List[PositiveInt]] = Field(
        description="Item sequence number",
        default=None,
    )
    itemSequence_ext: Optional[Element] = Field(
        description="Placeholder element for itemSequence extensions",
        default=None,
        alias="_itemSequence",
    )
    detailSequence: Optional[List[PositiveInt]] = Field(
        description="Detail sequence number",
        default=None,
    )
    detailSequence_ext: Optional[Element] = Field(
        description="Placeholder element for detailSequence extensions",
        default=None,
        alias="_detailSequence",
    )
    subdetailSequence: Optional[List[PositiveInt]] = Field(
        description="Subdetail sequence number",
        default=None,
    )
    subdetailSequence_ext: Optional[Element] = Field(
        description="Placeholder element for subdetailSequence extensions",
        default=None,
        alias="_subdetailSequence",
    )
    provider: Optional[List[Reference]] = Field(
        description="Authorized providers",
        default=None,
    )
    productOrService: Optional[CodeableConcept] = Field(
        description="Billing, service, product, or drug code",
        default=None,
    )
    modifier: Optional[List[CodeableConcept]] = Field(
        description="Service/Product billing modifiers",
        default=None,
    )
    programCode: Optional[List[CodeableConcept]] = Field(
        description="Program the product or service is provided under",
        default=None,
    )
    servicedDate: Optional[Date] = Field(
        description="Date or dates of service or product delivery",
        default=None,
    )
    servicedDate_ext: Optional[Element] = Field(
        description="Placeholder element for servicedDate extensions",
        default=None,
        alias="_servicedDate",
    )
    servicedPeriod: Optional[Period] = Field(
        description="Date or dates of service or product delivery",
        default=None,
    )
    locationCodeableConcept: Optional[CodeableConcept] = Field(
        description="Place of service or where product was supplied",
        default=None,
    )
    locationAddress: Optional[Address] = Field(
        description="Place of service or where product was supplied",
        default=None,
    )
    locationReference: Optional[Reference] = Field(
        description="Place of service or where product was supplied",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="Count of products or services",
        default=None,
    )
    unitPrice: Optional[Money] = Field(
        description="Fee, charge or cost per item",
        default=None,
    )
    factor: Optional[Decimal] = Field(
        description="Price scaling factor",
        default=None,
    )
    factor_ext: Optional[Element] = Field(
        description="Placeholder element for factor extensions",
        default=None,
        alias="_factor",
    )
    net: Optional[Money] = Field(
        description="Total item cost",
        default=None,
    )
    bodySite: Optional[CodeableConcept] = Field(
        description="Anatomical location",
        default=None,
    )
    subSite: Optional[List[CodeableConcept]] = Field(
        description="Anatomical sub-location",
        default=None,
    )
    noteNumber: Optional[List[PositiveInt]] = Field(
        description="Applicable note numbers",
        default=None,
    )
    noteNumber_ext: Optional[Element] = Field(
        description="Placeholder element for noteNumber extensions",
        default=None,
        alias="_noteNumber",
    )
    adjudication: Optional[List[ClaimResponseItemAdjudication]] = Field(
        description="Added items adjudication",
        default=None,
    )
    detail: Optional[List[ClaimResponseAddItemDetail]] = Field(
        description="Insurer added line details",
        default=None,
    )

    @property
    def serviced(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="serviced",
        )

    @property
    def location(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="location",
        )

    @field_validator(
        *(
            "detail",
            "adjudication",
            "noteNumber",
            "subSite",
            "bodySite",
            "net",
            "factor",
            "unitPrice",
            "quantity",
            "programCode",
            "modifier",
            "productOrService",
            "provider",
            "subdetailSequence",
            "detailSequence",
            "itemSequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def serviced_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Date, Period],
            field_name_base="serviced",
            required=False,
        )

    @model_validator(mode="after")
    def location_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Address, Reference],
            field_name_base="location",
            required=False,
        )


class ClaimResponseTotal(BackboneElement):
    """
    Categorized monetary totals for the adjudication.
    """

    category: Optional[CodeableConcept] = Field(
        description="Type of adjudication information",
        default=None,
    )
    amount: Optional[Money] = Field(
        description="Financial total for the category",
        default=None,
    )

    @field_validator(
        *(
            "amount",
            "category",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClaimResponsePayment(BackboneElement):
    """
    Payment details for the adjudication of the claim.
    """

    type: Optional[CodeableConcept] = Field(
        description="Partial or complete payment",
        default=None,
    )
    adjustment: Optional[Money] = Field(
        description="Payment adjustment for non-claim issues",
        default=None,
    )
    adjustmentReason: Optional[CodeableConcept] = Field(
        description="Explanation for the adjustment",
        default=None,
    )
    date: Optional[Date] = Field(
        description="Expected date of payment",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    amount: Optional[Money] = Field(
        description="Payable amount after adjustment",
        default=None,
    )
    identifier: Optional[Identifier] = Field(
        description="Business identifier for the payment",
        default=None,
    )

    @field_validator(
        *(
            "identifier",
            "amount",
            "date",
            "adjustmentReason",
            "adjustment",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClaimResponseProcessNote(BackboneElement):
    """
    A note that describes or explains adjudication results in a human readable form.
    """

    number: Optional[PositiveInt] = Field(
        description="Note instance identifier",
        default=None,
    )
    number_ext: Optional[Element] = Field(
        description="Placeholder element for number extensions",
        default=None,
        alias="_number",
    )
    type: Optional[Code] = Field(
        description="display | print | printoper",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    text: Optional[String] = Field(
        description="Note explanatory text",
        default=None,
    )
    text_ext: Optional[Element] = Field(
        description="Placeholder element for text extensions",
        default=None,
        alias="_text",
    )
    language: Optional[CodeableConcept] = Field(
        description="Language of the text",
        default=None,
    )

    @field_validator(
        *(
            "language",
            "text",
            "type",
            "number",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClaimResponseInsurance(BackboneElement):
    """
    Financial instruments for reimbursement for the health care products and services specified on the claim.
    """

    sequence: Optional[PositiveInt] = Field(
        description="Insurance instance identifier",
        default=None,
    )
    sequence_ext: Optional[Element] = Field(
        description="Placeholder element for sequence extensions",
        default=None,
        alias="_sequence",
    )
    focal: Optional[Boolean] = Field(
        description="Coverage to be used for adjudication",
        default=None,
    )
    focal_ext: Optional[Element] = Field(
        description="Placeholder element for focal extensions",
        default=None,
        alias="_focal",
    )
    coverage: Optional[Reference] = Field(
        description="Insurance information",
        default=None,
    )
    businessArrangement: Optional[String] = Field(
        description="Additional provider contract number",
        default=None,
    )
    businessArrangement_ext: Optional[Element] = Field(
        description="Placeholder element for businessArrangement extensions",
        default=None,
        alias="_businessArrangement",
    )
    claimResponse: Optional[Reference] = Field(
        description="Adjudication results",
        default=None,
    )

    @field_validator(
        *(
            "claimResponse",
            "businessArrangement",
            "coverage",
            "focal",
            "sequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClaimResponseError(BackboneElement):
    """
    Errors encountered during the processing of the adjudication.
    """

    itemSequence: Optional[PositiveInt] = Field(
        description="Item sequence number",
        default=None,
    )
    itemSequence_ext: Optional[Element] = Field(
        description="Placeholder element for itemSequence extensions",
        default=None,
        alias="_itemSequence",
    )
    detailSequence: Optional[PositiveInt] = Field(
        description="Detail sequence number",
        default=None,
    )
    detailSequence_ext: Optional[Element] = Field(
        description="Placeholder element for detailSequence extensions",
        default=None,
        alias="_detailSequence",
    )
    subDetailSequence: Optional[PositiveInt] = Field(
        description="Subdetail sequence number",
        default=None,
    )
    subDetailSequence_ext: Optional[Element] = Field(
        description="Placeholder element for subDetailSequence extensions",
        default=None,
        alias="_subDetailSequence",
    )
    code: Optional[CodeableConcept] = Field(
        description="Error code detailing processing issues",
        default=None,
    )

    @field_validator(
        *(
            "code",
            "subDetailSequence",
            "detailSequence",
            "itemSequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClaimResponse(DomainResource):
    """
    This resource provides the adjudication details from the processing of a Claim resource.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/ClaimResponse"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business Identifier for a claim response",
        default=None,
    )
    status: Optional[Code] = Field(
        description="active | cancelled | draft | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    type: Optional[CodeableConcept] = Field(
        description="More granular claim type",
        default=None,
    )
    subType: Optional[CodeableConcept] = Field(
        description="More granular claim type",
        default=None,
    )
    use: Optional[Code] = Field(
        description="claim | preauthorization | predetermination",
        default=None,
    )
    use_ext: Optional[Element] = Field(
        description="Placeholder element for use extensions",
        default=None,
        alias="_use",
    )
    patient: Optional[Reference] = Field(
        description="The recipient of the products and services",
        default=None,
    )
    created: Optional[DateTime] = Field(
        description="Response creation date",
        default=None,
    )
    created_ext: Optional[Element] = Field(
        description="Placeholder element for created extensions",
        default=None,
        alias="_created",
    )
    insurer: Optional[Reference] = Field(
        description="Party responsible for reimbursement",
        default=None,
    )
    requestor: Optional[Reference] = Field(
        description="Party responsible for the claim",
        default=None,
    )
    request: Optional[Reference] = Field(
        description="Id of resource triggering adjudication",
        default=None,
    )
    outcome: Optional[Code] = Field(
        description="queued | complete | error | partial",
        default=None,
    )
    outcome_ext: Optional[Element] = Field(
        description="Placeholder element for outcome extensions",
        default=None,
        alias="_outcome",
    )
    disposition: Optional[String] = Field(
        description="Disposition Message",
        default=None,
    )
    disposition_ext: Optional[Element] = Field(
        description="Placeholder element for disposition extensions",
        default=None,
        alias="_disposition",
    )
    preAuthRef: Optional[String] = Field(
        description="Preauthorization reference",
        default=None,
    )
    preAuthRef_ext: Optional[Element] = Field(
        description="Placeholder element for preAuthRef extensions",
        default=None,
        alias="_preAuthRef",
    )
    preAuthPeriod: Optional[Period] = Field(
        description="Preauthorization reference effective period",
        default=None,
    )
    payeeType: Optional[CodeableConcept] = Field(
        description="Party to be paid any benefits payable",
        default=None,
    )
    item: Optional[List[ClaimResponseItem]] = Field(
        description="Adjudication for claim line items",
        default=None,
    )
    addItem: Optional[List[ClaimResponseAddItem]] = Field(
        description="Insurer added line items",
        default=None,
    )
    adjudication: Optional[List[ClaimResponseItemAdjudication]] = Field(
        description="Header-level adjudication",
        default=None,
    )
    total: Optional[List[ClaimResponseTotal]] = Field(
        description="Adjudication totals",
        default=None,
    )
    payment: Optional[ClaimResponsePayment] = Field(
        description="Payment Details",
        default=None,
    )
    fundsReserve: Optional[CodeableConcept] = Field(
        description="Funds reserved status",
        default=None,
    )
    formCode: Optional[CodeableConcept] = Field(
        description="Printed form identifier",
        default=None,
    )
    form: Optional[Attachment] = Field(
        description="Printed reference or actual form",
        default=None,
    )
    processNote: Optional[List[ClaimResponseProcessNote]] = Field(
        description="Note concerning adjudication",
        default=None,
    )
    communicationRequest: Optional[List[Reference]] = Field(
        description="Request for additional information",
        default=None,
    )
    insurance: Optional[List[ClaimResponseInsurance]] = Field(
        description="Patient insurance information",
        default=None,
    )
    error: Optional[List[ClaimResponseError]] = Field(
        description="Processing errors",
        default=None,
    )
    resourceType: Literal["ClaimResponse"] = Field(
        description=None,
        default="ClaimResponse",
    )

    @field_validator(
        *(
            "error",
            "insurance",
            "communicationRequest",
            "processNote",
            "form",
            "formCode",
            "fundsReserve",
            "payment",
            "total",
            "adjudication",
            "addItem",
            "item",
            "payeeType",
            "preAuthPeriod",
            "preAuthRef",
            "disposition",
            "outcome",
            "request",
            "requestor",
            "insurer",
            "created",
            "patient",
            "use",
            "subType",
            "type",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
