# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    PositiveInt,
    Boolean,
    Date,
    Decimal,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    Period,
    BackboneElement,
    Quantity,
    Attachment,
    Address,
    Money,
    DomainResource,
)


class ClaimRelated(BackboneElement):
    """
    Other claims which are related to this claim such as prior submissions or claims for related services or for the same event.
    """

    claim: Optional[Reference] = Field(
        description="Reference to the related claim",
        default=None,
    )
    relationship: Optional[CodeableConcept] = Field(
        description="How the reference claim is related",
        default=None,
    )
    reference: Optional[Identifier] = Field(
        description="File or case reference",
        default=None,
    )

    @field_validator(
        *(
            "reference",
            "relationship",
            "claim",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClaimPayee(BackboneElement):
    """
    The party to be reimbursed for cost of the products and services according to the terms of the policy.
    """

    type: Optional[CodeableConcept] = Field(
        description="Category of recipient",
        default=None,
    )
    party: Optional[Reference] = Field(
        description="Recipient reference",
        default=None,
    )

    @field_validator(
        *(
            "party",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClaimCareTeam(BackboneElement):
    """
    The members of the team who provided the products and services.
    """

    sequence: Optional[PositiveInt] = Field(
        description="Order of care team",
        default=None,
    )
    sequence_ext: Optional[Element] = Field(
        description="Placeholder element for sequence extensions",
        default=None,
        alias="_sequence",
    )
    provider: Optional[Reference] = Field(
        description="Practitioner or organization",
        default=None,
    )
    responsible: Optional[Boolean] = Field(
        description="Indicator of the lead practitioner",
        default=None,
    )
    responsible_ext: Optional[Element] = Field(
        description="Placeholder element for responsible extensions",
        default=None,
        alias="_responsible",
    )
    role: Optional[CodeableConcept] = Field(
        description="Function within the team",
        default=None,
    )
    qualification: Optional[CodeableConcept] = Field(
        description="Practitioner credential or specialization",
        default=None,
    )

    @field_validator(
        *(
            "qualification",
            "role",
            "responsible",
            "provider",
            "sequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClaimSupportingInfo(BackboneElement):
    """
    Additional information codes regarding exceptions, special considerations, the condition, situation, prior or concurrent issues.
    """

    sequence: Optional[PositiveInt] = Field(
        description="Information instance identifier",
        default=None,
    )
    sequence_ext: Optional[Element] = Field(
        description="Placeholder element for sequence extensions",
        default=None,
        alias="_sequence",
    )
    category: Optional[CodeableConcept] = Field(
        description="Classification of the supplied information",
        default=None,
    )
    code: Optional[CodeableConcept] = Field(
        description="Type of information",
        default=None,
    )
    timingDate: Optional[Date] = Field(
        description="When it occurred",
        default=None,
    )
    timingDate_ext: Optional[Element] = Field(
        description="Placeholder element for timingDate extensions",
        default=None,
        alias="_timingDate",
    )
    timingPeriod: Optional[Period] = Field(
        description="When it occurred",
        default=None,
    )
    valueBoolean: Optional[Boolean] = Field(
        description="Data to be provided",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueString: Optional[String] = Field(
        description="Data to be provided",
        default=None,
    )
    valueString_ext: Optional[Element] = Field(
        description="Placeholder element for valueString extensions",
        default=None,
        alias="_valueString",
    )
    valueQuantity: Optional[Quantity] = Field(
        description="Data to be provided",
        default=None,
    )
    valueAttachment: Optional[Attachment] = Field(
        description="Data to be provided",
        default=None,
    )
    valueReference: Optional[Reference] = Field(
        description="Data to be provided",
        default=None,
    )
    reason: Optional[CodeableConcept] = Field(
        description="Explanation for the information",
        default=None,
    )

    @property
    def timing(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="timing",
        )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *(
            "reason",
            "code",
            "category",
            "sequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def timing_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Date, Period],
            field_name_base="timing",
            required=False,
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Boolean, String, Quantity, Attachment, Reference],
            field_name_base="value",
            required=False,
        )


class ClaimDiagnosis(BackboneElement):
    """
    Information about diagnoses relevant to the claim items.
    """

    sequence: Optional[PositiveInt] = Field(
        description="Diagnosis instance identifier",
        default=None,
    )
    sequence_ext: Optional[Element] = Field(
        description="Placeholder element for sequence extensions",
        default=None,
        alias="_sequence",
    )
    diagnosisCodeableConcept: Optional[CodeableConcept] = Field(
        description="Nature of illness or problem",
        default=None,
    )
    diagnosisReference: Optional[Reference] = Field(
        description="Nature of illness or problem",
        default=None,
    )
    type: Optional[List[CodeableConcept]] = Field(
        description="Timing or nature of the diagnosis",
        default=None,
    )
    onAdmission: Optional[CodeableConcept] = Field(
        description="Present on admission",
        default=None,
    )
    packageCode: Optional[CodeableConcept] = Field(
        description="Package billing code",
        default=None,
    )

    @property
    def diagnosis(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="diagnosis",
        )

    @field_validator(
        *(
            "packageCode",
            "onAdmission",
            "type",
            "sequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def diagnosis_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Reference],
            field_name_base="diagnosis",
            required=True,
        )


class ClaimProcedure(BackboneElement):
    """
    Procedures performed on the patient relevant to the billing items with the claim.
    """

    sequence: Optional[PositiveInt] = Field(
        description="Procedure instance identifier",
        default=None,
    )
    sequence_ext: Optional[Element] = Field(
        description="Placeholder element for sequence extensions",
        default=None,
        alias="_sequence",
    )
    type: Optional[List[CodeableConcept]] = Field(
        description="Category of Procedure",
        default=None,
    )
    date: Optional[DateTime] = Field(
        description="When the procedure was performed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    procedureCodeableConcept: Optional[CodeableConcept] = Field(
        description="Specific clinical procedure",
        default=None,
    )
    procedureReference: Optional[Reference] = Field(
        description="Specific clinical procedure",
        default=None,
    )
    udi: Optional[List[Reference]] = Field(
        description="Unique device identifier",
        default=None,
    )

    @property
    def procedure(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="procedure",
        )

    @field_validator(
        *(
            "udi",
            "date",
            "type",
            "sequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def procedure_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Reference],
            field_name_base="procedure",
            required=True,
        )


class ClaimInsurance(BackboneElement):
    """
    Financial instruments for reimbursement for the health care products and services specified on the claim.
    """

    sequence: Optional[PositiveInt] = Field(
        description="Insurance instance identifier",
        default=None,
    )
    sequence_ext: Optional[Element] = Field(
        description="Placeholder element for sequence extensions",
        default=None,
        alias="_sequence",
    )
    focal: Optional[Boolean] = Field(
        description="Coverage to be used for adjudication",
        default=None,
    )
    focal_ext: Optional[Element] = Field(
        description="Placeholder element for focal extensions",
        default=None,
        alias="_focal",
    )
    identifier: Optional[Identifier] = Field(
        description="Pre-assigned Claim number",
        default=None,
    )
    coverage: Optional[Reference] = Field(
        description="Insurance information",
        default=None,
    )
    businessArrangement: Optional[String] = Field(
        description="Additional provider contract number",
        default=None,
    )
    businessArrangement_ext: Optional[Element] = Field(
        description="Placeholder element for businessArrangement extensions",
        default=None,
        alias="_businessArrangement",
    )
    preAuthRef: Optional[List[String]] = Field(
        description="Prior authorization reference number",
        default=None,
    )
    preAuthRef_ext: Optional[Element] = Field(
        description="Placeholder element for preAuthRef extensions",
        default=None,
        alias="_preAuthRef",
    )
    claimResponse: Optional[Reference] = Field(
        description="Adjudication results",
        default=None,
    )

    @field_validator(
        *(
            "claimResponse",
            "preAuthRef",
            "businessArrangement",
            "coverage",
            "identifier",
            "focal",
            "sequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClaimAccident(BackboneElement):
    """
    Details of an accident which resulted in injuries which required the products and services listed in the claim.
    """

    date: Optional[Date] = Field(
        description="When the incident occurred",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    type: Optional[CodeableConcept] = Field(
        description="The nature of the accident",
        default=None,
    )
    locationAddress: Optional[Address] = Field(
        description="Where the event occurred",
        default=None,
    )
    locationReference: Optional[Reference] = Field(
        description="Where the event occurred",
        default=None,
    )

    @property
    def location(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="location",
        )

    @field_validator(
        *(
            "type",
            "date",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def location_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Address, Reference],
            field_name_base="location",
            required=False,
        )


class ClaimItemDetailSubDetail(BackboneElement):
    """
    A claim detail line. Either a simple (a product or service) or a 'group' of sub-details which are simple items.
    """

    sequence: Optional[PositiveInt] = Field(
        description="Item instance identifier",
        default=None,
    )
    sequence_ext: Optional[Element] = Field(
        description="Placeholder element for sequence extensions",
        default=None,
        alias="_sequence",
    )
    revenue: Optional[CodeableConcept] = Field(
        description="Revenue or cost center code",
        default=None,
    )
    category: Optional[CodeableConcept] = Field(
        description="Benefit classification",
        default=None,
    )
    productOrService: Optional[CodeableConcept] = Field(
        description="Billing, service, product, or drug code",
        default=None,
    )
    modifier: Optional[List[CodeableConcept]] = Field(
        description="Service/Product billing modifiers",
        default=None,
    )
    programCode: Optional[List[CodeableConcept]] = Field(
        description="Program the product or service is provided under",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="Count of products or services",
        default=None,
    )
    unitPrice: Optional[Money] = Field(
        description="Fee, charge or cost per item",
        default=None,
    )
    factor: Optional[Decimal] = Field(
        description="Price scaling factor",
        default=None,
    )
    factor_ext: Optional[Element] = Field(
        description="Placeholder element for factor extensions",
        default=None,
        alias="_factor",
    )
    net: Optional[Money] = Field(
        description="Total item cost",
        default=None,
    )
    udi: Optional[List[Reference]] = Field(
        description="Unique device identifier",
        default=None,
    )

    @field_validator(
        *(
            "udi",
            "net",
            "factor",
            "unitPrice",
            "quantity",
            "programCode",
            "modifier",
            "productOrService",
            "category",
            "revenue",
            "sequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClaimItemDetail(BackboneElement):
    """
    A claim detail line. Either a simple (a product or service) or a 'group' of sub-details which are simple items.
    """

    sequence: Optional[PositiveInt] = Field(
        description="Item instance identifier",
        default=None,
    )
    sequence_ext: Optional[Element] = Field(
        description="Placeholder element for sequence extensions",
        default=None,
        alias="_sequence",
    )
    revenue: Optional[CodeableConcept] = Field(
        description="Revenue or cost center code",
        default=None,
    )
    category: Optional[CodeableConcept] = Field(
        description="Benefit classification",
        default=None,
    )
    productOrService: Optional[CodeableConcept] = Field(
        description="Billing, service, product, or drug code",
        default=None,
    )
    modifier: Optional[List[CodeableConcept]] = Field(
        description="Service/Product billing modifiers",
        default=None,
    )
    programCode: Optional[List[CodeableConcept]] = Field(
        description="Program the product or service is provided under",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="Count of products or services",
        default=None,
    )
    unitPrice: Optional[Money] = Field(
        description="Fee, charge or cost per item",
        default=None,
    )
    factor: Optional[Decimal] = Field(
        description="Price scaling factor",
        default=None,
    )
    factor_ext: Optional[Element] = Field(
        description="Placeholder element for factor extensions",
        default=None,
        alias="_factor",
    )
    net: Optional[Money] = Field(
        description="Total item cost",
        default=None,
    )
    udi: Optional[List[Reference]] = Field(
        description="Unique device identifier",
        default=None,
    )
    subDetail: Optional[List[ClaimItemDetailSubDetail]] = Field(
        description="Product or service provided",
        default=None,
    )

    @field_validator(
        *(
            "subDetail",
            "udi",
            "net",
            "factor",
            "unitPrice",
            "quantity",
            "programCode",
            "modifier",
            "productOrService",
            "category",
            "revenue",
            "sequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ClaimItem(BackboneElement):
    """
    A claim line. Either a simple  product or service or a 'group' of details which can each be a simple items or groups of sub-details.
    """

    sequence: Optional[PositiveInt] = Field(
        description="Item instance identifier",
        default=None,
    )
    sequence_ext: Optional[Element] = Field(
        description="Placeholder element for sequence extensions",
        default=None,
        alias="_sequence",
    )
    careTeamSequence: Optional[List[PositiveInt]] = Field(
        description="Applicable careTeam members",
        default=None,
    )
    careTeamSequence_ext: Optional[Element] = Field(
        description="Placeholder element for careTeamSequence extensions",
        default=None,
        alias="_careTeamSequence",
    )
    diagnosisSequence: Optional[List[PositiveInt]] = Field(
        description="Applicable diagnoses",
        default=None,
    )
    diagnosisSequence_ext: Optional[Element] = Field(
        description="Placeholder element for diagnosisSequence extensions",
        default=None,
        alias="_diagnosisSequence",
    )
    procedureSequence: Optional[List[PositiveInt]] = Field(
        description="Applicable procedures",
        default=None,
    )
    procedureSequence_ext: Optional[Element] = Field(
        description="Placeholder element for procedureSequence extensions",
        default=None,
        alias="_procedureSequence",
    )
    informationSequence: Optional[List[PositiveInt]] = Field(
        description="Applicable exception and supporting information",
        default=None,
    )
    informationSequence_ext: Optional[Element] = Field(
        description="Placeholder element for informationSequence extensions",
        default=None,
        alias="_informationSequence",
    )
    revenue: Optional[CodeableConcept] = Field(
        description="Revenue or cost center code",
        default=None,
    )
    category: Optional[CodeableConcept] = Field(
        description="Benefit classification",
        default=None,
    )
    productOrService: Optional[CodeableConcept] = Field(
        description="Billing, service, product, or drug code",
        default=None,
    )
    modifier: Optional[List[CodeableConcept]] = Field(
        description="Product or service billing modifiers",
        default=None,
    )
    programCode: Optional[List[CodeableConcept]] = Field(
        description="Program the product or service is provided under",
        default=None,
    )
    servicedDate: Optional[Date] = Field(
        description="Date or dates of service or product delivery",
        default=None,
    )
    servicedDate_ext: Optional[Element] = Field(
        description="Placeholder element for servicedDate extensions",
        default=None,
        alias="_servicedDate",
    )
    servicedPeriod: Optional[Period] = Field(
        description="Date or dates of service or product delivery",
        default=None,
    )
    locationCodeableConcept: Optional[CodeableConcept] = Field(
        description="Place of service or where product was supplied",
        default=None,
    )
    locationAddress: Optional[Address] = Field(
        description="Place of service or where product was supplied",
        default=None,
    )
    locationReference: Optional[Reference] = Field(
        description="Place of service or where product was supplied",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="Count of products or services",
        default=None,
    )
    unitPrice: Optional[Money] = Field(
        description="Fee, charge or cost per item",
        default=None,
    )
    factor: Optional[Decimal] = Field(
        description="Price scaling factor",
        default=None,
    )
    factor_ext: Optional[Element] = Field(
        description="Placeholder element for factor extensions",
        default=None,
        alias="_factor",
    )
    net: Optional[Money] = Field(
        description="Total item cost",
        default=None,
    )
    udi: Optional[List[Reference]] = Field(
        description="Unique device identifier",
        default=None,
    )
    bodySite: Optional[CodeableConcept] = Field(
        description="Anatomical location",
        default=None,
    )
    subSite: Optional[List[CodeableConcept]] = Field(
        description="Anatomical sub-location",
        default=None,
    )
    encounter: Optional[List[Reference]] = Field(
        description="Encounters related to this billed item",
        default=None,
    )
    detail: Optional[List[ClaimItemDetail]] = Field(
        description="Product or service provided",
        default=None,
    )

    @property
    def serviced(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="serviced",
        )

    @property
    def location(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="location",
        )

    @field_validator(
        *(
            "detail",
            "encounter",
            "subSite",
            "bodySite",
            "udi",
            "net",
            "factor",
            "unitPrice",
            "quantity",
            "programCode",
            "modifier",
            "productOrService",
            "category",
            "revenue",
            "informationSequence",
            "procedureSequence",
            "diagnosisSequence",
            "careTeamSequence",
            "sequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def serviced_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Date, Period],
            field_name_base="serviced",
            required=False,
        )

    @model_validator(mode="after")
    def location_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Address, Reference],
            field_name_base="location",
            required=False,
        )


class Claim(DomainResource):
    """
    A provider issued list of professional services and products which have been provided, or are to be provided, to a patient which is sent to an insurer for reimbursement.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Claim"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business Identifier for claim",
        default=None,
    )
    status: Optional[Code] = Field(
        description="active | cancelled | draft | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    type: Optional[CodeableConcept] = Field(
        description="Category or discipline",
        default=None,
    )
    subType: Optional[CodeableConcept] = Field(
        description="More granular claim type",
        default=None,
    )
    use: Optional[Code] = Field(
        description="claim | preauthorization | predetermination",
        default=None,
    )
    use_ext: Optional[Element] = Field(
        description="Placeholder element for use extensions",
        default=None,
        alias="_use",
    )
    patient: Optional[Reference] = Field(
        description="The recipient of the products and services",
        default=None,
    )
    billablePeriod: Optional[Period] = Field(
        description="Relevant time frame for the claim",
        default=None,
    )
    created: Optional[DateTime] = Field(
        description="Resource creation date",
        default=None,
    )
    created_ext: Optional[Element] = Field(
        description="Placeholder element for created extensions",
        default=None,
        alias="_created",
    )
    enterer: Optional[Reference] = Field(
        description="Author of the claim",
        default=None,
    )
    insurer: Optional[Reference] = Field(
        description="Target",
        default=None,
    )
    provider: Optional[Reference] = Field(
        description="Party responsible for the claim",
        default=None,
    )
    priority: Optional[CodeableConcept] = Field(
        description="Desired processing ugency",
        default=None,
    )
    fundsReserve: Optional[CodeableConcept] = Field(
        description="For whom to reserve funds",
        default=None,
    )
    related: Optional[List[ClaimRelated]] = Field(
        description="Prior or corollary claims",
        default=None,
    )
    prescription: Optional[Reference] = Field(
        description="Prescription authorizing services and products",
        default=None,
    )
    originalPrescription: Optional[Reference] = Field(
        description="Original prescription if superseded by fulfiller",
        default=None,
    )
    payee: Optional[ClaimPayee] = Field(
        description="Recipient of benefits payable",
        default=None,
    )
    referral: Optional[Reference] = Field(
        description="Treatment referral",
        default=None,
    )
    facility: Optional[Reference] = Field(
        description="Servicing facility",
        default=None,
    )
    careTeam: Optional[List[ClaimCareTeam]] = Field(
        description="Members of the care team",
        default=None,
    )
    supportingInfo: Optional[List[ClaimSupportingInfo]] = Field(
        description="Supporting information",
        default=None,
    )
    diagnosis: Optional[List[ClaimDiagnosis]] = Field(
        description="Pertinent diagnosis information",
        default=None,
    )
    procedure: Optional[List[ClaimProcedure]] = Field(
        description="Clinical procedures performed",
        default=None,
    )
    insurance: Optional[List[ClaimInsurance]] = Field(
        description="Patient insurance information",
        default=None,
    )
    accident: Optional[ClaimAccident] = Field(
        description="Details of the event",
        default=None,
    )
    item: Optional[List[ClaimItem]] = Field(
        description="Product or service provided",
        default=None,
    )
    total: Optional[Money] = Field(
        description="Total claim cost",
        default=None,
    )
    resourceType: Literal["Claim"] = Field(
        description=None,
        default="Claim",
    )

    @field_validator(
        *(
            "total",
            "item",
            "accident",
            "insurance",
            "procedure",
            "diagnosis",
            "supportingInfo",
            "careTeam",
            "facility",
            "referral",
            "payee",
            "originalPrescription",
            "prescription",
            "related",
            "fundsReserve",
            "priority",
            "provider",
            "insurer",
            "enterer",
            "created",
            "billablePeriod",
            "patient",
            "use",
            "subType",
            "type",
            "status",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
