# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Canonical,
    DateTime,
    Decimal,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    CodeableConcept,
    Period,
    Timing,
    BackboneElement,
    Quantity,
    Money,
    Annotation,
    DomainResource,
)


class ChargeItemPerformer(BackboneElement):
    """
    Indicates who or what performed or participated in the charged service.
    """

    function: Optional[CodeableConcept] = Field(
        description="What type of performance was done",
        default=None,
    )
    actor: Optional[Reference] = Field(
        description="Individual who was performing",
        default=None,
    )

    @field_validator(
        *(
            "actor",
            "function",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ChargeItem(DomainResource):
    """
    The resource ChargeItem describes the provision of healthcare provider products for a certain patient, therefore referring not only to the product, but containing in addition details of the provision, like date, time, amounts and participating organizations and persons. Main Usage of the ChargeItem is to enable the billing process and internal cost allocation.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/ChargeItem"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business Identifier for item",
        default=None,
    )
    definitionUri: Optional[List[Uri]] = Field(
        description="Defining information about the code of this charge item",
        default=None,
    )
    definitionUri_ext: Optional[Element] = Field(
        description="Placeholder element for definitionUri extensions",
        default=None,
        alias="_definitionUri",
    )
    definitionCanonical: Optional[List[Canonical]] = Field(
        description="Resource defining the code of this ChargeItem",
        default=None,
    )
    definitionCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for definitionCanonical extensions",
        default=None,
        alias="_definitionCanonical",
    )
    status: Optional[Code] = Field(
        description="planned | billable | not-billable | aborted | billed | entered-in-error | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    partOf: Optional[List[Reference]] = Field(
        description="Part of referenced ChargeItem",
        default=None,
    )
    code: Optional[CodeableConcept] = Field(
        description="A code that identifies the charge, like a billing code",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="Individual service was done for/to",
        default=None,
    )
    context: Optional[Reference] = Field(
        description="Encounter / Episode associated with event",
        default=None,
    )
    occurrenceDateTime: Optional[DateTime] = Field(
        description="When the charged service was applied",
        default=None,
    )
    occurrenceDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for occurrenceDateTime extensions",
        default=None,
        alias="_occurrenceDateTime",
    )
    occurrencePeriod: Optional[Period] = Field(
        description="When the charged service was applied",
        default=None,
    )
    occurrenceTiming: Optional[Timing] = Field(
        description="When the charged service was applied",
        default=None,
    )
    performer: Optional[List[ChargeItemPerformer]] = Field(
        description="Who performed charged service",
        default=None,
    )
    performingOrganization: Optional[Reference] = Field(
        description="Organization providing the charged service",
        default=None,
    )
    requestingOrganization: Optional[Reference] = Field(
        description="Organization requesting the charged service",
        default=None,
    )
    costCenter: Optional[Reference] = Field(
        description="Organization that has ownership of the (potential, future) revenue",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="Quantity of which the charge item has been serviced",
        default=None,
    )
    bodysite: Optional[List[CodeableConcept]] = Field(
        description="Anatomical location, if relevant",
        default=None,
    )
    factorOverride: Optional[Decimal] = Field(
        description="Factor overriding the associated rules",
        default=None,
    )
    factorOverride_ext: Optional[Element] = Field(
        description="Placeholder element for factorOverride extensions",
        default=None,
        alias="_factorOverride",
    )
    priceOverride: Optional[Money] = Field(
        description="Price overriding the associated rules",
        default=None,
    )
    overrideReason: Optional[String] = Field(
        description="Reason for overriding the list price/factor",
        default=None,
    )
    overrideReason_ext: Optional[Element] = Field(
        description="Placeholder element for overrideReason extensions",
        default=None,
        alias="_overrideReason",
    )
    enterer: Optional[Reference] = Field(
        description="Individual who was entering",
        default=None,
    )
    enteredDate: Optional[DateTime] = Field(
        description="Date the charge item was entered",
        default=None,
    )
    enteredDate_ext: Optional[Element] = Field(
        description="Placeholder element for enteredDate extensions",
        default=None,
        alias="_enteredDate",
    )
    reason: Optional[List[CodeableConcept]] = Field(
        description="Why was the charged  service rendered?",
        default=None,
    )
    service: Optional[List[Reference]] = Field(
        description="Which rendered service is being charged?",
        default=None,
    )
    productReference: Optional[Reference] = Field(
        description="Product charged",
        default=None,
    )
    productCodeableConcept: Optional[CodeableConcept] = Field(
        description="Product charged",
        default=None,
    )
    account: Optional[List[Reference]] = Field(
        description="Account to place this charge",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Comments made about the ChargeItem",
        default=None,
    )
    supportingInformation: Optional[List[Reference]] = Field(
        description="Further information supporting this charge",
        default=None,
    )
    resourceType: Literal["ChargeItem"] = Field(
        description=None,
        default="ChargeItem",
    )

    @property
    def occurrence(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="occurrence",
        )

    @property
    def product(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="product",
        )

    @field_validator(
        *(
            "supportingInformation",
            "note",
            "account",
            "service",
            "reason",
            "enteredDate",
            "enterer",
            "overrideReason",
            "priceOverride",
            "factorOverride",
            "bodysite",
            "quantity",
            "costCenter",
            "requestingOrganization",
            "performingOrganization",
            "performer",
            "context",
            "subject",
            "code",
            "partOf",
            "status",
            "definitionCanonical",
            "definitionUri",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def occurrence_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, Period, Timing],
            field_name_base="occurrence",
            required=False,
        )

    @model_validator(mode="after")
    def product_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Reference, CodeableConcept],
            field_name_base="product",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
