# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Canonical,
    DateTime,
    Boolean,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    CodeableConcept,
    Period,
    BackboneElement,
    Annotation,
    Timing,
    Quantity,
    DomainResource,
)


class CarePlanActivityDetail(BackboneElement):
    """
    A simple summary of a planned activity suitable for a general care plan system (e.g. form driven) that doesn't know about specific resources such as procedure etc.
    """

    kind: Optional[Code] = Field(
        description="Appointment | CommunicationRequest | DeviceRequest | MedicationRequest | NutritionOrder | Task | ServiceRequest | VisionPrescription",
        default=None,
    )
    kind_ext: Optional[Element] = Field(
        description="Placeholder element for kind extensions",
        default=None,
        alias="_kind",
    )
    instantiatesCanonical: Optional[List[Canonical]] = Field(
        description="Instantiates FHIR protocol or definition",
        default=None,
    )
    instantiatesCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesCanonical extensions",
        default=None,
        alias="_instantiatesCanonical",
    )
    instantiatesUri: Optional[List[Uri]] = Field(
        description="Instantiates external protocol or definition",
        default=None,
    )
    instantiatesUri_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesUri extensions",
        default=None,
        alias="_instantiatesUri",
    )
    code: Optional[CodeableConcept] = Field(
        description="Detail type of activity",
        default=None,
    )
    reasonCode: Optional[List[CodeableConcept]] = Field(
        description="Why activity should be done or why activity was prohibited",
        default=None,
    )
    reasonReference: Optional[List[Reference]] = Field(
        description="Why activity is needed",
        default=None,
    )
    goal: Optional[List[Reference]] = Field(
        description="Goals this activity relates to",
        default=None,
    )
    status: Optional[Code] = Field(
        description="not-started | scheduled | in-progress | on-hold | completed | cancelled | stopped | unknown | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    statusReason: Optional[CodeableConcept] = Field(
        description="Reason for current status",
        default=None,
    )
    doNotPerform: Optional[Boolean] = Field(
        description="If true, activity is prohibiting action",
        default=None,
    )
    doNotPerform_ext: Optional[Element] = Field(
        description="Placeholder element for doNotPerform extensions",
        default=None,
        alias="_doNotPerform",
    )
    scheduledTiming: Optional[Timing] = Field(
        description="When activity is to occur",
        default=None,
    )
    scheduledPeriod: Optional[Period] = Field(
        description="When activity is to occur",
        default=None,
    )
    scheduledString: Optional[String] = Field(
        description="When activity is to occur",
        default=None,
    )
    scheduledString_ext: Optional[Element] = Field(
        description="Placeholder element for scheduledString extensions",
        default=None,
        alias="_scheduledString",
    )
    location: Optional[Reference] = Field(
        description="Where it should happen",
        default=None,
    )
    performer: Optional[List[Reference]] = Field(
        description="Who will be responsible?",
        default=None,
    )
    productCodeableConcept: Optional[CodeableConcept] = Field(
        description="What is to be administered/supplied",
        default=None,
    )
    productReference: Optional[Reference] = Field(
        description="What is to be administered/supplied",
        default=None,
    )
    dailyAmount: Optional[Quantity] = Field(
        description="How to consume/day?",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="How much to administer/supply/consume",
        default=None,
    )
    description: Optional[String] = Field(
        description="Extra info describing activity to perform",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )

    @property
    def scheduled(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="scheduled",
        )

    @property
    def product(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="product",
        )

    @field_validator(
        *(
            "description",
            "quantity",
            "dailyAmount",
            "performer",
            "location",
            "doNotPerform",
            "statusReason",
            "status",
            "goal",
            "reasonReference",
            "reasonCode",
            "code",
            "instantiatesUri",
            "instantiatesCanonical",
            "kind",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def scheduled_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Timing, Period, String],
            field_name_base="scheduled",
            required=False,
        )

    @model_validator(mode="after")
    def product_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Reference],
            field_name_base="product",
            required=False,
        )


class CarePlanActivity(BackboneElement):
    """
    Identifies a planned action to occur as part of the plan.  For example, a medication to be used, lab tests to perform, self-monitoring, education, etc.
    """

    outcomeCodeableConcept: Optional[List[CodeableConcept]] = Field(
        description="Results of the activity",
        default=None,
    )
    outcomeReference: Optional[List[Reference]] = Field(
        description="Appointment, Encounter, Procedure, etc.",
        default=None,
    )
    progress: Optional[List[Annotation]] = Field(
        description="Comments about the activity status/progress",
        default=None,
    )
    reference: Optional[Reference] = Field(
        description="Activity details defined in specific resource",
        default=None,
    )
    detail: Optional[CarePlanActivityDetail] = Field(
        description="In-line definition of activity",
        default=None,
    )

    @field_validator(
        *(
            "detail",
            "reference",
            "progress",
            "outcomeReference",
            "outcomeCodeableConcept",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CarePlan(DomainResource):
    """
    Describes the intention of how one or more practitioners intend to deliver care for a particular patient, group or community for a period of time, possibly limited to care for a specific condition or set of conditions.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/CarePlan"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="External Ids for this plan",
        default=None,
    )
    instantiatesCanonical: Optional[List[Canonical]] = Field(
        description="Instantiates FHIR protocol or definition",
        default=None,
    )
    instantiatesCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesCanonical extensions",
        default=None,
        alias="_instantiatesCanonical",
    )
    instantiatesUri: Optional[List[Uri]] = Field(
        description="Instantiates external protocol or definition",
        default=None,
    )
    instantiatesUri_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesUri extensions",
        default=None,
        alias="_instantiatesUri",
    )
    basedOn: Optional[List[Reference]] = Field(
        description="Fulfills CarePlan",
        default=None,
    )
    replaces: Optional[List[Reference]] = Field(
        description="CarePlan replaced by this CarePlan",
        default=None,
    )
    partOf: Optional[List[Reference]] = Field(
        description="Part of referenced CarePlan",
        default=None,
    )
    status: Optional[Code] = Field(
        description="draft | active | on-hold | revoked | completed | entered-in-error | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    intent: Optional[Code] = Field(
        description="proposal | plan | order | option",
        default=None,
    )
    intent_ext: Optional[Element] = Field(
        description="Placeholder element for intent extensions",
        default=None,
        alias="_intent",
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="Type of plan",
        default=None,
    )
    title: Optional[String] = Field(
        description="Human-friendly name for the care plan",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    description: Optional[String] = Field(
        description="Summary of nature of plan",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    subject: Optional[Reference] = Field(
        description="Who the care plan is for",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Encounter created as part of",
        default=None,
    )
    period: Optional[Period] = Field(
        description="Time period plan covers",
        default=None,
    )
    created: Optional[DateTime] = Field(
        description="Date record was first recorded",
        default=None,
    )
    created_ext: Optional[Element] = Field(
        description="Placeholder element for created extensions",
        default=None,
        alias="_created",
    )
    author: Optional[Reference] = Field(
        description="Who is the designated responsible party",
        default=None,
    )
    contributor: Optional[List[Reference]] = Field(
        description="Who provided the content of the care plan",
        default=None,
    )
    careTeam: Optional[List[Reference]] = Field(
        description="Who\u0027s involved in plan?",
        default=None,
    )
    addresses: Optional[List[Reference]] = Field(
        description="Health issues this plan addresses",
        default=None,
    )
    supportingInfo: Optional[List[Reference]] = Field(
        description="Information considered as part of plan",
        default=None,
    )
    goal: Optional[List[Reference]] = Field(
        description="Desired outcome of plan",
        default=None,
    )
    activity: Optional[List[CarePlanActivity]] = Field(
        description="Action to occur as part of plan",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Comments about the plan",
        default=None,
    )
    resourceType: Literal["CarePlan"] = Field(
        description=None,
        default="CarePlan",
    )

    @field_validator(
        *(
            "note",
            "activity",
            "goal",
            "supportingInfo",
            "addresses",
            "careTeam",
            "contributor",
            "author",
            "created",
            "period",
            "encounter",
            "subject",
            "description",
            "title",
            "category",
            "intent",
            "status",
            "partOf",
            "replaces",
            "basedOn",
            "instantiatesUri",
            "instantiatesCanonical",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("activity",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cpl_3_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="detail.empty() or reference.empty()",
            human="Provide a reference or detail, not both",
            key="cpl-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
