# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    DateTime,
    Markdown,
    Canonical,
    Url,
    UnsignedInt,
)

from fhircraft.fhir.resources.datatypes.R4.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    ContactDetail,
    UsageContext,
    CodeableConcept,
    BackboneElement,
    Reference,
    Coding,
    DomainResource,
)


class CapabilityStatementSoftware(BackboneElement):
    """
    Software that is covered by this capability statement.  It is used when the capability statement describes the capabilities of a particular software version, independent of an installation.
    """

    name: Optional[String] = Field(
        description="A name the software is known by",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    version: Optional[String] = Field(
        description="Version covered by this statement",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    releaseDate: Optional[DateTime] = Field(
        description="Date this version was released",
        default=None,
    )
    releaseDate_ext: Optional[Element] = Field(
        description="Placeholder element for releaseDate extensions",
        default=None,
        alias="_releaseDate",
    )

    @field_validator(
        *(
            "releaseDate",
            "version",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CapabilityStatementImplementation(BackboneElement):
    """
    Identifies a specific implementation instance that is described by the capability statement - i.e. a particular installation, rather than the capabilities of a software program.
    """

    description: Optional[String] = Field(
        description="Describes this specific instance",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    url: Optional[Url] = Field(
        description="Base URL for the installation",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    custodian: Optional[Reference] = Field(
        description="Organization that manages the data",
        default=None,
    )

    @field_validator(
        *(
            "custodian",
            "url",
            "description",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CapabilityStatementRestSecurity(BackboneElement):
    """
    Information about security implementation from an interface perspective - what a client needs to know.
    """

    cors: Optional[Boolean] = Field(
        description="Adds CORS Headers (http://enable-cors.org/)",
        default=None,
    )
    cors_ext: Optional[Element] = Field(
        description="Placeholder element for cors extensions",
        default=None,
        alias="_cors",
    )
    service: Optional[List[CodeableConcept]] = Field(
        description="OAuth | SMART-on-FHIR | NTLM | Basic | Kerberos | Certificates",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="General description of how security works",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )

    @field_validator(
        *(
            "description",
            "service",
            "cors",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CapabilityStatementRestResourceInteraction(BackboneElement):
    """
    Identifies a restful operation supported by the solution.
    """

    code: Optional[Code] = Field(
        description="read | vread | update | patch | delete | history-instance | history-type | create | search-type",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    documentation: Optional[Markdown] = Field(
        description="Anything special about operation behavior",
        default=None,
    )
    documentation_ext: Optional[Element] = Field(
        description="Placeholder element for documentation extensions",
        default=None,
        alias="_documentation",
    )

    @field_validator(
        *(
            "documentation",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CapabilityStatementRestResourceSearchParam(BackboneElement):
    """
    Search parameters for implementations to support and/or make use of - either references to ones defined in the specification, or additional ones defined for/by the implementation.
    """

    name: Optional[String] = Field(
        description="Name of search parameter",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    definition: Optional[Canonical] = Field(
        description="Source of definition for parameter",
        default=None,
    )
    definition_ext: Optional[Element] = Field(
        description="Placeholder element for definition extensions",
        default=None,
        alias="_definition",
    )
    type: Optional[Code] = Field(
        description="number | date | string | token | reference | composite | quantity | uri | special",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    documentation: Optional[Markdown] = Field(
        description="Server-specific usage",
        default=None,
    )
    documentation_ext: Optional[Element] = Field(
        description="Placeholder element for documentation extensions",
        default=None,
        alias="_documentation",
    )

    @field_validator(
        *(
            "documentation",
            "type",
            "definition",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CapabilityStatementRestResourceOperation(BackboneElement):
    """
    Definition of an operation or a named query together with its parameters and their meaning and type. Consult the definition of the operation for details about how to invoke the operation, and the parameters.
    """

    name: Optional[String] = Field(
        description="Name by which the operation/query is invoked",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    definition: Optional[Canonical] = Field(
        description="The defined operation/query",
        default=None,
    )
    definition_ext: Optional[Element] = Field(
        description="Placeholder element for definition extensions",
        default=None,
        alias="_definition",
    )
    documentation: Optional[Markdown] = Field(
        description="Specific details about operation behavior",
        default=None,
    )
    documentation_ext: Optional[Element] = Field(
        description="Placeholder element for documentation extensions",
        default=None,
        alias="_documentation",
    )

    @field_validator(
        *(
            "documentation",
            "definition",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CapabilityStatementRestResource(BackboneElement):
    """
    A specification of the restful capabilities of the solution for a specific resource type.
    """

    type: Optional[Code] = Field(
        description="A resource type that is supported",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    profile: Optional[Canonical] = Field(
        description="Base System profile for all uses of resource",
        default=None,
    )
    profile_ext: Optional[Element] = Field(
        description="Placeholder element for profile extensions",
        default=None,
        alias="_profile",
    )
    supportedProfile: Optional[List[Canonical]] = Field(
        description="Profiles for use cases supported",
        default=None,
    )
    supportedProfile_ext: Optional[Element] = Field(
        description="Placeholder element for supportedProfile extensions",
        default=None,
        alias="_supportedProfile",
    )
    documentation: Optional[Markdown] = Field(
        description="Additional information about the use of the resource type",
        default=None,
    )
    documentation_ext: Optional[Element] = Field(
        description="Placeholder element for documentation extensions",
        default=None,
        alias="_documentation",
    )
    interaction: Optional[List[CapabilityStatementRestResourceInteraction]] = Field(
        description="What operations are supported?",
        default=None,
    )
    versioning: Optional[Code] = Field(
        description="no-version | versioned | versioned-update",
        default=None,
    )
    versioning_ext: Optional[Element] = Field(
        description="Placeholder element for versioning extensions",
        default=None,
        alias="_versioning",
    )
    readHistory: Optional[Boolean] = Field(
        description="Whether vRead can return past versions",
        default=None,
    )
    readHistory_ext: Optional[Element] = Field(
        description="Placeholder element for readHistory extensions",
        default=None,
        alias="_readHistory",
    )
    updateCreate: Optional[Boolean] = Field(
        description="If update can commit to a new identity",
        default=None,
    )
    updateCreate_ext: Optional[Element] = Field(
        description="Placeholder element for updateCreate extensions",
        default=None,
        alias="_updateCreate",
    )
    conditionalCreate: Optional[Boolean] = Field(
        description="If allows/uses conditional create",
        default=None,
    )
    conditionalCreate_ext: Optional[Element] = Field(
        description="Placeholder element for conditionalCreate extensions",
        default=None,
        alias="_conditionalCreate",
    )
    conditionalRead: Optional[Code] = Field(
        description="not-supported | modified-since | not-match | full-support",
        default=None,
    )
    conditionalRead_ext: Optional[Element] = Field(
        description="Placeholder element for conditionalRead extensions",
        default=None,
        alias="_conditionalRead",
    )
    conditionalUpdate: Optional[Boolean] = Field(
        description="If allows/uses conditional update",
        default=None,
    )
    conditionalUpdate_ext: Optional[Element] = Field(
        description="Placeholder element for conditionalUpdate extensions",
        default=None,
        alias="_conditionalUpdate",
    )
    conditionalDelete: Optional[Code] = Field(
        description="not-supported | single | multiple - how conditional delete is supported",
        default=None,
    )
    conditionalDelete_ext: Optional[Element] = Field(
        description="Placeholder element for conditionalDelete extensions",
        default=None,
        alias="_conditionalDelete",
    )
    referencePolicy: Optional[List[Code]] = Field(
        description="literal | logical | resolves | enforced | local",
        default=None,
    )
    referencePolicy_ext: Optional[Element] = Field(
        description="Placeholder element for referencePolicy extensions",
        default=None,
        alias="_referencePolicy",
    )
    searchInclude: Optional[List[String]] = Field(
        description="_include values supported by the server",
        default=None,
    )
    searchInclude_ext: Optional[Element] = Field(
        description="Placeholder element for searchInclude extensions",
        default=None,
        alias="_searchInclude",
    )
    searchRevInclude: Optional[List[String]] = Field(
        description="_revinclude values supported by the server",
        default=None,
    )
    searchRevInclude_ext: Optional[Element] = Field(
        description="Placeholder element for searchRevInclude extensions",
        default=None,
        alias="_searchRevInclude",
    )
    searchParam: Optional[List[CapabilityStatementRestResourceSearchParam]] = Field(
        description="Search parameters supported by implementation",
        default=None,
    )
    operation: Optional[List[CapabilityStatementRestResourceOperation]] = Field(
        description="Definition of a resource operation",
        default=None,
    )

    @field_validator(
        *(
            "operation",
            "searchParam",
            "searchRevInclude",
            "searchInclude",
            "referencePolicy",
            "conditionalDelete",
            "conditionalUpdate",
            "conditionalRead",
            "conditionalCreate",
            "updateCreate",
            "readHistory",
            "versioning",
            "interaction",
            "documentation",
            "supportedProfile",
            "profile",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CapabilityStatementRestInteraction(BackboneElement):
    """
    A specification of restful operations supported by the system.
    """

    code: Optional[Code] = Field(
        description="transaction | batch | search-system | history-system",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    documentation: Optional[Markdown] = Field(
        description="Anything special about operation behavior",
        default=None,
    )
    documentation_ext: Optional[Element] = Field(
        description="Placeholder element for documentation extensions",
        default=None,
        alias="_documentation",
    )

    @field_validator(
        *(
            "documentation",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CapabilityStatementRestSearchParam(BackboneElement):
    """
    Search parameters that are supported for searching all resources for implementations to support and/or make use of - either references to ones defined in the specification, or additional ones defined for/by the implementation.
    """

    name: Optional[String] = Field(
        description="Name of search parameter",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    definition: Optional[Canonical] = Field(
        description="Source of definition for parameter",
        default=None,
    )
    definition_ext: Optional[Element] = Field(
        description="Placeholder element for definition extensions",
        default=None,
        alias="_definition",
    )
    type: Optional[Code] = Field(
        description="number | date | string | token | reference | composite | quantity | uri | special",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    documentation: Optional[Markdown] = Field(
        description="Server-specific usage",
        default=None,
    )
    documentation_ext: Optional[Element] = Field(
        description="Placeholder element for documentation extensions",
        default=None,
        alias="_documentation",
    )

    @field_validator(
        *(
            "documentation",
            "type",
            "definition",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CapabilityStatementRestOperation(BackboneElement):
    """
    Definition of an operation or a named query together with its parameters and their meaning and type.
    """

    name: Optional[String] = Field(
        description="Name by which the operation/query is invoked",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    definition: Optional[Canonical] = Field(
        description="The defined operation/query",
        default=None,
    )
    definition_ext: Optional[Element] = Field(
        description="Placeholder element for definition extensions",
        default=None,
        alias="_definition",
    )
    documentation: Optional[Markdown] = Field(
        description="Specific details about operation behavior",
        default=None,
    )
    documentation_ext: Optional[Element] = Field(
        description="Placeholder element for documentation extensions",
        default=None,
        alias="_documentation",
    )

    @field_validator(
        *(
            "documentation",
            "definition",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CapabilityStatementRest(BackboneElement):
    """
    A definition of the restful capabilities of the solution, if any.
    """

    mode: Optional[Code] = Field(
        description="client | server",
        default=None,
    )
    mode_ext: Optional[Element] = Field(
        description="Placeholder element for mode extensions",
        default=None,
        alias="_mode",
    )
    documentation: Optional[Markdown] = Field(
        description="General description of implementation",
        default=None,
    )
    documentation_ext: Optional[Element] = Field(
        description="Placeholder element for documentation extensions",
        default=None,
        alias="_documentation",
    )
    security: Optional[CapabilityStatementRestSecurity] = Field(
        description="Information about security of implementation",
        default=None,
    )
    resource: Optional[List[CapabilityStatementRestResource]] = Field(
        description="Resource served on the REST interface",
        default=None,
    )
    interaction: Optional[List[CapabilityStatementRestInteraction]] = Field(
        description="What operations are supported?",
        default=None,
    )
    searchParam: Optional[List[CapabilityStatementRestSearchParam]] = Field(
        description="Search parameters for searching all resources",
        default=None,
    )
    operation: Optional[List[CapabilityStatementRestOperation]] = Field(
        description="Definition of a system level operation",
        default=None,
    )
    compartment: Optional[List[Canonical]] = Field(
        description="Compartments served/used by system",
        default=None,
    )
    compartment_ext: Optional[Element] = Field(
        description="Placeholder element for compartment extensions",
        default=None,
        alias="_compartment",
    )

    @field_validator(
        *(
            "compartment",
            "operation",
            "searchParam",
            "interaction",
            "resource",
            "security",
            "documentation",
            "mode",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("resource",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cpb_12_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="searchParam.select(name).isDistinct()",
            human="Search parameter names must be unique in the context of a resource.",
            key="cpb-12",
            severity="error",
        )


class CapabilityStatementMessagingEndpoint(BackboneElement):
    """
    An endpoint (network accessible address) to which messages and/or replies are to be sent.
    """

    protocol: Optional[Coding] = Field(
        description="http | ftp | mllp +",
        default=None,
    )
    address: Optional[Url] = Field(
        description="Network address or identifier of the end-point",
        default=None,
    )
    address_ext: Optional[Element] = Field(
        description="Placeholder element for address extensions",
        default=None,
        alias="_address",
    )

    @field_validator(
        *(
            "address",
            "protocol",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CapabilityStatementMessagingSupportedMessage(BackboneElement):
    """
    References to message definitions for messages this system can send or receive.
    """

    mode: Optional[Code] = Field(
        description="sender | receiver",
        default=None,
    )
    mode_ext: Optional[Element] = Field(
        description="Placeholder element for mode extensions",
        default=None,
        alias="_mode",
    )
    definition: Optional[Canonical] = Field(
        description="Message supported by this system",
        default=None,
    )
    definition_ext: Optional[Element] = Field(
        description="Placeholder element for definition extensions",
        default=None,
        alias="_definition",
    )

    @field_validator(
        *(
            "definition",
            "mode",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CapabilityStatementMessaging(BackboneElement):
    """
    A description of the messaging capabilities of the solution.
    """

    endpoint: Optional[List[CapabilityStatementMessagingEndpoint]] = Field(
        description="Where messages should be sent",
        default=None,
    )
    reliableCache: Optional[UnsignedInt] = Field(
        description="Reliable Message Cache Length (min)",
        default=None,
    )
    reliableCache_ext: Optional[Element] = Field(
        description="Placeholder element for reliableCache extensions",
        default=None,
        alias="_reliableCache",
    )
    documentation: Optional[Markdown] = Field(
        description="Messaging interface behavior details",
        default=None,
    )
    documentation_ext: Optional[Element] = Field(
        description="Placeholder element for documentation extensions",
        default=None,
        alias="_documentation",
    )
    supportedMessage: Optional[List[CapabilityStatementMessagingSupportedMessage]] = (
        Field(
            description="Messages supported by this system",
            default=None,
        )
    )

    @field_validator(
        *(
            "supportedMessage",
            "documentation",
            "reliableCache",
            "endpoint",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CapabilityStatementDocument(BackboneElement):
    """
    A document definition.
    """

    mode: Optional[Code] = Field(
        description="producer | consumer",
        default=None,
    )
    mode_ext: Optional[Element] = Field(
        description="Placeholder element for mode extensions",
        default=None,
        alias="_mode",
    )
    documentation: Optional[Markdown] = Field(
        description="Description of document support",
        default=None,
    )
    documentation_ext: Optional[Element] = Field(
        description="Placeholder element for documentation extensions",
        default=None,
        alias="_documentation",
    )
    profile: Optional[Canonical] = Field(
        description="Constraint on the resources used in the document",
        default=None,
    )
    profile_ext: Optional[Element] = Field(
        description="Placeholder element for profile extensions",
        default=None,
        alias="_profile",
    )

    @field_validator(
        *(
            "profile",
            "documentation",
            "mode",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class CapabilityStatement(DomainResource):
    """
    A Capability Statement documents a set of capabilities (behaviors) of a FHIR Server for a particular version of FHIR that may be used as a statement of actual server functionality or a statement of required or desired server implementation.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/CapabilityStatement"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this capability statement, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    version: Optional[String] = Field(
        description="Business version of the capability statement",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    name: Optional[String] = Field(
        description="Name for this capability statement (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this capability statement (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the capability statement",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for capability statement (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this capability statement is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    kind: Optional[Code] = Field(
        description="instance | capability | requirements",
        default=None,
    )
    kind_ext: Optional[Element] = Field(
        description="Placeholder element for kind extensions",
        default=None,
        alias="_kind",
    )
    instantiates: Optional[List[Canonical]] = Field(
        description="Canonical URL of another capability statement this implements",
        default=None,
    )
    instantiates_ext: Optional[Element] = Field(
        description="Placeholder element for instantiates extensions",
        default=None,
        alias="_instantiates",
    )
    imports: Optional[List[Canonical]] = Field(
        description="Canonical URL of another capability statement this adds to",
        default=None,
    )
    imports_ext: Optional[Element] = Field(
        description="Placeholder element for imports extensions",
        default=None,
        alias="_imports",
    )
    software: Optional[CapabilityStatementSoftware] = Field(
        description="Software that is covered by this capability statement",
        default=None,
    )
    implementation: Optional[CapabilityStatementImplementation] = Field(
        description="If this describes a specific instance",
        default=None,
    )
    fhirVersion: Optional[Code] = Field(
        description="FHIR Version the system supports",
        default=None,
    )
    fhirVersion_ext: Optional[Element] = Field(
        description="Placeholder element for fhirVersion extensions",
        default=None,
        alias="_fhirVersion",
    )
    format: Optional[List[Code]] = Field(
        description="formats supported (xml | json | ttl | mime type)",
        default=None,
    )
    format_ext: Optional[Element] = Field(
        description="Placeholder element for format extensions",
        default=None,
        alias="_format",
    )
    patchFormat: Optional[List[Code]] = Field(
        description="Patch formats supported",
        default=None,
    )
    patchFormat_ext: Optional[Element] = Field(
        description="Placeholder element for patchFormat extensions",
        default=None,
        alias="_patchFormat",
    )
    implementationGuide: Optional[List[Canonical]] = Field(
        description="Implementation guides supported",
        default=None,
    )
    implementationGuide_ext: Optional[Element] = Field(
        description="Placeholder element for implementationGuide extensions",
        default=None,
        alias="_implementationGuide",
    )
    rest: Optional[List[CapabilityStatementRest]] = Field(
        description="If the endpoint is a RESTful one",
        default=None,
    )
    messaging: Optional[List[CapabilityStatementMessaging]] = Field(
        description="If messaging is supported",
        default=None,
    )
    document: Optional[List[CapabilityStatementDocument]] = Field(
        description="Document definition",
        default=None,
    )
    resourceType: Literal["CapabilityStatement"] = Field(
        description=None,
        default="CapabilityStatement",
    )

    @field_validator(
        *(
            "document",
            "messaging",
            "rest",
            "implementationGuide",
            "patchFormat",
            "format",
            "fhirVersion",
            "implementation",
            "software",
            "imports",
            "instantiates",
            "kind",
            "copyright",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "experimental",
            "status",
            "title",
            "name",
            "version",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("rest",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cpb_9_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="resource.select(type).isDistinct()",
            human="A given resource can only be described once per RESTful mode.",
            key="cpb-9",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_cpb_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.matches('[A-Z]([A-Za-z0-9_]){0,254}')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="cpb-0",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_cpb_1_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="rest.exists() or messaging.exists() or document.exists()",
            human="A Capability Statement SHALL have at least one of REST, messaging or document element.",
            key="cpb-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_cpb_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="(description.count() + software.count() + implementation.count()) > 0",
            human="A Capability Statement SHALL have at least one of description, software, or implementation element.",
            key="cpb-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_cpb_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="messaging.endpoint.empty() or kind = 'instance'",
            human="Messaging end-point is required (and is only permitted) when a statement is for an implementation.",
            key="cpb-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_cpb_7_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="document.select(profile&mode).isDistinct()",
            human="The set of documents must be unique by the combination of profile and mode.",
            key="cpb-7",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_cpb_14_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="(kind != 'instance') or implementation.exists()",
            human="If kind = instance, implementation must be present and software may be present",
            key="cpb-14",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_cpb_15_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="(kind != 'capability') or (implementation.exists().not() and software.exists())",
            human="If kind = capability, implementation must be absent, software must be present",
            key="cpb-15",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_cpb_16_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="(kind!='requirements') or (implementation.exists().not() and software.exists().not())",
            human="If kind = requirements, implementation and software must be absent",
            key="cpb-16",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(as(canonical) = '#').exists() or descendants().where(as(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
